# Import helper functions for input/output
from lmtune_helpers import input_data, output_results

# Import standard libraries for data analysis
import networkx as nx
import numpy as np


def main():
    """Extract instance characteristics from the VRP problem instance."""
    results = {}
    try:
        # Get the instance data using the provided helper function
        instance_data = input_data()
        
        # Extract basic parameters from the instance
        n = instance_data.get('N', 0)  # number of customer nodes
        capacity = instance_data.get('Capacity', 0)
        demand = instance_data.get('Demand', [])
        distance_list = instance_data.get('Distance', [])
        
        if n < 0 or capacity <= 0 or len(demand) != n:
            raise ValueError('Invalid instance data: Check N, Capacity, and Demand fields.')
        
        # Total nodes including depot
        total_nodes = n + 1
        
        # Reshape the distance list into a square matrix
        if len(distance_list) != total_nodes * total_nodes:
            raise ValueError('Distance list length does not match (N+1)^2.')
        distance_matrix = np.array(distance_list).reshape((total_nodes, total_nodes))
        
        # Create a filtered graph based on a threshold on distance
        # We consider an edge present if the distance is less than the mean distance (excluding zeros)
        nonzero_distances = distance_matrix[distance_matrix > 0]
        mean_distance = np.mean(nonzero_distances) if nonzero_distances.size > 0 else 0

        G = nx.Graph()
        # Add all nodes: 0 is depot, 1..n are customers
        G.add_nodes_from(range(total_nodes))
        
        # Add edges if the distance is less than mean_distance
        for i in range(total_nodes):
            for j in range(i+1, total_nodes):
                if distance_matrix[i, j] < mean_distance:
                    # add edge with weight (not strictly needed for density)
                    G.add_edge(i, j, weight=distance_matrix[i, j])
        
        # Compute graph characteristics from the filtered graph
        graph_density = nx.density(G) if G.number_of_nodes() > 1 else 0
        clustering_coefficient = nx.average_clustering(G) if G.number_of_nodes() > 0 else 0
        degrees = [deg for node, deg in G.degree()]
        average_degree = np.mean(degrees) if degrees else 0
        
        # Problem scale: total number of nodes
        problem_scale = total_nodes
        
        # Constraint tightness: ratio of average customer demand to vehicle capacity
        avg_demand = np.mean(demand) if demand else 0
        constraint_tightness = avg_demand / capacity if capacity > 0 else 0
        
        # Prepare README text - ~200 words explanation
        readme_text = (
            "The instance was analyzed by first extracting the fundamental parameters defined in the VRP JSON schema, including the number of customer nodes, vehicle capacity, customer demands, and a flattened distance matrix. "
            "The distance matrix, representing the travel cost between the depot and each customer node, was reshaped into its proper two-dimensional form. From this, a graph was constructed where each node corresponds to a depot or customer, and an edge between any two nodes was established if the corresponding distance was lower than the mean distance among all non-zero distances, thereby highlighting closer relationships. "
            "Using NetworkX, we computed several graph metrics such as graph density, average clustering coefficient, and average node degree, which provide insights into the connectivity and structure of the instance. "
            "The ‘problem_scale’ parameter was determined by counting the total number of nodes (including the depot), reflecting the overall size of the problem, while the ‘constraint_tightness’ was calculated as the ratio of average customer demand to the vehicle capacity, indicating how restrictive the capacity constraints are relative to the demands. "
            "These metrics collectively characterize the structural complexity and potential difficulty of solving the vehicle routing problem, assisting in identifying the most suitable solver parameters. Overall, this detailed, multi-faceted analysis merges statistical properties of the input data with graph-based structural properties to yield a robust description of the problem instance."
        )
        
        # Assemble the results dictionary with exactly five characteristic parameters
        results = {
            "README": readme_text,
            "graph_density": graph_density,
            "clustering_coefficient": clustering_coefficient,
            "average_degree": average_degree,
            "problem_scale": problem_scale,
            "constraint_tightness": constraint_tightness
        }
        
    except Exception as e:
        results = {"error": str(e)}
    
    # Output results using the helper function
    output_results(results)


if __name__ == "__main__":
    main()
