# Import helper functions for input/output
from lmtune_helpers import input_data, output_results

# Import standard libraries for data analysis
import networkx as nx
import numpy as np


def main():
    """Extract instance characteristics from the VRP JSON instance and compute key metrics."""

    # Get the instance data using the helper function
    instance_data = input_data()

    # Initialize the results dictionary with placeholders
    results = {
        "README": "",
        "problem_scale": 0,
        "average_customer_demand": 0.0,
        "demand_std_dev": 0.0,
        "average_distance": 0.0,
        "distance_std_dev": 0.0
    }

    try:
        # Extract basic problem parameters
        # N: number of customer nodes
        n = instance_data.get('N', 0)
        capacity = instance_data.get('Capacity', 0)  # Not directly used in metrics
        demand_list = instance_data.get('Demand', [])
        distance_flat = instance_data.get('Distance', [])

        # Compute problem scale as total nodes (customers + depot)
        problem_scale = n + 1

        # Compute demand statistics
        if demand_list:
            average_customer_demand = float(np.mean(demand_list))
            demand_std_dev = float(np.std(demand_list))
        else:
            average_customer_demand = 0.0
            demand_std_dev = 0.0

        # Reconstruct the distance matrix from the flattened list
        matrix_size = n + 1
        if len(distance_flat) != matrix_size * matrix_size:
            raise ValueError(f"Distance array length {len(distance_flat)} is not equal to (N+1)^2 = {matrix_size*matrix_size}")
        distance_matrix = np.array(distance_flat).reshape((matrix_size, matrix_size))

        # Exclude diagonal entries (self-distances) for computing average and std deviation
        non_diag_mask = ~np.eye(matrix_size, dtype=bool)
        distances = distance_matrix[non_diag_mask]
        if distances.size > 0:
            average_distance = float(np.mean(distances))
            distance_std_dev = float(np.std(distances))
        else:
            average_distance = 0.0
            distance_std_dev = 0.0

        # Populate the results dictionary with calculated values
        results["problem_scale"] = problem_scale
        results["average_customer_demand"] = average_customer_demand
        results["demand_std_dev"] = demand_std_dev
        results["average_distance"] = average_distance
        results["distance_std_dev"] = distance_std_dev

        # Prepare the README description (~200 words)
        readme_text = (
            "The given VRP instance is represented as a JSON structure containing essential fields such as the number of customer nodes, "
            "vehicle capacity, customer demands, and a flattened distance matrix that includes both the depot and customer nodes. "
            "In this analysis, the distance matrix is treated as a complete weighted graph where each node represents a location, and edges "
            "carry weights that denote the travel distance between nodes. Although the graph is inherently complete, our focus was on deriving "
            "quantitative characteristics that highlight the complexity and design of the instance. The 'problem_scale' metric quantifies the instance by "
            "counting the total nodes (depot plus customers), serving as a basic measure of the problem size. Additionally, we computed statistical "
            "measures for customer demands including the average demand and standard deviation, which reveal the distribution and variability of "
            "resource requirements across customers. For the distance data, we reconstructed a matrix from the provided list and excluded self-loops "
            "to calculate the average distance and associated standard deviation. These metrics reflect the geographical distribution and spread of nodes, "
            "which can significantly impact routing decisions and solver performance. Together, these parameters offer insights into both the scale "
            "and structural complexity of the VRP instance, thereby aiding in the effective tuning of solver parameters for optimal performance."
        )
        results["README"] = readme_text

    except Exception as e:
        # In case of any errors, capture the error message in the results
        results["error"] = str(e)

    # Return the results using the helper function
    output_results(results)


if __name__ == "__main__":
    main()
