# Import helper functions for input/output
from lmtune_helpers import input_data, output_results

# Import standard libraries for data analysis
import networkx as nx
import numpy as np


def main():
    """Extract instance characteristics from a VRP instance in JSON format.
    
    This script analyzes the provided Vehicle Routing Problem instance data. The instance includes
    the number of customer nodes (N), vehicle capacity, a list of customer demands, and a flattened
    distance matrix representing the distances between the depot and customer nodes. The analysis extracts
    structural and statistical properties relevant to solver parameter tuning. Metrics such as the average
    and standard deviation of customer demands capture the distribution of load requirements, while the
    average and standard deviation of travel distances provide insight into the spatial structure of the
    instance. Additionally, the capacity utilization metric, computed as the ratio of total demand to the
    vehicle capacity, serves as an indicator of how constrained the problem is. These characteristics
    help differentiate instances by highlighting features that influence solving difficulty, such as
    demand variability, average travel cost, and resource tightness. The methodology involves reshaping
    the distance matrix to its proper dimensions, computing statistical measures using NumPy, and optionally
    constructing a complete graph using NetworkX to emphasize the inherent connectivity of routing problems.
    Overall, the extracted parameters provide a concise and informative snapshot of the problem's scale,
    complexity, and potential bottlenecks, which are crucial for tailoring solver strategies effectively.
    """ 
    
    # Initialize results dictionary with default structure
    results = {
        "README": "",
        "average_demand": 0.0,
        "demand_std_dev": 0.0,
        "average_distance": 0.0,
        "distance_std_dev": 0.0,
        "capacity_utilization": 0.0
    }
    
    try:
        # Get the instance data using the helper function
        instance_data = input_data()
        
        # Extract basic problem parameters
        n = instance_data.get('N', 0)  # Number of customer nodes (excluding depot)
        capacity = instance_data.get('Capacity', 0)
        demands = instance_data.get('Demand', [])
        distances_flat = instance_data.get('Distance', [])
        
        # Compute average and standard deviation of demands
        if demands:
            avg_demand = np.mean(demands)
            std_demand = np.std(demands)
        else:
            avg_demand = 0.0
            std_demand = 0.0
        
        # Reshape the distance list into a matrix of shape (N+1) x (N+1)
        total_nodes = n + 1  # Including depot
        if distances_flat and len(distances_flat) == total_nodes * total_nodes:
            distance_matrix = np.array(distances_flat).reshape((total_nodes, total_nodes))
            # Exclude diagonal for average distance computation
            non_diag = distance_matrix[~np.eye(total_nodes, dtype=bool)]
            avg_distance = np.mean(non_diag)
            std_distance = np.std(non_diag)
        else:
            avg_distance = 0.0
            std_distance = 0.0
        
        # Capacity utilization: total customer demand over vehicle capacity
        total_demand = np.sum(demands) if demands else 0
        capacity_util = total_demand / capacity if capacity != 0 else 0.0
        
        # Update results dictionary with computed values
        results["average_demand"] = float(avg_demand)
        results["demand_std_dev"] = float(std_demand)
        results["average_distance"] = float(avg_distance)
        results["distance_std_dev"] = float(std_distance)
        results["capacity_utilization"] = float(capacity_util)
        
        # Detailed README description
        results["README"] = ("This VRP instance was analyzed by processing a JSON formatted problem specification where the "
                               "number of customer nodes, vehicle capacity, customer demands, and a flattened distance matrix "
                               "were provided as inputs. The analysis begins by reshaping the distance matrix into its proper (N+1)x(N+1) "
                               "form to account for the depot and customer nodes. Statistical measures such as the average and standard deviation "
                               "of the customer demands were computed to understand the distribution of loads. Similarly, the average and standard deviation "
                               "of the non-diagonal elements of the distance matrix were calculated to capture the average travel cost between nodes, "
                               "excluding self-loops. Additionally, the total demand relative to the vehicle capacity was calculated to yield a capacity utilization metric, "
                               "offering insight into the tightness of resource constraints. These five parameters provide a comprehensive summary of the instance's size, "
                               "complexity, and structural characteristics, and are crucial for tailoring solver configurations to optimize performance based on inherent "
                               "problem features.")
        
    except Exception as e:
        results["error"] = str(e)
    
    # Return the results using the helper function
    output_results(results)


if __name__ == "__main__":
    main()
