# Import helper functions for input/output
from lmtune_helpers import input_data, output_results

# Import standard libraries for data analysis
import networkx as nx
import numpy as np


def main():
    """Extract instance characteristics from a Vehicle Routing Problem instance.
    
    This script extracts key characteristics from a VRP instance provided in JSON format. The input data includes the
    number of customer nodes (N), vehicle capacity, customer demands, and a flattened distance matrix that represents
    distances between the depot and customer nodes. Using these inputs, a weighted, undirected graph is built using
    NetworkX. Essential metrics computed include the graph density (indicating how many edges exist relative to the
    total possible connections), the average degree (representing mean connectivity per node), the coefficient of
    variation for customer demands (providing insight into demand variability), the standard deviation of non-diagonal
    distance matrix values (reflecting the geographic spread), and the capacity ratio (the vehicle capacity relative
    to the total demand).
    
    These characteristics are crucial for understanding the structural complexity and computational challenges of
    the instance, thereby assisting in optimal solver parameter tuning. The analysis combines network statistics
    and fundamental statistics to produce a comprehensive profile of the instance.
    """
    
    try:
        # Retrieve the instance data
        instance_data = input_data()

        # Extract basic instance parameters
        n = instance_data.get('N', 0)  # number of customer nodes
        capacity = instance_data.get('Capacity', 0)
        demands = instance_data.get('Demand', [])
        distances_flat = instance_data.get('Distance', [])

        total_nodes = n + 1  # including the depot
        if len(distances_flat) != total_nodes * total_nodes:
            raise ValueError(f"Expected flattened distance matrix of size {total_nodes * total_nodes}, got {len(distances_flat)}")
        
        # Construct the full distance matrix
        distance_matrix = np.array(distances_flat).reshape((total_nodes, total_nodes))

        # Build an undirected graph from the distance matrix
        G = nx.Graph()
        for i in range(total_nodes):
            G.add_node(i)
        for i in range(total_nodes):
            for j in range(i + 1, total_nodes):
                G.add_edge(i, j, weight=distance_matrix[i, j])

        # Compute graph density
        graph_density = nx.density(G)

        # Compute average degree
        degrees = [d for _, d in G.degree()]
        average_degree = float(np.mean(degrees))

        # Compute coefficient of variation of demands (std/mean)
        if demands:
            demands_array = np.array(demands, dtype=float)
            mean_demand = float(np.mean(demands_array))
            std_demand = float(np.std(demands_array))
            demand_cv = std_demand / mean_demand if mean_demand != 0 else 0.0
        else:
            demand_cv = 0.0

        # Compute standard deviation of distance values (ignoring diagonal zeros)
        off_diag = distance_matrix[~np.eye(total_nodes, dtype=bool)]
        distance_std = float(np.std(off_diag))

        # Compute capacity ratio: vehicle capacity relative to total customer demand
        total_demand = sum(demands) if demands else 0
        capacity_ratio = float(capacity / total_demand) if total_demand != 0 else 0.0

        # Prepare results dictionary with analysis details and required characteristic parameters
        results = {
            "README": """This Vehicle Routing Problem instance was analyzed using a systematic approach to extract its key structural and statistical properties. The provided input data was used to build a weighted, undirected graph with nodes representing the depot and customer locations, and edge weights derived from the distance matrix. Graph density was calculated as the ratio of existing edges to the total number of potential edges, reflecting overall connectivity. The average degree metric indicates the mean number of connections per node, offering additional insight into network structure. In parallel, the statistical evaluation of customer demands was performed by computing the coefficient of variation, which highlights the relative variability in demands. Furthermore, the standard deviation of off-diagonal distance values quantified the dispersion in travel distances between distinct nodes. Finally, the capacity ratio—computed as the vehicle capacity divided by the total customer demand—provides an essential measure of resource slack. Collectively, these metrics form a comprehensive profile of the instance, informing solver parameter tuning and aiding in the assessment of computational complexity and constraint tightness.""",
            "graph_density": graph_density,
            "average_degree": average_degree,
            "demand_cv": demand_cv,
            "distance_std": distance_std,
            "capacity_ratio": capacity_ratio
        }
    except Exception as e:
        results = {"error": str(e)}

    # Output the results
    output_results(results)


if __name__ == "__main__":
    main()
