# Import helper functions for input/output
from lmtune_helpers import input_data, output_results

# Import standard libraries for data analysis
import networkx as nx
import numpy as np


def main():
    """Extract instance characteristics from a VRP problem instance."""
    # Initialize results dictionary with default values
    results = {
        "README": "",
        "problem_scale": 0,
        "average_demand": 0.0,
        "demand_variance": 0.0,
        "distance_variance": 0.0,
        "constraint_tightness": 0.0
    }

    try:
        # Get the instance data using the helper function
        instance_data = input_data()

        # Extract basic problem parameters
        # N: number of customer nodes; Capacity: vehicle capacity
        N = instance_data.get('N', 0)
        capacity = instance_data.get('Capacity', 0)
        demand_list = instance_data.get('Demand', [])
        distance_list = instance_data.get('Distance', [])

        # Calculate problem scale as total nodes (depot + customers)
        problem_scale = N + 1

        # Compute average demand and variance of demands using NumPy
        if demand_list:
            avg_demand = float(np.mean(demand_list))
            var_demand = float(np.var(demand_list))
        else:
            avg_demand = 0.0
            var_demand = 0.0

        # Compute variance of distances, which gives an idea of spread in the distance matrix values
        if distance_list:
            var_distance = float(np.var(distance_list))
        else:
            var_distance = 0.0

        # Compute constraint tightness as ratio of vehicle capacity to total demand
        total_demand = sum(demand_list) if demand_list else 1
        constraint_tightness = float(capacity) / float(total_demand) if total_demand != 0 else 0.0

        # Construct a complete graph representing the VRP instance nodes
        # There are (N+1) nodes including depot. Even though in a complete graph, density is 1, we can verify connectivity properties.
        G = nx.complete_graph(problem_scale)
        # For demonstration, we can compute graph density (it will be 1.0 for complete graphs)
        graph_density = nx.density(G)
        # However, since the complete graph is implied by the instance, we use problem_scale instead to judge the instance size.

        # Prepare README description (~200 words)
        readme = (
            "This VRP instance was analyzed by first reading the input JSON data which describes the number of customer nodes, a single depot, vehicle maximum capacity, individual customer demands, and a flattened distance matrix for all node pairs. "
            "The instance was deconstructed in terms of its structural components by calculating the problem scale as the total number of nodes (customers plus depot). Statistical tools from NumPy were then utilized to compute the average and variance of customer demands, providing insight into the distribution and variability of the workload across nodes. "
            "Furthermore, the distance matrix was analyzed to extract its variance, giving an indication of how varied the inter-node distances are. To understand constraint implications in the model, the ratio of vehicle capacity to the total demand, termed as ‘constraint tightness’, was evaluated and is critical for understanding the leeway available in vehicle loading. "
            "In addition, a complete graph was constructed using NetworkX to represent the inherent connectivity of the VRP, although being a complete graph, its density remains 1. This analysis collectively helps in understanding both the quantitative and topological characteristics of the instance, which are vital parameters for tuning solver strategies."
        )

        # Update the results dictionary with computed values
        results["README"] = readme
        results["problem_scale"] = problem_scale
        results["average_demand"] = avg_demand
        results["demand_variance"] = var_demand
        results["distance_variance"] = var_distance
        results["constraint_tightness"] = constraint_tightness

    except Exception as e:
        # Handle any errors during analysis
        results = {"error": str(e)}

    # Return the results using the helper function
    output_results(results)


if __name__ == "__main__":
    main()
