# Import helper functions for input/output
from lmtune_helpers import input_data, output_results

# Import standard libraries for data analysis
import networkx as nx
import numpy as np


def main():
    """Extract instance characteristics from a VRP problem instance."""
    # Get instance data
    instance_data = input_data()

    # Initialize results dictionary with placeholders
    results = {
        "README": "",
        "average_distance": 0.0,
        "distance_std": 0.0,
        "demand_variance": 0.0,
        "total_nodes": 0,
        "capacity_ratio": 0.0
    }

    try:
        # Extract basic parameters from instance data
        n = instance_data.get('N', 0)  # Number of customers
        capacity = instance_data.get('Capacity', 0)  # Vehicle capacity
        demand_list = instance_data.get('Demand', [])
        distance_list = instance_data.get('Distance', [])

        # Compute total nodes including depot
        total_nodes = n + 1

        # Convert lists to numpy arrays for statistical analysis
        distance_array = np.array(distance_list, dtype=float)
        demand_array = np.array(demand_list, dtype=float)

        # Compute average distance from the flattened matrix
        if distance_array.size > 0:
            average_distance = float(np.mean(distance_array))
            distance_std = float(np.std(distance_array))
        else:
            average_distance = 0.0
            distance_std = 0.0

        # Compute variance of demand
        if demand_array.size > 0:
            demand_variance = float(np.var(demand_array))
        else:
            demand_variance = 0.0

        # Compute capacity ratio as capacity divided by total customer demand
        total_demand = np.sum(demand_array) if demand_array.size > 0 else 0
        capacity_ratio = float(capacity / total_demand) if total_demand > 0 else 0.0

        # Prepare README description (~200 words)
        readme_text = (
            "This instance was analyzed by reading the problem parameters from a JSON formatted Vehicle Routing Problem (VRP) instance. "
            "The input includes the number of customer nodes (N), the vehicle Capacity, an array detailing the Demand for each customer, "
            "and a flattened Distance matrix representing the travel costs between the depot and customers. Using NumPy, statistical "
            "metrics were computed from the distance values and customer demands. The analysis involved calculating the average distance "
            "(average_distance) which gives a central measure of travel cost, along with its standard deviation (distance_std) to understand "
            "the variability among the distances. The demand variance (demand_variance) reflects how heterogeneous the customer demands are, "
            "which can affect route optimization strategies. Additionally, the total number of nodes (total_nodes) including the depot was computed "
            "to quantify the size of the instance. The capacity ratio (capacity_ratio) indicates the tightness of the vehicle capacity constraint relative "
            "to the total customer demand – a key factor in assessing the balance between resource availability and need. Together, these five characteristics "
            "provide valuable insights into the structural and statistical complexity of the VRP instance. They serve as essential parameters for guiding solver configuration, "
            "helping to tailor optimization algorithms to the instance's specific challenges and ensuring efficient and effective problem-solving strategies."
        )

        # Update results dictionary with computed values
        results["README"] = readme_text
        results["average_distance"] = average_distance
        results["distance_std"] = distance_std
        results["demand_variance"] = demand_variance
        results["total_nodes"] = total_nodes
        results["capacity_ratio"] = capacity_ratio

    except Exception as e:
        # In case of error, record the error message
        results["error"] = str(e)

    # Output the results using helper function
    output_results(results)


if __name__ == "__main__":
    main()
