# Import helper functions for input/output
from lmtune_helpers import input_data, output_results

# Import standard libraries for data analysis
import networkx as nx
import numpy as np


def main():
    """Extract instance characteristics from a VRP instance JSON data."""
    results = {}
    try:
        # Get the instance data using the helper function
        instance_data = input_data()
        
        # Extract basic instance parameters
        N = instance_data.get('N', 0)  # number of customers
        capacity = instance_data.get('Capacity', 0)
        demand_list = instance_data.get('Demand', [])
        distance_flat = instance_data.get('Distance', [])
        
        # Compute problem scale: total number of nodes (customers + depot)
        problem_scale = N + 1
        
        # Compute average demand and variance in customer demand
        if demand_list:
            avg_demand = float(np.mean(demand_list))
            demand_variance = float(np.var(demand_list))
        else:
            avg_demand = 0.0
            demand_variance = 0.0
        
        # Process distance matrix: It is a flattened array for a (N+1)x(N+1) matrix
        matrix_size = problem_scale * problem_scale
        if len(distance_flat) != matrix_size:
            raise ValueError(f"Distance array length {len(distance_flat)} does not match expected size {matrix_size}.")
        distance_matrix = np.array(distance_flat).reshape((problem_scale, problem_scale))
        
        # Compute average distance for off-diagonal elements
        # Exclude the diagonal (self-distance = 0) from the analysis
        off_diagonal = distance_matrix[~np.eye(problem_scale, dtype=bool)].flatten()
        if off_diagonal.size > 0:
            avg_distance = float(np.mean(off_diagonal))
            std_distance = float(np.std(off_diagonal))
            # Variation as coefficient of variation
            distance_variation = std_distance / avg_distance if avg_distance != 0 else 0.0
        else:
            avg_distance = 0.0
            distance_variation = 0.0
        
        # Construct README string of about 200 words
        readme_text = (
            "This script analyzes a Vehicle Routing Problem (VRP) instance provided in JSON format to extract key structural characteristics that influence solver performance. "
            "The analysis begins by parsing the instance data which includes the number of customer nodes (N), the vehicle capacity, demand for each customer, and the flattened distance matrix representing distances among the depot and customer nodes. "
            "For analysis, the problem scale is defined as the total number of nodes, including the depot, to capture the instance size. In addition, statistical measures of customer demand such as average demand and variance are computed to understand the load distribution across customers. "
            "The distance matrix is reshaped into a two-dimensional form and the average distance between distinct nodes is calculated after excluding self-loops, which helps in estimating the typical travel cost between nodes. "
            "Furthermore, the relative variability of distances is measured using the coefficient of variation, which is the ratio of the standard deviation to the mean distance. "
            "These five parameters—problem scale, average demand, demand variance, average distance, and distance variation—provide a comprehensive overview of the instance's structural complexity. "
            "Such insights are critical for configuring and optimizing solver parameters, as they highlight the scale, irregularities in demand, and distance dispersion within the problem instance. "
            "These factors together impact constraint propagation and search efficiency in a constraint programming solver by indicating how tightly coupled the constraints are and suggesting potential areas where specialized heuristics may improve solution performance."
        )
        
        # Assemble results dictionary with exactly 5 characteristic parameters
        results = {
            "README": readme_text,
            "problem_scale": problem_scale,
            "avg_demand": avg_demand,
            "demand_variance": demand_variance,
            "avg_distance": avg_distance,
            "distance_variation": distance_variation
        }
        
    except Exception as e:
        results = {"README": "An error occurred during instance analysis.", "error": str(e)}
        
    # Return the results using the helper function
    output_results(results)


if __name__ == "__main__":
    main()
