# Import helper functions for input/output
from lmtune_helpers import input_data, output_results

# Import standard libraries for data analysis
import networkx as nx
import numpy as np


def main():
    """Extract instance characteristics from a VRP instance in JSON format.

    This function retrieves the instance data using the helper function input_data(),
    analyzes key structural and statistical features of the instance, and computes five
    characteristic parameters that provide insights into the problem complexity and
    potential solver performance. These include graph density computed from a complete
    graph representation of the distance matrix, average distance and its standard
    deviation computed from non-diagonal elements of the distance matrix, a demand
    variability measure based on the coefficient of variation of customer demands, and
    the overall problem scale measured by the total number of nodes (including the depot).
    The results are then returned via output_results in the mandated dictionary format.
    """
    
    results = {}

    try:
        # Get the instance data using the helper function
        instance_data = input_data()
        
        # Extract basic parameters from the JSON
        n = instance_data.get('N', 0)  # number of customer nodes
        capacity = instance_data.get('Capacity', 0)  # vehicle capacity
        demands = instance_data.get('Demand', [])  # list of demands
        distances_flat = instance_data.get('Distance', [])  # flattened distance matrix

        # Check that the distance matrix has the expected size: (n+1)^2
        expected_size = (n + 1) * (n + 1)
        if len(distances_flat) != expected_size:
            raise ValueError(f"Distance matrix size mismatch: expected {expected_size} elements, got {len(distances_flat)}.")
        
        # Convert the flattened distance list into a 2D numpy array (matrix)
        distance_matrix = np.array(distances_flat).reshape((n + 1, n + 1))
        
        # Construct a graph using NetworkX
        # Nodes: 0 to n (0 is depot)
        G = nx.Graph()
        G.add_nodes_from(range(n + 1))
        
        # Add edges for every pair (i, j) with i < j, using the distance as weight
        for i in range(n + 1):
            for j in range(i + 1, n + 1):
                # Use the distance value; optionally, we exclude self-loops
                weight = distance_matrix[i, j]
                # We assume that a valid edge has a non-negative distance;
                # if weight is 0 and i!=j it may represent a zero cost edge, so we add it
                G.add_edge(i, j, weight=weight)
        
        # Compute graph density. Since the graph is complete (or nearly so), density is ratio of edges present
        graph_density = nx.density(G)
        
        # Compute average distance and its standard deviation from the distance matrix
        # Exclude the diagonal (i == j) as they are zeros
        non_diagonal = distance_matrix[~np.eye(distance_matrix.shape[0],dtype=bool)]
        average_distance = float(np.mean(non_diagonal))
        distance_std_dev = float(np.std(non_diagonal))
        
        # Compute demand variability: coefficient of variation (std dev / mean)
        if len(demands) == 0:
            demand_coefficient = 0.0
        else:
            demands_array = np.array(demands)
            mean_demand = np.mean(demands_array)
            std_demand = np.std(demands_array)
            demand_coefficient = float(std_demand / mean_demand) if mean_demand != 0 else 0.0
        
        # Problem scale: total number of nodes including the depot
        problem_scale = n + 1
        
        # Prepare README text (approx 200 words)
        readme_text = (
            "This instance was analyzed by first extracting the core components from the provided JSON data. "
            "The data includes the number of customer nodes, vehicle capacity, a list of demands, and a flattened distance matrix representing the distances between all nodes, including the depot. "
            "We reshaped the distance array into a square matrix to accurately reflect the pairwise distances between nodes. "
            "Using NetworkX, we constructed a graph where each node represents either the depot or a customer, and each edge carries the corresponding distance value as its weight. "
            "From this graph, we computed the graph density, which indicates how complete the graph is by comparing the number of actual edges to the number of possible edges. "
            "In parallel, we computed statistical properties of the distance matrix, excluding self-loops, to derive the average distance between nodes and the variability in these distances via standard deviation. "
            "The demand variability was assessed using the coefficient of variation, highlighting how spread out the customer demands are relative to the mean. "
            "Lastly, we measured the overall problem scale by counting the total number of nodes, thereby capturing the instance size. "
            "These metrics provide important insights into the structural and statistical complexity of the instance, which are essential in configuring solver parameters for optimal performance."
        )
        
        # Build the results dictionary with exactly 5 characteristic parameters
        results = {
            "README": readme_text,
            "graph_density": graph_density,
            "average_distance": average_distance,
            "distance_std_dev": distance_std_dev,
            "demand_coefficient": demand_coefficient,
            "problem_scale": problem_scale
        }
        
    except Exception as e:
        results = {"README": "An error occurred during instance analysis.", "error": str(e)}
    
    # Return the results using the helper function
    output_results(results)


if __name__ == "__main__":
    main()
