"""
Agent module for creating a LangGraph ReAct agent with script generation tools.
"""

import os
import logging
import ast
import json
import subprocess
import tempfile
import sys
import argparse
from typing import Dict, List, Any, Optional, Union

from rich.console import Console
from rich.panel import Panel
from rich.json import JSON
from rich.tree import Tree
from rich.syntax import Syntax

from langchain_core.tools import tool
from langchain_core.messages import HumanMessage, SystemMessage
from langchain_core.runnables.config import RunnableConfig
from langgraph.prebuilt import create_react_agent
from langchain_core.callbacks.base import BaseCallbackHandler
from lmtune.llm_factory import LLMFactory
from lmtune.prompt_loader import PromptLoader
from lmtune.config import Config
from lmtune.utils import print_code
from lmtune.monitor import EnhancedToolCallbackHandler, SimpleToolCallbackHandler, ToolStats

# Set up logging
logger = logging.getLogger(__name__)


# Removed MinimalToolCallbackHandler - replaced with monitor.py classes


class StringEditor:
    """A class to manage a string that can be edited by the agent."""

    def __init__(self, initial_content: str = ""):
        """Initialize the string editor with optional initial content."""
        self.content = initial_content
        self.lines = initial_content.split("\n")
        self.script_output = None
        self.result_dict = None

    def get_content(self) -> str:
        """Get the current content of the string."""
        return self.content

    def clear(self) -> str:
        """Clear the string content."""
        self.content = ""
        self.lines = [""]
        self.script_output = None
        self.result_dict = None
        return "String cleared successfully."

    def get_result_formatter(self) -> str:
        """
        Returns a Python function that can be used to format results as JSON.
        This is a helper that agents can include in their scripts.
        """
        return '''
def output_results(result_dict):
    """
    Helper function to format and print results as JSON.
    
    Args:
        result_dict: A dictionary containing the script results
    """
    import json
    print(json.dumps(result_dict))
'''

    def view(self, view_range: Optional[List[int]] = None) -> str:
        """
        View the content of the string, optionally within a specific range of lines.

        Args:
            view_range: A list of two integers specifying start and end line numbers (1-indexed).
                        If end line is -1, it means read to the end.

        Returns:
            The content of the string within the specified range, or the entire string if no range is specified.
        """
        if not self.content.strip():
            return "String is empty."

        self.lines = self.content.split("\n")

        if not view_range:
            return self.content

        start_line, end_line = view_range

        # Adjust for 1-indexed input
        start_idx = max(0, start_line - 1)

        # Handle -1 as "read to the end"
        if end_line == -1:
            end_idx = len(self.lines)
        else:
            end_idx = min(end_line, len(self.lines))

        if start_idx >= len(self.lines) or start_idx >= end_idx:
            return f"Error: Invalid range. String has {len(self.lines)} lines."

        selected_lines = self.lines[start_idx:end_idx]
        return "\n".join(selected_lines)

    def insert(self, insert_line: int, new_str: str) -> str:
        """
        Insert text at a specific location in the string.

        Args:
            insert_line: The line number after which to insert the text (0 for beginning of file).
            new_str: The text to insert.

        Returns:
            A success message along with context of the change.
        """
        self.lines = self.content.split("\n")

        # Handle empty string case
        if not self.lines or (len(self.lines) == 1 and not self.lines[0]):
            self.content = new_str
            self.lines = self.content.split("\n")
            return f"Text inserted at the beginning of empty string. Content now:\n{self.content}"

        # Validate insert position
        if insert_line < 0 or insert_line > len(self.lines):
            return (
                f"Error: Invalid insertion point. String has {len(self.lines)} lines."
            )

        # Insert the new string
        if insert_line == 0:
            # Insert at the beginning
            new_lines = new_str.split("\n") + self.lines
        else:
            # Insert after the specified line
            new_lines = (
                self.lines[:insert_line]
                + new_str.split("\n")
                + self.lines[insert_line:]
            )

        self.content = "\n".join(new_lines)
        self.lines = new_lines

        # Return success message with context
        context_start = max(0, insert_line - 2)
        context_end = min(len(new_lines), insert_line + len(new_str.split("\n")) + 2)
        context = "\n".join(new_lines[context_start:context_end])

        return f"Text inserted successfully. Context:\n{context}"

    def run(self) -> str:
        """
        Validate Python code using AST parsing and check for required output_results call.

        Returns:
            "true" if the code is valid Python syntax and contains required components,
            or an error message if validation fails.
        """
        if not self.content.strip():
            return "Error: No code to validate. The string is empty."

        # First validate Python syntax
        try:
            # Parse the content as Python code
            parsed_ast = ast.parse(self.content)

            # Check for required output_results function call
            has_output_results_call = False

            # Analyze the AST to find output_results function calls
            for node in ast.walk(parsed_ast):
                # Check for output_results() call
                if isinstance(node, ast.Expr) and isinstance(node.value, ast.Call):
                    call = node.value
                    if (
                        isinstance(call.func, ast.Name)
                        and call.func.id == "output_results"
                    ):
                        if call.args and len(call.args) > 0:
                            has_output_results_call = True
                            break

            # Validate requirements
            if not has_output_results_call:
                return "Error: The script must call output_results(result_dict) at the end, where result_dict is a dictionary containing your script's results."

        except SyntaxError as e:
            # Return detailed syntax error information
            line_num = e.lineno if hasattr(e, "lineno") else "?"
            col_num = e.offset if hasattr(e, "offset") else "?"
            error_msg = str(e)

            # Enhanced syntax error reporting with hints
            context = ""
            if hasattr(e, "text") and e.text:
                context = f"\nLine content: {e.text.strip()}"

            # Common syntax error hints
            hint = ""
            if "expected ':'" in error_msg:
                hint = "\nHint: Check if you're missing a colon at the end of a statement (if, for, def, etc.)"
            elif "unexpected EOF" in error_msg:
                hint = (
                    "\nHint: You might have unclosed parentheses, brackets, or quotes"
                )
            elif "unexpected indent" in error_msg:
                hint = (
                    "\nHint: Check your indentation - Python is sensitive to whitespace"
                )
            elif "invalid syntax" in error_msg and "=" in context:
                hint = "\nHint: Check for invalid assignment or comparison operators"
            elif "EOL while scanning string literal" in error_msg:
                hint = "\nHint: You have an unclosed string (missing quote)"

            return f"Syntax error at line {line_num}, column {col_num}: {error_msg}{context}{hint}"
        except Exception as e:
            # Catch any other parsing errors
            error_type = type(e).__name__
            return f"Error validating code ({error_type}): {str(e)}"

        return "true"

    def execute_script(self, problem_folder=None, problem_type=None, timeout: int = None) -> str:
        """
        Execute the Python script and capture the output.
        Always uses example.json as the input data.
        
        Args:
            problem_folder: Path to the problem folder
            problem_type: Type of problem (subfolder within problem_folder)
            timeout: Maximum execution time in seconds before terminating the script.
                    Defaults to Config.DEFAULT_SCRIPT_TIMEOUT (30 seconds).
        
        Returns:
            Output of the executed script or error message
        """
        # Always use the Config timeout value (which is set from command line)
        timeout = Config.DEFAULT_SCRIPT_TIMEOUT
        if not self.content.strip():
            return "Error: No code to execute. The string is empty."

        # First validate the code
        validation_result = self.run()  # Still use the internal run method
        if validation_result != "true":
            return f"Error: {validation_result}"

        # Execute the code in a subprocess
        try:
            with tempfile.NamedTemporaryFile(
                suffix=".py", mode="w", delete=False
            ) as temp_file:
                temp_file_path = temp_file.name
                
                # Get problem folder path
                if problem_folder:
                    problem_folder_path = os.path.expanduser(problem_folder)
                else:
                    # Default to current directory if not provided
                    problem_folder_path = os.getcwd()
                
                # Always use example.json as the input data
                input_json = None
                if problem_type:
                    example_file = os.path.join(problem_folder_path, problem_type, "example.json")
                    try:
                        with open(example_file, 'r') as f:
                            input_json = json.load(f)
                        logger.debug(f"Loaded example.json from {example_file}")
                    except FileNotFoundError:
                        return f"Error: example.json not found at {example_file}"
                    except json.JSONDecodeError as e:
                        return f"Error: Invalid JSON in example.json: {e}"
                else:
                    logger.warning("No problem_type provided to execute_script, using empty data")
                    # This happens when execute_script is called without proper context
                
                # Create module with helper functions and data
                # Properly serialize the JSON data
                if input_json is not None:
                    data_repr = repr(input_json)
                else:
                    data_repr = '{}'
                    
                helper_module_code = f'''
# Helper module for lmtune scripts
import json

# Store input data
_INPUT_DATA = {data_repr}

def input_data():
    """Get the input data for the problem instance.
    
    Returns:
        The parsed input data (JSON object or string)
    """
    return _INPUT_DATA

def output_results(result_dict):
    """Format and print results as JSON.
    
    Args:
        result_dict: A dictionary containing the script results
    """
    print(json.dumps(result_dict))
'''
                
                # Create a temporary module file
                helper_module_path = os.path.join(os.path.dirname(temp_file_path), "lmtune_helpers.py")
                with open(helper_module_path, 'w') as module_file:
                    module_file.write(helper_module_code)
                
                # Debug logging
                logger.debug(f"Created helper module at: {helper_module_path}")
                logger.debug(f"Helper module content preview: {helper_module_code[:200]}...")
                
                # Create a minimal header with dependencies
                helper_functions = '''
# /// script
# requires-python = ">=3.11"
# dependencies = ["pydantic", "networkx", "numpy"]
# ///

'''
                # Write the script with the uv header
                temp_file.write(helper_functions + self.content)
                
                # Debug: Log the script being executed
                logger.debug(f"Script content preview: {self.content[:200]}...")

                # Log the file we're about to execute
                logger.debug(
                    f"Executing Python script from temporary file: {temp_file_path}"
                )

            # Set up the environment to include the directory with our helper module
            env = os.environ.copy()
            module_dir = os.path.dirname(temp_file_path)
            
            # Use PYTHONPATH to make the helper module importable
            if 'PYTHONPATH' in env:
                env['PYTHONPATH'] = f"{module_dir}:{env['PYTHONPATH']}"
            else:
                env['PYTHONPATH'] = module_dir
                
            result = subprocess.run(
                ["uv", "run", temp_file_path], 
                capture_output=True, 
                text=True, 
                timeout=timeout,
                env=env
            )

            # Clean up the temporary files
            os.remove(temp_file_path)
            if os.path.exists(helper_module_path):
                os.remove(helper_module_path)

            if result.returncode != 0:
                error_msg = result.stderr
                # Enhanced error reporting with detailed tracebacks
                error_details = "Code execution failed with error:\n" + error_msg
                logger.error(error_details)

                # Format error message for clearer agent understanding
                if "ImportError" in error_msg:
                    return f"Code execution failed with error: Import error in script. Missing package or incorrect import statement:\n{error_msg}"
                elif "NameError" in error_msg:
                    return f"Code execution failed with error: Undefined variable or function:\n{error_msg}"
                elif "SyntaxError" in error_msg:
                    return f"Code execution failed with error: Syntax error in code:\n{error_msg}"
                elif "IndexError" in error_msg:
                    return f"Code execution failed with error: Index error in code. Likely accessing an invalid array index:\n{error_msg}"
                elif "KeyError" in error_msg:
                    return f"Code execution failed with error: Key error in code. Trying to access a nonexistent dictionary key:\n{error_msg}"
                elif "TypeError" in error_msg and "NoneType" in error_msg:
                    return f"Code execution failed with error: Trying to use a None value as if it were another type (like a list, dict, etc.):\n{error_msg}"
                elif "TypeError" in error_msg:
                    return f"Code execution failed with error: Type error - operation applied to incorrect data type:\n{error_msg}"
                elif "ZeroDivisionError" in error_msg:
                    return f"Code execution failed with error: Division by zero:\n{error_msg}"
                elif "ModuleNotFoundError" in error_msg:
                    return f"Code execution failed with error: Module not found. Missing package or incorrect import:\n{error_msg}"
                elif "AttributeError" in error_msg:
                    return f"Code execution failed with error: Attribute error. Accessing nonexistent attribute or method:\n{error_msg}"
                else:
                    # Generic error handling for other errors
                    return f"Code execution failed with error: {error_msg}"

            # Store and return the output
            self.script_output = result.stdout
            output = result.stdout
            
            # Check if output is empty or contains error indicators
            if not output.strip():
                return "Script executed but produced no output. This might indicate an error or the script might not be calling output_results()."
            
            # Check for common error patterns in stdout
            output_lower = output.lower()
            if any(err in output_lower for err in ["traceback", "error:", "exception:", "failed"]):
                return f"Script executed but appears to contain errors:\n{output}"

            # Check if output is empty or nearly empty
            if not output or len(output.strip()) == 0:
                if result.stderr:
                    logger.error(
                        f"Empty or nearly empty output, but stderr contains: {result.stderr}"
                    )
                    return f"Script executed without errors but produced no output. Stderr contains:\n{result.stderr}"
                else:
                    logger.warning("Script executed without producing any output")
                    return "Script executed successfully but produced no output. Please add print statements if you want to see results."

            # Log the raw output for debugging
            logger.debug(f"Raw stdout: {output[:500]}...")

            return f"Script executed successfully. Output:\n{output}"

        except subprocess.TimeoutExpired:
            return f"Error: Code execution timed out after {timeout} seconds. The script was terminated because it exceeded the timeout limit. Check for infinite loops or inefficient code."
        except Exception as e:
            logger.exception("Unexpected error during script execution")
            return f"Error during script execution: {str(e)}"

    def get_script_output(self) -> Optional[str]:
        """Get the most recently generated script output."""
        return self.script_output

    def get_result_dict(self) -> Optional[Dict[str, Any]]:
        """
        Get the result dictionary from the last script execution.

        Returns:
            The parsed JSON dictionary from the script output, or None if no valid JSON was found.
        """
        if not self.script_output:
            return None

        try:
            # Look for json.dumps output in the last lines
            output_lines = self.script_output.strip().split("\n")
            for line in reversed(output_lines):
                try:
                    result_dict = json.loads(line)
                    if isinstance(result_dict, dict):
                        self.result_dict = result_dict
                        return result_dict
                except json.JSONDecodeError:
                    continue
        except Exception:
            pass

        return None


class ScriptEditor:
    """A class that manages editing and running Python scripts via a LangGraph ReAct agent."""

    def __init__(
        self,
        model_code: str = Config.DEFAULT_MODEL_CODE,
        use_script_prompt: bool = True,
        args = None,
        monitor_level: str = "detailed",
    ):
        """
        Initialize the ScriptEditor.

        Args:
            model_code: Model code to use (default: from Config.DEFAULT_MODEL_CODE).
            use_script_prompt: Whether to use the script-specialized prompt.
            args: Command line arguments.
            monitor_level: Tool monitoring level (simple, detailed, minimal).
        """
        self.model_name = model_code
        self.use_script_prompt = use_script_prompt
        self.system_prompt = None
        self.args = args
        self.monitor_level = monitor_level
        
        # Store problem path info for execute_script
        self.problem_folder = getattr(args, 'problem_folder', None) if args else None
        self.problem_type = getattr(args, 'problem', None) if args else None

        # Initialize the string editor
        self.editor = StringEditor()

        # Create the LLM - use default temperature handling in LLMFactory
        self.llm_factory = LLMFactory()
        self.model = self.llm_factory.create_model(model_code=self.model_name)

        # Load system prompt if needed
        if self.use_script_prompt:
            # Load the specific script generation prompt
            try:
                self.system_prompt = PromptLoader.load_script_system_prompt()
                logger.info("Successfully loaded and applied script system prompt")

                # Apply the system prompt to the model
                self.model = self.model.with_config(
                    {"system_prompt": self.system_prompt}
                )
            except Exception as e:
                logger.error(f"Error loading prompt: {str(e)}")

        # Create the tools
        self.tools = self._create_tools()

        # Create the agent with standard config
        self.agent = create_react_agent(self.model, tools=self.tools)

    def _create_tools(self) -> List[Any]:
        """Create tools for the agent."""

        # Store editor reference to avoid closure reference to self
        editor = self.editor

        @tool
        def clear() -> str:
            """Clear the current string content."""
            if Config.VERBOSE_TOOL_OUTPUT:
                print(f"Tool called: clear()")
            result = editor.clear()
            if Config.VERBOSE_TOOL_OUTPUT:
                print(f"Tool output: {result}")
            return result

        @tool
        def view(view_range: Optional[List[int]] = None) -> str:
            """
            View the content of the string, optionally within a specific range of lines.
            If view_range is provided, it should be a list of two integers: [start_line, end_line].
            Lines are 1-indexed.
            """
            if Config.VERBOSE_TOOL_OUTPUT:
                print(f"Tool called: view(view_range={view_range})")
            result = editor.view(view_range)
            if Config.VERBOSE_TOOL_OUTPUT:
                print(
                    f"Tool output (truncated): {result[:100]}..."
                    if len(result) > 100
                    else f"Tool output: {result}"
                )
            return result

        @tool
        def insert(insert_line: int, new_str: str) -> str:
            """
            Insert text at a specific location in the string.
            insert_line: The line number after which to insert the text (0 for beginning of file).
            new_str: The text to insert.
            """
            if Config.VERBOSE_TOOL_OUTPUT:
                print(f"Tool called: insert(insert_line={insert_line})")
            result = editor.insert(insert_line, new_str)
            if Config.VERBOSE_TOOL_OUTPUT:
                print(
                    f"Tool output (truncated): {result[:100]}..."
                    if len(result) > 100
                    else f"Tool output: {result}"
                )
            return result

        @tool
        def check() -> str:
            """
            Validate the code in the string editor.
            Returns 'true' if the code is valid, or an error message if not.
            """
            if Config.VERBOSE_TOOL_OUTPUT:
                print(f"Tool called: check()")
            result = editor.run()
            if Config.VERBOSE_TOOL_OUTPUT:
                print(f"Tool output: {result}")
            return result

        @tool
        def execute_script() -> str:
            """
            Execute the Python script in the string editor and return the output.
            The timeout is determined by the --timeout command line parameter.
            """
            if Config.VERBOSE_TOOL_OUTPUT:
                print(f"Tool called: execute_script() with timeout={Config.DEFAULT_SCRIPT_TIMEOUT}s")
            
            # Use the agent's stored problem info
            result = editor.execute_script(problem_folder=self.problem_folder, 
                                         problem_type=self.problem_type)
                
            if Config.VERBOSE_TOOL_OUTPUT:
                print(f"Tool output: {result}")
            return result

        # Return the list of tools
        return [clear, view, insert, check, execute_script]

    def set_content(self, content: str) -> None:
        """Set the content of the string editor."""
        self.editor = StringEditor(content)

    def get_content(self) -> str:
        """Get the current content of the string editor."""
        return self.editor.get_content()

    def get_script_output(self) -> Optional[str]:
        """Get the most recently generated script output."""
        return self.editor.get_script_output()

    def get_result_dict(self) -> Optional[Dict[str, Any]]:
        """Get the extracted result dictionary from the script output."""
        return self.editor.get_result_dict()

    def run(self, user_input: str) -> Dict[str, Any]:
        """
        Run the agent with the given user input.

        Args:
            user_input: The user's query or instructions.

        Returns:
            The result of the agent's execution.
        """
        # Create proper messages with system prompt
        if self.use_script_prompt and self.system_prompt:
            # Create a system message with the loaded prompt
            messages = [
                SystemMessage(content=self.system_prompt),
                HumanMessage(content=user_input),
            ]
        else:
            messages = [HumanMessage(content=user_input)]

        # Add proper configuration with recursion_limit and monitoring
        callbacks = []
        if self.monitor_level == "minimal" or Config.MINIMAL_TOOL_TRACE:
            # Use simple monitoring for minimal output (original style)
            callbacks = [SimpleToolCallbackHandler()]
        elif self.monitor_level == "simple":
            # Use enhanced monitoring with minimal details
            callbacks = [EnhancedToolCallbackHandler(
                show_args=False,
                show_output=False,
                truncate_output=0
            )]
        else:  # detailed
            # Use enhanced monitoring for detailed output
            callbacks = [EnhancedToolCallbackHandler(
                show_args=True,
                show_output=True,
                truncate_output=1000
            )]

        config = RunnableConfig(recursion_limit=200, callbacks=callbacks)

        # Invoke the agent with configuration
        return self.agent.invoke({"messages": messages}, config=config)


def validate_problem_directory(problem_type, problem_folder):
    """
    Validate that the problem directory contains all required files.
    
    Args:
        problem_type: The type of problem (subfolder within problem_folder)
        problem_folder: Path to the folder containing problem subfolders
        
    Raises:
        ValueError: If any required files are missing
    """
    import os
    import glob
    
    # Resolve the problem folder path
    base_dir = os.path.expanduser(problem_folder)
    problem_dir = os.path.join(base_dir, problem_type)
    
    # Check if problem directory exists
    if not os.path.isdir(problem_dir):
        raise ValueError(f"Problem type '{problem_type}' not found in {base_dir}")
    
    # Check for required files
    required_files = []
    
    # Check for example.json
    example_json_path = os.path.join(problem_dir, "example.json")
    if not os.path.isfile(example_json_path):
        required_files.append("example.json")
    
    # Check for json-schema.md
    schema_path = os.path.join(problem_dir, "json-schema.md")
    if not os.path.isfile(schema_path):
        required_files.append("json-schema.md")
    
    # Check for at least one .mzn file
    mzn_files = glob.glob(os.path.join(problem_dir, "*.mzn"))
    if not mzn_files:
        required_files.append("*.mzn file")
    
    if required_files:
        raise ValueError(f"Missing required files in {problem_dir}: {', '.join(required_files)}")


def generate_problem_prompt(problem_type, problem_folder):
    """
    Generate a prompt based on a problem type using the universal MiniZinc tuning prompt.

    Args:
        problem_type: The type of problem (subfolder within problem_folder)
        problem_folder: Path to the folder containing problem subfolders

    Returns:
        A prompt string for the agent
    """
    import os
    import glob

    # First validate the problem directory
    validate_problem_directory(problem_type, problem_folder)

    # Resolve the problem folder path
    base_dir = os.path.expanduser(problem_folder)
    problem_dir = os.path.join(base_dir, problem_type)

    # Load schema content
    schema_path = os.path.join(problem_dir, "json-schema.md")
    with open(schema_path, "r", encoding="utf-8") as f:
        schema_content = f.read()

    # Find the MiniZinc model file (first .mzn file in the directory)
    mzn_files = glob.glob(os.path.join(problem_dir, "*.mzn"))
    model_path = mzn_files[0]  # Use the first MiniZinc file found
    with open(model_path, "r", encoding="utf-8") as f:
        model_content = f"```\n{f.read()}\n```"
    
    model_name = os.path.basename(model_path)
    print(f"Using MiniZinc model: {model_name}")

    # Use the universal MiniZinc tuning prompt
    from lmtune.prompt_loader import PromptLoader
    prompt_template = PromptLoader.load_mzn_tuning_prompt()

    # Replace placeholders in template
    prompt = (
        prompt_template.replace("${INSTANCE}", "example.json")
        .replace("${INSTANCE_BASE}", "example")
        .replace("${SCHEMA}", schema_content)
        .replace("${PROBLEM}", problem_type)
        .replace("${MODEL}", model_content)
    )
    
    # Debug: Log the prompt to verify it's correct
    logger.info(f"Generated prompt length: {len(prompt)} characters")
    logger.info(f"Prompt preview (first 500 chars): {prompt[:500]}...")
    
    # Also check if the critical import line is in the prompt
    if "from lmtune_helpers import input_data, output_results" in prompt:
        logger.info("✓ Prompt contains the required import statement")
    else:
        logger.warning("✗ WARNING: Prompt does NOT contain the required import statement!")
    
    # Write prompt to debug file
    import os
    debug_file = os.path.join(os.path.expanduser(problem_folder), problem_type, "debug_prompt.txt")
    try:
        with open(debug_file, 'w') as f:
            f.write("=== GENERATED PROMPT FOR AGENT ===\n\n")
            f.write(prompt)
        print(f"Debug: Full prompt written to {debug_file}")
    except Exception as e:
        logger.warning(f"Could not write debug prompt: {e}")

    return prompt


def main():
    """Main entry point for the script builder."""
    parser = argparse.ArgumentParser(description="Python Script Builder")
    parser.add_argument(
        "description",
        nargs="?",
        help="Description of the script to generate",
        default="Create a simple Python script that calculates and prints the first 10 Fibonacci numbers.",
    )
    parser.add_argument(
        "--mc",
        default=Config.DEFAULT_MODEL_CODE,
        help="Model code to use (format: 'OA:model' for OpenAI, 'OR:provider/model' for OpenRouter, 'AT:model' for Anthropic)",
    )
    parser.add_argument(
        "--problem-folder",
        required=True,
        help="Folder path containing problem subfolders",
    )
    parser.add_argument(
        "--problem",
        required=True,
        help="Problem type (subfolder within the problem folder)",
    )
    parser.add_argument(
        "--timeout",
        type=int,
        default=Config.DEFAULT_SCRIPT_TIMEOUT,
        help=f"Timeout in seconds for script execution (default: {Config.DEFAULT_SCRIPT_TIMEOUT})",
    )
    parser.add_argument(
        "--monitor",
        choices=["simple", "detailed", "minimal"],
        default="detailed",
        help="Tool monitoring level: simple (basic ✓/✗), detailed (with args/output), minimal (original style)",
    )
    parser.add_argument(
        "--show-stats",
        action="store_true",
        help="Show tool usage statistics at the end",
    )

    args = parser.parse_args()
    
    # Set the timeout in the config
    Config.DEFAULT_SCRIPT_TIMEOUT = args.timeout
    print(f"Script execution timeout set to {Config.DEFAULT_SCRIPT_TIMEOUT} seconds")
    
    # Configure monitoring based on command line option
    if args.monitor == "minimal":
        Config.MINIMAL_TOOL_TRACE = True
    else:
        Config.MINIMAL_TOOL_TRACE = False
    
    # Generate prompt from problem if specified
    description = args.description
    try:
        description = generate_problem_prompt(
            args.problem, args.problem_folder
        )
        print(f"Generated prompt for problem: {args.problem}")
    except Exception as e:
        print(f"Error generating problem prompt: {e}")
        return 1

    # Print the model code being used
    print(f"Using model: {args.mc}")
    print(f"Using problem folder: {args.problem_folder}")

    try:
        # Create a new Script editor and pass args to it
        agent = ScriptEditor(
            model_code=args.mc, 
            use_script_prompt=True, 
            args=args,
            monitor_level=args.monitor
        )

        logger.info(f"Generating script for problem")
        agent.run(description)
        
        # Print the generated code
        print_code(agent.get_content(), title="Generated Python Script")
        
        # Get and display script output if available
        script_output = agent.get_script_output()
        if script_output:
            console = Console()
            console.print(Panel("Script Output", style="bold cyan"))
            
            # Try to parse as JSON to format nicely
            try:
                json_data = json.loads(script_output.strip())
                console.print(JSON.from_data(json_data))
            except (json.JSONDecodeError, TypeError):
                # If not valid JSON or already parsed, print as regular text
                console.print(Panel(script_output, style="white on black"))

            # Get and display the result dictionary
            result_dict = agent.get_result_dict()
            if result_dict:
                console.print(Panel("Extracted Result Dictionary", style="bold green"))
                
                # Create a tree for structured display
                tree = Tree("Results")
                
                for key, value in result_dict.items():
                    branch = tree.add(f"[bold]{key}[/bold]")
                    if isinstance(value, dict):
                        for sub_key, sub_value in value.items():
                            branch.add(f"[yellow]{sub_key}[/yellow]: {sub_value}")
                    else:
                        branch.add(str(value))
                
                console.print(tree)

                # Save the working Python code with timestamp filename
                try:
                    import datetime

                    timestamp = datetime.datetime.now().strftime("%Y%m%d%H%M%S")
                    
                    # Create directory if it doesn't exist
                    script_dir = os.path.join(os.path.expanduser(args.problem_folder), args.problem)
                    os.makedirs(script_dir, exist_ok=True)

                    # Save the Python script with timestamp
                    filename = f"lmtuner{timestamp}.py"
                    script_path = os.path.join(script_dir, filename)

                    with open(script_path, "w") as f:
                        f.write(agent.get_content())

                    print(f"\nSuccessful script saved to: {script_path}")
                except Exception as e:
                    logger.error(f"Error saving script: {e}")
            else:
                logger.warning("No valid JSON dictionary found in the script output.")
        
        # Show tool usage statistics if requested
        if args.show_stats:
            tool_stats = ToolStats.get_instance()
            tool_stats.display_stats()
    
    except ValueError as e:
        # Catch provider errors specifically
        if "Provider returned error" in str(e):
            print(f"\nError: The LLM provider returned an error. Please try again or use a different model.")
            print(f"Details: {str(e)}")
            return 1
        # Re-raise other ValueErrors
        raise
    except Exception as e:
        # Catch any other exceptions
        print(f"\nAn unexpected error occurred: {str(e)}")
        print(f"Error details: {str(e)}")
        return 1

    return 0


if __name__ == "__main__":
    sys.exit(main())
