"""
Builder module for creating a LangGraph ReAct agent with script generation tools.
"""

import os
import logging
import ast
import json
import subprocess
import tempfile
import re
import time
from typing import Dict, List, Any, Optional, Tuple
import sys
import argparse

from langchain_core.tools import tool
from langchain_core.messages import HumanMessage, SystemMessage
from langchain_core.runnables.config import RunnableConfig
from langgraph.prebuilt import create_react_agent
from langchain_core.tracers import ConsoleCallbackHandler
from langchain_core.callbacks.base import BaseCallbackHandler
from langchain_core.outputs import LLMResult
from typing import Dict, List, Any, Optional, Union, Sequence
from lmtune.llm_factory import LLMFactory
from lmtune.prompt_loader import PromptLoader
from lmtune.config import Config
from lmtune.utils import print_code

# Set up logging
logger = logging.getLogger(__name__)


class MinimalToolCallbackHandler(BaseCallbackHandler):
    """A minimal callback handler that only logs tool calls with a simple format."""

    def __init__(self):
        """Initialize the handler."""
        super().__init__()
        self._current_tool = None

    def on_tool_start(
        self, serialized: Dict[str, Any], input_str: str, **kwargs: Any
    ) -> None:
        """Store the tool name for the current tool call."""
        tool_name = serialized.get("name", "unknown_tool")
        self._current_tool = tool_name
        print(f"[Tool] ▶ {tool_name}", end="")

    def on_tool_end(self, output: Any, **kwargs: Any) -> None:
        """Print tool result on the same line."""
        if output:
            # Check for errors in the output
            if isinstance(output, str) and any(
                error_term in output.lower()
                for error_term in [
                    "error:",
                    "failed",
                    "execution failed",
                    "traceback",
                    "exception",
                ]
            ):
                print(f" ✗")
                # Print the detailed error for the agent to see
                if "Code execution failed with error:" in output:
                    # Extract and display the actual error message
                    print(output)
            else:
                print(f" ✓")
                # For execute_script, always show a summary of the output
                if self._current_tool == "execute_script" and isinstance(output, str):
                    # Show first 200 chars of output to help debug
                    if len(output) > 200:
                        print(f"  Output preview: {output[:200]}...")
                    else:
                        print(f"  Output: {output}")
                # For other tools, show output if it's informative
                elif self._current_tool in ["check", "view"] and isinstance(output, str) and len(output) < 100:
                    print(f"  → {output}")
        else:
            print(f" ✓")

    def on_tool_error(
        self, error: Union[Exception, KeyboardInterrupt], **kwargs: Any
    ) -> None:
        """Print tool error on the same line and provide detailed error information."""
        print(f" ✗")
        # Print detailed error information
        error_message = str(error)
        error_type = type(error).__name__
        print(f"Tool execution error: {error_type}: {error_message}")

        # If there's a traceback, show it too
        import traceback

        tb_str = traceback.format_exc()
        if tb_str and tb_str != "NoneType: None\n":
            print("Traceback:")
            print(tb_str)

    # Disable all other callbacks for minimal output
    def on_llm_start(self, *args, **kwargs) -> None:
        pass

    def on_llm_end(self, *args, **kwargs) -> None:
        pass

    def on_llm_error(self, *args, **kwargs) -> None:
        pass

    def on_chain_start(self, *args, **kwargs) -> None:
        pass

    def on_chain_end(self, *args, **kwargs) -> None:
        pass

    def on_chain_error(self, *args, **kwargs) -> None:
        pass

    def on_text(self, *args, **kwargs) -> None:
        pass


class StringEditor:
    """A class to manage a string that can be edited by the agent."""

    def __init__(self, initial_content: str = ""):
        """Initialize the string editor with optional initial content."""
        self.content = initial_content
        self.lines = initial_content.split("\n")
        self.script_output = None

    def get_content(self) -> str:
        """Get the current content of the string."""
        return self.content

    def clear(self) -> str:
        """Clear the string content."""
        self.content = ""
        self.lines = [""]
        self.script_output = None
        return "String cleared successfully."

    def view(self, view_range: Optional[List[int]] = None) -> str:
        """
        View the content of the string, optionally within a specific range of lines.

        Args:
            view_range: A list of two integers specifying start and end line numbers (1-indexed).
                        If end line is -1, it means read to the end.

        Returns:
            The content of the string within the specified range, or the entire string if no range is specified.
        """
        if not self.content.strip():
            return "String is empty."

        self.lines = self.content.split("\n")

        if not view_range:
            return self.content

        start_line, end_line = view_range

        # Adjust for 1-indexed input
        start_idx = max(0, start_line - 1)

        # Handle -1 as "read to the end"
        if end_line == -1:
            end_idx = len(self.lines)
        else:
            end_idx = min(end_line, len(self.lines))

        if start_idx >= len(self.lines) or start_idx >= end_idx:
            return f"Error: Invalid range. String has {len(self.lines)} lines."

        selected_lines = self.lines[start_idx:end_idx]
        return "\n".join(selected_lines)

    def insert(self, insert_line: int, new_str: str) -> str:
        """
        Insert text at a specific location in the string.

        Args:
            insert_line: The line number after which to insert the text (0 for beginning of file).
            new_str: The text to insert.

        Returns:
            A success message along with context of the change.
        """
        self.lines = self.content.split("\n")

        # Handle empty string case
        if not self.lines or (len(self.lines) == 1 and not self.lines[0]):
            self.content = new_str
            self.lines = self.content.split("\n")
            return f"Text inserted at the beginning of empty string. Content now:\n{self.content}"

        # Validate insert position
        if insert_line < 0 or insert_line > len(self.lines):
            return (
                f"Error: Invalid insertion point. String has {len(self.lines)} lines."
            )

        # Insert the new string
        if insert_line == 0:
            # Insert at the beginning
            new_lines = new_str.split("\n") + self.lines
        else:
            # Insert after the specified line
            new_lines = (
                self.lines[:insert_line]
                + new_str.split("\n")
                + self.lines[insert_line:]
            )

        self.content = "\n".join(new_lines)
        self.lines = new_lines

        # Return success message with context
        context_start = max(0, insert_line - 2)
        context_end = min(len(new_lines), insert_line + len(new_str.split("\n")) + 2)
        context = "\n".join(new_lines[context_start:context_end])

        return f"Text inserted successfully. Context:\n{context}"

    def run(self) -> str:
        """
        Validate Python code using AST parsing.

        Returns:
            "true" if the code is valid Python syntax,
            or an error message if validation fails.
        """
        if not self.content.strip():
            return "Error: No code to validate. The string is empty."

        # First validate Python syntax
        try:
            # Parse the content as Python code
            ast.parse(self.content)
        except SyntaxError as e:
            # Return detailed syntax error information
            line_num = e.lineno if hasattr(e, "lineno") else "?"
            col_num = e.offset if hasattr(e, "offset") else "?"
            error_msg = str(e)

            # Enhanced syntax error reporting with hints
            context = ""
            if hasattr(e, "text") and e.text:
                context = f"\nLine content: {e.text.strip()}"

            # Common syntax error hints
            hint = ""
            if "expected ':'" in error_msg:
                hint = "\nHint: Check if you're missing a colon at the end of a statement (if, for, def, etc.)"
            elif "unexpected EOF" in error_msg:
                hint = (
                    "\nHint: You might have unclosed parentheses, brackets, or quotes"
                )
            elif "unexpected indent" in error_msg:
                hint = (
                    "\nHint: Check your indentation - Python is sensitive to whitespace"
                )
            elif "invalid syntax" in error_msg and "=" in context:
                hint = "\nHint: Check for invalid assignment or comparison operators"
            elif "EOL while scanning string literal" in error_msg:
                hint = "\nHint: You have an unclosed string (missing quote)"

            return f"Syntax error at line {line_num}, column {col_num}: {error_msg}{context}{hint}"
        except Exception as e:
            # Catch any other parsing errors
            error_type = type(e).__name__
            return f"Error validating code ({error_type}): {str(e)}"

        return "true"

    def execute_script(self) -> str:
        """
        Execute the Python script and capture the output.

        Returns:
            Output of the executed script or error message
        """
        if not self.content.strip():
            return "Error: No code to execute. The string is empty."

        # First validate the code
        validation_result = self.run()
        if validation_result != "true":
            return f"Error: {validation_result}"

        # Execute the code in a subprocess
        try:
            with tempfile.NamedTemporaryFile(
                suffix=".py", mode="w", delete=False
            ) as temp_file:
                temp_file_path = temp_file.name
                # Add inline dependencies using PEP 723
                inline_deps = """# /// script
# requires-python = \">=3.11\"
# dependencies = [\"pydantic\"]
# ///

"""
                temp_file.write(inline_deps + self.content)

                # Log the file we're about to execute
                logger.debug(
                    f"Executing Python script from temporary file: {temp_file_path}"
                )

            result = subprocess.run(
                ["uv", "run", temp_file_path], capture_output=True, text=True, timeout=10
            )

            # Clean up the temporary file
            os.remove(temp_file_path)

            if result.returncode != 0:
                error_msg = result.stderr
                # Enhanced error reporting with detailed tracebacks
                error_details = "Code execution failed with error:\n" + error_msg
                logger.error(error_details)

                # Format error message for clearer agent understanding
                if "ImportError" in error_msg:
                    return f"Code execution failed with error: Import error in script. Missing package or incorrect import statement:\n{error_msg}"
                elif "NameError" in error_msg:
                    return f"Code execution failed with error: Undefined variable or function:\n{error_msg}"
                elif "SyntaxError" in error_msg:
                    return f"Code execution failed with error: Syntax error in code:\n{error_msg}"
                elif "IndexError" in error_msg:
                    return f"Code execution failed with error: Index error in code. Likely accessing an invalid array index:\n{error_msg}"
                elif "KeyError" in error_msg:
                    return f"Code execution failed with error: Key error in code. Trying to access a nonexistent dictionary key:\n{error_msg}"
                elif "TypeError" in error_msg and "NoneType" in error_msg:
                    return f"Code execution failed with error: Trying to use a None value as if it were another type (like a list, dict, etc.):\n{error_msg}"
                elif "TypeError" in error_msg:
                    return f"Code execution failed with error: Type error - operation applied to incorrect data type:\n{error_msg}"
                elif "ZeroDivisionError" in error_msg:
                    return f"Code execution failed with error: Division by zero:\n{error_msg}"
                elif "ModuleNotFoundError" in error_msg:
                    return f"Code execution failed with error: Module not found. Missing package or incorrect import:\n{error_msg}"
                elif "AttributeError" in error_msg:
                    return f"Code execution failed with error: Attribute error. Accessing nonexistent attribute or method:\n{error_msg}"
                else:
                    # Generic error handling for other errors
                    return f"Code execution failed with error: {error_msg}"

            # Store and return the output
            self.script_output = result.stdout
            output = result.stdout

            # Check if output is empty or nearly empty
            if not output or len(output.strip()) == 0:
                if result.stderr:
                    logger.error(
                        f"Empty or nearly empty output, but stderr contains: {result.stderr}"
                    )
                    return f"Script executed without errors but produced no output. Stderr contains:\n{result.stderr}"
                else:
                    logger.warning("Script executed without producing any output")
                    return "Script executed successfully but produced no output. Please add print statements if you want to see results."

            # Log the raw output for debugging
            logger.debug(f"Raw stdout: {output[:500]}...")

            return f"Script executed successfully. Output:\n{output}"

        except subprocess.TimeoutExpired:
            return "Error: Code execution timed out after 10 seconds. Check for infinite loops or inefficient code."
        except Exception as e:
            logger.exception("Unexpected error during script execution")
            return f"Error during script execution: {str(e)}"

    def get_script_output(self) -> Optional[str]:
        """Get the most recently generated script output."""
        return self.script_output


class ScriptEditor:
    """A class that manages editing and running Python scripts via a LangGraph ReAct agent."""

    def __init__(
        self,
        model_code: str = Config.DEFAULT_MODEL_CODE,
        use_script_prompt: bool = True,
    ):
        """
        Initialize the ScriptEditor.

        Args:
            model_code: Model code to use (default: from Config.DEFAULT_MODEL_CODE).
            use_script_prompt: Whether to use the script-specialized prompt.
        """
        self.model_name = model_code
        self.use_script_prompt = use_script_prompt
        self.system_prompt = None

        # Initialize the string editor
        self.editor = StringEditor()

        # Create the LLM - use default temperature handling in LLMFactory
        self.llm_factory = LLMFactory()
        self.model = self.llm_factory.create_model(model_code=self.model_name)

        # Load system prompt if needed
        if self.use_script_prompt:
            # Load the specific script generation prompt
            try:
                self.system_prompt = PromptLoader.load_script_system_prompt()
                logger.info("Successfully loaded and applied script system prompt")

                # Apply the system prompt to the model
                self.model = self.model.with_config(
                    {"system_prompt": self.system_prompt}
                )
            except Exception as e:
                logger.error(f"Error loading prompt: {str(e)}")

        # Create the tools
        self.tools = self._create_tools()

        # Create the agent with standard config
        self.agent = create_react_agent(self.model, tools=self.tools)

    def _create_tools(self) -> List[Any]:
        """Create tools for the agent."""

        # Store editor reference to avoid closure reference to self
        editor = self.editor

        @tool
        def clear() -> str:
            """Clear the current string content."""
            if Config.VERBOSE_TOOL_OUTPUT:
                print(f"Tool called: clear()")
            result = editor.clear()
            if Config.VERBOSE_TOOL_OUTPUT:
                print(f"Tool output: {result}")
            return result

        @tool
        def view(view_range: Optional[List[int]] = None) -> str:
            """
            View the content of the string, optionally within a specific range of lines.
            If view_range is provided, it should be a list of two integers: [start_line, end_line].
            Lines are 1-indexed.
            """
            if Config.VERBOSE_TOOL_OUTPUT:
                print(f"Tool called: view(view_range={view_range})")
            result = editor.view(view_range)
            if Config.VERBOSE_TOOL_OUTPUT:
                print(
                    f"Tool output (truncated): {result[:100]}..."
                    if len(result) > 100
                    else f"Tool output: {result}"
                )
            return result

        @tool
        def insert(insert_line: int, new_str: str) -> str:
            """
            Insert text at a specific location in the string.
            insert_line: The line number after which to insert the text (0 for beginning of file).
            new_str: The text to insert.
            """
            if Config.VERBOSE_TOOL_OUTPUT:
                print(f"Tool called: insert(insert_line={insert_line})")
            result = editor.insert(insert_line, new_str)
            if Config.VERBOSE_TOOL_OUTPUT:
                print(
                    f"Tool output (truncated): {result[:100]}..."
                    if len(result) > 100
                    else f"Tool output: {result}"
                )
            return result

        @tool
        def run() -> str:
            """
            Validate the code in the string editor.
            Returns 'true' if the code is valid, or an error message if not.
            """
            if Config.VERBOSE_TOOL_OUTPUT:
                print(f"Tool called: run()")
            result = editor.run()
            if Config.VERBOSE_TOOL_OUTPUT:
                print(f"Tool output: {result}")
            return result

        @tool
        def execute_script() -> str:
            """
            Execute the Python script in the string editor and return the output.
            """
            if Config.VERBOSE_TOOL_OUTPUT:
                print(f"Tool called: execute_script()")
            result = editor.execute_script()
            if Config.VERBOSE_TOOL_OUTPUT:
                print(f"Tool output: {result}")
            return result

        # Return the list of tools
        return [clear, view, insert, run, execute_script]

    def set_content(self, content: str) -> None:
        """Set the content of the string editor."""
        self.editor = StringEditor(content)

    def get_content(self) -> str:
        """Get the current content of the string editor."""
        return self.editor.get_content()

    def get_script_output(self) -> Optional[str]:
        """Get the most recently generated script output."""
        return self.editor.get_script_output()

    def run(self, user_input: str) -> Dict[str, Any]:
        """
        Run the agent with the given user input.

        Args:
            user_input: The user's query or instructions.

        Returns:
            The result of the agent's execution.
        """
        # Create proper messages with system prompt
        if self.use_script_prompt and self.system_prompt:
            # Create a system message with the loaded prompt
            messages = [
                SystemMessage(content=self.system_prompt),
                HumanMessage(content=user_input),
            ]
        else:
            messages = [HumanMessage(content=user_input)]

        # Add proper configuration with recursion_limit
        callbacks = []
        if Config.MINIMAL_TOOL_TRACE:
            # Add the minimal tool callback handler for simplified output
            callbacks = [MinimalToolCallbackHandler()]

        config = RunnableConfig(recursion_limit=200, callbacks=callbacks)

        # Invoke the agent with configuration
        return self.agent.invoke({"messages": messages}, config=config)


def main():
    """Main entry point for the script builder."""
    parser = argparse.ArgumentParser(description="Python Script Builder")
    parser.add_argument(
        "description",
        nargs="?",
        help="Description of the script to generate",
        default="Create a simple Python script that calculates and prints the first 10 Fibonacci numbers.",
    )
    parser.add_argument(
        "--mc",
        default=Config.DEFAULT_MODEL_CODE,
        help="Model code to use (format: 'OA:model' for OpenAI, 'OR:provider/model' for OpenRouter, 'AT:model' for Anthropic)",
    )
    args = parser.parse_args()

    # Print the model code being used
    print(f"Using model: {args.mc}")

    # Create a new Script editor
    agent = ScriptEditor(model_code=args.mc, use_script_prompt=True)

    logger.info(f"Generating script for: {args.description}")
    agent.run(args.description)

    # Print the generated code
    print_code(agent.get_content(), title="Generated Python Script")

    # Get and display script output if available
    script_output = agent.get_script_output()
    if script_output:
        print("\nScript Output:")
        print("-" * 80)
        print(script_output)
        print("-" * 80)

    return 0


if __name__ == "__main__":
    sys.exit(main())
