from pathlib import Path
from typing import Dict
import logging


class Config:
    """
    Configuration settings for the script generation system.
    """

    # Directories
    PROMPTS_DIR: Path = Path("./prompts")

    # Model Configuration
    MODEL_CODES: Dict[str, str] = {
        "OPENAI": "OA:gpt-4o",
        "CLAUDE": "AT:claude-3-sonnet-20240229",
        "CLAUDE_OR": "OR:anthropic/claude-3.7-sonnet",
        "CLAUDE_SONNET4": "AT:claude-sonnet-4-20250514",
        "GEMINI": "OR:google/gemini-2.5-pro-preview-03-25",
        "GEMINI_FLASH": "OR:google/gemini-2.5-flash-preview-05-20",
        "O3MINI_OR": "OR:openai/o3-mini-high",
        "O3MINI": "OA:o3-mini-2025-01-31",
        "O3": "OA:o3-2025-04-16",
        "O4MINI": "OA:o4-mini-2025-04-16",
    }
    DEFAULT_MODEL_CODE: str = MODEL_CODES["CLAUDE_SONNET4"]  # Using Anthropic Claude Sonnet 4

    # Tool Configuration
    VERBOSE_TOOL_OUTPUT: bool = False  # Set to True for detailed tool output logging
    MINIMAL_TOOL_TRACE: bool = (
        True  # Set to True for minimal one-line tool usage logging
    )
    
    # Script Execution Configuration
    DEFAULT_SCRIPT_TIMEOUT: int = 30  # Default timeout for script execution in seconds

    # Prompt file names
    SCRIPT_SYSTEM_PROMPT_FILE: str = "script_system_prompt.md"
    MZN_TUNING_PROMPT_FILE: str = "mzn-tuning.md"

    # Logger Configuration
    LOG_LEVEL: int = logging.INFO

    @classmethod
    def ensure_directories(cls) -> None:
        """Ensure required directories exist"""
        cls.PROMPTS_DIR.mkdir(exist_ok=True)

    @classmethod
    def configure_logging(cls, verbose: bool = False) -> None:
        """Configure logging with appropriate format and level"""
        log_level = logging.DEBUG if verbose else cls.LOG_LEVEL

        # Import here to avoid circular imports
        from lmtune.utils import setup_rich_logging

        setup_rich_logging(level=log_level)

        # Set high level for noisy libraries
        logging.getLogger("httpx").setLevel(logging.WARNING)
        logging.getLogger("httpcore").setLevel(logging.WARNING)
