#!/usr/bin/env python3
"""Convert MiniZinc DZN files to JSON format using PyMzn."""

import json
import argparse
from pathlib import Path
import pymzn


def convert_dzn_to_json(dzn_file: Path, output_file: Path = None) -> bool:
    """Convert a single DZN file to JSON format using PyMzn."""
    if output_file is None:
        # Default output filename
        output_file = dzn_file.with_suffix('.json')
    
    try:
        # Use PyMzn to parse the DZN file
        data = pymzn.dzn2dict(str(dzn_file))
        
        # Write to JSON file
        with open(output_file, 'w') as f:
            json.dump(data, f, indent=2)
        
        print(f"Converted {dzn_file.name} -> {output_file.name}")
        return True
    except Exception as e:
        print(f"Error converting {dzn_file.name}: {e}")
        return False


def convert_directory(directory: Path, pattern: str = "*.dzn") -> None:
    """Convert all DZN files in a directory to JSON format."""
    dzn_files = list(directory.glob(pattern))
    
    if not dzn_files:
        print(f"No DZN files found matching pattern '{pattern}' in {directory}")
        return
    
    print(f"Found {len(dzn_files)} DZN files to convert")
    
    success_count = 0
    for dzn_file in dzn_files:
        if convert_dzn_to_json(dzn_file):
            success_count += 1
    
    print(f"\nConversion complete: {success_count}/{len(dzn_files)} files converted successfully")


def main():
    """Command line interface for DZN to JSON conversion using PyMzn."""
    parser = argparse.ArgumentParser(description="Convert MiniZinc DZN files to JSON format using PyMzn")
    parser.add_argument("input", help="Input DZN file or directory")
    parser.add_argument("-o", "--output", help="Output JSON file (only for single file conversion)")
    parser.add_argument("-p", "--pattern", default="*.dzn", 
                       help="File pattern for directory conversion (default: *.dzn)")
    
    args = parser.parse_args()
    
    input_path = Path(args.input)
    
    if input_path.is_file():
        # Single file conversion
        output_path = Path(args.output) if args.output else None
        convert_dzn_to_json(input_path, output_path)
    elif input_path.is_dir():
        # Directory conversion
        if args.output:
            print("Warning: --output option is ignored for directory conversion")
        convert_directory(input_path, args.pattern)
    else:
        print(f"Error: {input_path} is neither a file nor a directory")
        return 1
    
    return 0


if __name__ == "__main__":
    exit(main())