"""
Tools for working with knowledge base memos.
"""

from typing import Dict, Any, Optional

from langchain_core.tools import tool
from pathlib import Path
import os


def get_memo_path(memo_name: str) -> Path:
    """
    Determine the path for a memo file based on memo name.

    Args:
        memo_name: The name of the memo

    Returns:
        Path object for the memo file
    """
    # Project root directory
    root_dir = Path(__file__).parent.parent.parent
    memos_dir = root_dir / "memos"

    # Ensure memos directory exists
    os.makedirs(memos_dir, exist_ok=True)

    # Construct filename
    filename = f"memo_{memo_name.lower()}.md"

    return memos_dir / filename


def read_memo_file(memo_name: str) -> str:
    """
    Read content from a memo file or create it if it doesn't exist.

    Args:
        memo_name: The name of the memo

    Returns:
        String content of the memo file
    """
    file_path = get_memo_path(memo_name)

    # If file doesn't exist, create it with default content
    if not file_path.exists():
        with open(file_path, "w") as f:
            title = f"# Knowledge Base for {memo_name.capitalize()}"

            f.write(f"{title}\n\n")
            f.write("## Insights\n\n")
            f.write("No insights recorded yet.\n\n")
            f.write("## Best Solutions\n\n")
            f.write("No solutions recorded yet.\n\n")
            f.write("## Example Code\n\n")
            f.write("No code examples recorded yet.\n\n")
            f.write("## What Works\n\n")
            f.write("No successful techniques recorded yet.\n\n")
            f.write("## What Doesn't Work\n\n")
            f.write("No unsuccessful techniques recorded yet.\n\n")

    # Read and return the file content
    with open(file_path, "r") as f:
        content = f.read()

    return content


def update_memo_file(memo_name: str, content: str) -> None:
    """
    Update a memo file with new content.

    Args:
        memo_name: The name of the memo
        content: The new content to write
    """
    file_path = get_memo_path(memo_name)

    # Write content to file
    with open(file_path, "w") as f:
        f.write(content)


class MemoTools:
    """
    A class containing tools for working with knowledge base memos.
    """

    @staticmethod
    def create_memo_tools():
        """Create and return memo tools for the agent."""

        @tool
        def view_memo(memo_name: str) -> str:
            """
            View the knowledge base for the specified memo.

            Args:
                memo_name: The name of the memo

            Returns:
                String content of the memo file
            """
            return read_memo_file(memo_name)

        @tool
        def edit_memo(memo_name: str, content: str) -> str:
            """
            Update the knowledge base for the specified memo.

            Args:
                memo_name: The name of the memo
                content: The new content to write

            Returns:
                Confirmation message
            """
            update_memo_file(memo_name, content)
            return f"Memo file for {memo_name} updated successfully."

        return [view_memo, edit_memo]
