from pathlib import Path
from lmtune.config import Config


class PromptLoader:
    """
    Simple utility class for loading prompts from template files.
    """

    @classmethod
    def load(cls, prompt_filename: str) -> str:
        """
        Load a prompt from a file in the configured prompts directory.

        Args:
            prompt_filename: Name of the prompt file

        Returns:
            The prompt text
        """
        prompt_path = Config.PROMPTS_DIR / prompt_filename

        if not prompt_path.exists():
            raise FileNotFoundError(
                f"Prompt file {prompt_filename} not found in {prompt_path}"
            )

        return prompt_path.read_text(encoding="utf-8").strip()

    @classmethod
    def load_script_system_prompt(cls) -> str:
        """
        Convenient shortcut to load the script system prompt.

        Returns:
            The script system prompt text
        """
        return cls.load(Config.SCRIPT_SYSTEM_PROMPT_FILE)
        
    @classmethod
    def load_mzn_tuning_prompt(cls) -> str:
        """
        Convenient shortcut to load the MiniZinc tuning prompt.

        Returns:
            The MiniZinc tuning prompt text
        """
        return cls.load(Config.MZN_TUNING_PROMPT_FILE)
