"""
Script runner for lmtune-generated Python scripts.
"""

import os
import sys
import argparse
import glob
import logging
import json
import subprocess
import tempfile
from pathlib import Path
from typing import Optional, Dict, Any

from rich.console import Console
from rich.syntax import Syntax
from rich.panel import Panel
from rich.json import JSON

from lmtune.config import Config
from lmtune.utils import setup_rich_logging, print_code

logger = logging.getLogger(__name__)
console = Console()

def find_latest_script(problem_folder: str, problem_type: str) -> Optional[str]:
    """
    Find the latest lmtuner*.py script in the specified problem folder.
    
    Args:
        problem_folder: Path to the problem folder
        problem_type: Type of problem (subfolder within problem_folder)
        
    Returns:
        Path to the latest script or None if no scripts found
    """
    problem_dir = os.path.join(os.path.expanduser(problem_folder), problem_type)
    
    # Find all lmtuner*.py files
    scripts = glob.glob(os.path.join(problem_dir, "lmtuner*.py"))
    
    if not scripts:
        return None
    
    # Sort by modification time (newest first)
    scripts.sort(key=lambda x: os.path.getmtime(x), reverse=True)
    
    return scripts[0]

def run_script(script_path: str, problem_folder: str, problem_type: str, instance: str) -> int:
    """
    Run a generated script with the specified parameters.
    
    Args:
        script_path: Path to the Python script to run
        problem_folder: Path to the problem folder 
        problem_type: Type of problem (subfolder within problem_folder)
        instance: Instance file name
        
    Returns:
        Exit code of the script execution
    """
    if not os.path.isfile(script_path):
        print(f"Error: Script not found at {script_path}")
        return 1
    
    # Load instance data
    instance_file = os.path.join(os.path.expanduser(problem_folder), problem_type, instance)
    
    try:
        with open(instance_file, 'r') as f:
            try:
                # Try to parse as JSON
                input_json = json.load(f)
            except json.JSONDecodeError:
                # If not valid JSON, return as text
                f.seek(0)
                input_json = f.read()
    except FileNotFoundError:
        print(f"Error: Instance file not found at {instance_file}")
        return 1
    
    # Create a temporary helper module file
    with tempfile.TemporaryDirectory() as temp_dir:
        # Create helper module
        helper_module_code = f'''
# Helper module for lmtune scripts
import json

# Store input data
_INPUT_DATA = {input_json if isinstance(input_json, dict) else json.dumps(input_json)}

def input_data():
    """Get the input data for the problem instance.
    
    Returns:
        The parsed input data (JSON object or string)
    """
    return _INPUT_DATA

def output_results(result_dict):
    """Format and print results as JSON.
    
    Args:
        result_dict: A dictionary containing the script results
    """
    print(json.dumps(result_dict))
'''
        
        # Write helper module
        helper_module_path = os.path.join(temp_dir, "lmtune_helpers.py")
        with open(helper_module_path, 'w') as module_file:
            module_file.write(helper_module_code)
        
        # Set up the environment to include the directory with our helper module
        env = os.environ.copy()
        
        # Use PYTHONPATH to make the helper module importable
        if 'PYTHONPATH' in env:
            env['PYTHONPATH'] = f"{temp_dir}:{env['PYTHONPATH']}"
        else:
            env['PYTHONPATH'] = temp_dir
        
        # Display what we're running
        print(f"Running script: {script_path}")
        print(f"With instance: {instance}")
        
        # Run the script with UV
        try:
            result = subprocess.run(
                ["uv", "run", script_path], 
                capture_output=True, 
                text=True,
                env=env
            )
            
            # Print stdout
            if result.stdout:
                # Try to parse as JSON for nice formatting
                try:
                    json_data = json.loads(result.stdout.strip())
                    console.print("\nScript Output:", style="bold cyan")
                    console.print(JSON.from_data(json_data))
                except (json.JSONDecodeError, TypeError):
                    # If not valid JSON, print as regular text
                    console.print("\nScript Output:", style="bold cyan")
                    console.print(Panel(result.stdout.strip(), style="white on black"))
            
            # Print stderr if there was an error
            if result.returncode != 0:
                console.print("\nScript Error:", style="bold red")
                console.print(Panel(result.stderr, style="red"))
                return result.returncode
            
            return 0
            
        except Exception as e:
            console.print(f"\nError running script: {e}", style="bold red")
            return 1

def main():
    """Main entry point for the script runner."""
    parser = argparse.ArgumentParser(description="LMTune Script Runner")
    
    parser.add_argument(
        "--problem-folder",
        required=True,
        help="Folder path containing problem subfolders",
    )
    parser.add_argument(
        "--problem",
        required=True,
        help="Problem type (subfolder within the problem folder)",
    )
    parser.add_argument(
        "--instance",
        required=True,
        help="Instance file name within the problem subfolder",
    )
    parser.add_argument(
        "--script",
        help="Path to the Python script to run (defaults to latest lmtuner*.py script)",
    )
    
    args = parser.parse_args()
    
    # Configure logging
    Config.configure_logging(verbose=False)
    
    # Resolve script path
    script_path = args.script
    if not script_path:
        script_path = find_latest_script(args.problem_folder, args.problem)
        if not script_path:
            print(f"Error: No generated scripts found in {os.path.join(args.problem_folder, args.problem)}")
            return 1
        print(f"Using latest script: {script_path}")
    
    # Run the script
    return run_script(
        script_path=script_path,
        problem_folder=args.problem_folder,
        problem_type=args.problem,
        instance=args.instance
    )

if __name__ == "__main__":
    sys.exit(main())