"""
Utility functions for the lmtune system.
"""

import logging
import os
from pathlib import Path
from typing import Dict, List, Any, Optional
from rich.console import Console
from rich.syntax import Syntax
from rich.panel import Panel
from rich.logging import RichHandler
from io import StringIO

# Create a rich console for formatted output
console = Console()


def format_python_code(code_string, title=None):
    """
    Format Python code with syntax highlighting using rich.

    Args:
        code_string: The Python code to format
        title: Optional title to show above the code

    Returns:
        Formatted code as a string that can be printed
    """
    # If no code provided, return empty string
    if not code_string or code_string.strip() == "":
        return ""

    # Create a syntax object with the code
    syntax = Syntax(
        code_string,
        "python",
        theme="monokai",
        line_numbers=True,
        code_width=88,
        word_wrap=True,
        background_color="black",
    )

    # If a title was provided, wrap in a panel
    if title:
        panel = Panel(
            syntax,
            title=f"[bold cyan]{title}[/bold cyan]",
            border_style="cyan",
            padding=(1, 2),
            expand=False
        )
        return panel

    return syntax


def print_code(code_string, title=None):
    """
    Print formatted Python code to the console.

    Args:
        code_string: The Python code to print
        title: Optional title to show above the code
    """
    formatted = format_python_code(code_string, title)
    console.print(formatted)


def get_formatted_code_string(code_string, title=None):
    """
    Get a string representation of formatted code (for logging).

    Args:
        code_string: The Python code to format
        title: Optional title to show above the code

    Returns:
        A string containing formatted code (with ANSI color codes)
    """
    # Create a string buffer
    string_io = StringIO()
    # Create a console that writes to the buffer
    buffer_console = Console(file=string_io, highlight=True)
    # Write the formatted code to the buffer
    formatted = format_python_code(code_string, title)
    buffer_console.print(formatted)
    # Return the contents of the buffer
    return string_io.getvalue()


def setup_rich_logging(level=logging.INFO):
    """
    Set up rich logging handler for more visually appealing logs.

    Args:
        level: The logging level to use
    """
    # Configure rich handler
    rich_handler = RichHandler(
        rich_tracebacks=True, markup=True, show_time=False, show_path=False
    )

    # Remove existing handlers
    for handler in logging.root.handlers[:]:
        logging.root.removeHandler(handler)

    # Add rich handler
    logging.basicConfig(
        level=level, format="%(message)s", datefmt="[%X]", handlers=[rich_handler]
    )

    # Set high level for noisy libraries
    logging.getLogger("httpx").setLevel(logging.WARNING)
    logging.getLogger("httpcore").setLevel(logging.WARNING)
