import torch
import torch.nn as nn
import torch.nn.functional as F
from layers.Autoformer_EncDec import series_decomp
from layers.Energy import EnergyEnhancer, EnergyPredictor

class Model(nn.Module):
    def __init__(self, configs, individual=False):
        """
        individual: Bool, whether shared model among different variates.
        """
        super(Model, self).__init__()
        self.task_name = configs.task_name
        self.seq_len = configs.seq_len
        if self.task_name == 'classification' or self.task_name == 'anomaly_detection' or self.task_name == 'imputation':
            self.pred_len = configs.seq_len
        else:
            self.pred_len = configs.pred_len
        # Series decomposition block from Autoformer
        self.decompsition = series_decomp(configs.moving_avg)
        self.individual = individual
        self.channels = configs.enc_in

        if self.individual:
            self.Linear_Seasonal = nn.ModuleList()
            self.Linear_Trend = nn.ModuleList()

            for i in range(self.channels):
                self.Linear_Seasonal.append(
                    nn.Linear(self.seq_len, self.pred_len))
                self.Linear_Trend.append(
                    nn.Linear(self.seq_len, self.pred_len))

                self.Linear_Seasonal[i].weight = nn.Parameter(
                    (1 / self.seq_len) * torch.ones([self.pred_len, self.seq_len]))
                self.Linear_Trend[i].weight = nn.Parameter(
                    (1 / self.seq_len) * torch.ones([self.pred_len, self.seq_len]))
        else:
            self.Linear_Seasonal = nn.Linear(self.seq_len, self.pred_len)
            self.Linear_Trend = nn.Linear(self.seq_len, self.pred_len)

            self.Linear_Seasonal.weight = nn.Parameter(
                (1 / self.seq_len) * torch.ones([self.pred_len, self.seq_len]))
            self.Linear_Trend.weight = nn.Parameter(
                (1 / self.seq_len) * torch.ones([self.pred_len, self.seq_len]))
        
        self.embed_dim = configs.embed_dim
        self.embed_dim_out = configs.embed_dim_out
        self.lambda_init = configs.lambda_init
        self.alpha = configs.alpha
        self.energy_enhancer = EnergyEnhancer(self.seq_len, self.channels, self.embed_dim, self.lambda_init, self.alpha)
        self.energy_predictor = EnergyPredictor(self.seq_len, self.pred_len, self.embed_dim_out)

    def encoder(self, x):
        x_enhanced, x_inverse_fft, loss_nonstat = self.energy_enhancer(x)
        seasonal_init, trend_init = self.decompsition(x_enhanced)
        seasonal_init, trend_init = seasonal_init.permute(
            0, 2, 1), trend_init.permute(0, 2, 1)
        if self.individual:
            seasonal_output = torch.zeros([seasonal_init.size(0), seasonal_init.size(1), self.pred_len],
                                          dtype=seasonal_init.dtype).to(seasonal_init.device)
            trend_output = torch.zeros([trend_init.size(0), trend_init.size(1), self.pred_len],
                                       dtype=trend_init.dtype).to(trend_init.device)
            for i in range(self.channels):
                seasonal_output[:, i, :] = self.Linear_Seasonal[i](
                    seasonal_init[:, i, :])
                trend_output[:, i, :] = self.Linear_Trend[i](
                    trend_init[:, i, :])
        else:
            seasonal_output = self.Linear_Seasonal(seasonal_init)
            trend_output = self.Linear_Trend(trend_init)

        y = seasonal_output + trend_output # (bs, n_vars, pred_len)
        y =  y.permute(0, 2, 1) # (bs, pred_len, n_vars)
        y = self.energy_predictor(x_inverse_fft, y) # (bs, pred_len, n_vars)

        return y, loss_nonstat

    def forecast(self, x_enc):
        # Encoder
        dec_out, loss_nonstat = self.encoder(x_enc)
        return dec_out, loss_nonstat

    def forward(self, x_enc, x_mark_enc=None, x_dec=None, x_mark_dec=None, mask=None):
        if self.task_name == 'long_term_forecast' or self.task_name == 'short_term_forecast':
            dec_out, loss_nonstat = self.forecast(x_enc)
            return dec_out[:, -self.pred_len:, :], loss_nonstat  # [B, L, D]
        return None
