import os
import torch.distributed as dist
import numpy as np
import torch


def convert_and_to(d, device):
    """
    Recursively processes a dictionary to convert numpy arrays to tensors and move them to a target device.

    Args:
    dictionary (dict): The dictionary to process.
    target_device (torch.device): The target device to move tensors to.

    Returns:
    dict: The processed dictionary with tensors moved to the target device.
    """
    processed_dict = {}
    for key, value in d.items():
        if isinstance(value, dict):
            # Recursively process sub-dictionaries
            processed_dict[key] = convert_and_to(value, device)
        elif isinstance(value, np.ndarray):
            # Convert numpy arrays to tensors and move to target device
            processed_dict[key] = torch.from_numpy(value).to(device)
        elif torch.is_tensor(value):
            # Convert numpy arrays to tensors and move to target device
            processed_dict[key] = value.to(device)
        else:
            # Keep other values as they are
            processed_dict[key] = value
    return processed_dict

def get_rank0_device(devices):
    if dist.is_initialized():
        if dist.get_rank() == 0:
            rank0_device = torch.device('cuda', int(os.environ['LOCAL_RANK'])) if torch.cuda.is_available() else torch.device('cpu')
            return rank0_device
        else:
            # Optionally, you can return the device for non-rank 0 processes as well
            return torch.device('cuda', int(os.environ['LOCAL_RANK'])) if torch.cuda.is_available() else torch.device('cpu')
    else:
        # Fallback to default device if distributed is not initialized
        if isinstance(devices, int):
            return torch.device('cuda', devices)
        
        return torch.device('cuda', devices[0])
    