import torch
import numpy as np

class AutoClip:
    def __init__(self, 
                 grad_clip_enabled: bool,
                 auto_clip_enabled: bool,
                 default_clip_value: float, 
                 auto_clip_percentile: float, 
                 auto_clip_hist_len: int, 
                 auto_clip_min_len: int,
                 ):
        self.grad_clip_enabled = grad_clip_enabled
        self.auto_clip_enabled = auto_clip_enabled
        self.default_clip_value = default_clip_value
        self.percentile = auto_clip_percentile

        history_size = auto_clip_hist_len
        min_history_length = auto_clip_min_len

        self.min_history_length = min_history_length

        if not (self.grad_clip_enabled and self.auto_clip_enabled):
            self.history_size = 1
        else:
            self.history_size = history_size

        self.gradient_norms = torch.zeros(self.history_size)
        self.cur_idx = 0
        self.cnt = 0

    def update_gradient_norm_history(self, gradient_norm):
        self.gradient_norms[self.cur_idx] = gradient_norm
        self.cur_idx = (self.cur_idx + 1) % self.history_size
        self.cnt += 1

    def get_clip_value(self):
        if not self.grad_clip_enabled:
            return np.inf

        if not self.auto_clip_enabled or self.cnt < self.min_history_length:
            return self.default_clip_value

        return torch.quantile(self.gradient_norms, self.percentile)

    def get_last_gradient_norm(self):
        if self.grad_clip_enabled:
            return self.gradient_norms[self.cur_idx - 1]
        else:
            return torch.tensor(float('nan'))
