import json
from . import bucket
from . import bigquery_client
from . import config
from json_repair import repair_json
from google.cloud import bigquery


def chunkify(lst, n):
    return [lst[i:i + n] for i in range(0, len(lst), n)]


def parse_json(text: str):
    """Parse a JSON string, repairing it if necessary."""

    revised_text = text.replace("```", "").replace("json", "").replace("\\\"", "\\\"").replace("\\", "\\\\").replace("\\\"", "\"").replace("\\\",\n", "\",\n").replace("\n", " ")
    try:
        return json.loads(revised_text)
    except json.JSONDecodeError:
        try:
            return json.loads(repair_json(revised_text))
        except json.JSONDecodeError:
            raise ValueError(f"Failed to decode the JSON string:\n{text}\n\nThe modified (pre-repair) string is:\n{revised_text}")


def upload_to_table(blob_name, dataset_id="pairings", project_id=config.GCP_PROJECT_ID, delete=False):
    """Upload a JSONL file to a BigQuery table."""

    blob = bucket.blob(blob_name)
    blob.reload()

    table_id = blob.name.replace("/", "_").replace(".jsonl", "").replace(".", "")
    if delete:
        bigquery_client.delete_table(f"{config.GCP_PROJECT_ID}.pairings.{table_id}", not_found_ok=True)
        
    dataset_ref = bigquery_client.dataset(dataset_id)
    table_ref = dataset_ref.table(table_id)

    job_config = bigquery.LoadJobConfig(
        source_format=bigquery.SourceFormat.NEWLINE_DELIMITED_JSON,
        autodetect=True  # Automatically detect schema from JSONL file
    )

    load_job = bigquery_client.load_table_from_uri(
        'gs://' + blob.id[:-(len(str(blob.generation)) + 1)], table_ref, job_config=job_config
    )

    load_job.result()

    table_url = f"https://console.cloud.google.com/bigquery?project={project_id}&p={project_id}&d={dataset_id}&t={table_id}&page=table"
    print(f"Table URL: {table_url}")
