import os
import sys
from PIL import Image, ImageDraw, ImageFont

width, height = 32, 32
font_size = int(width * 0.85)
font_color = (255, 255, 255)
font_dir = "VIPER/multilingual_mixed_fonts"

if len(sys.argv) != 2:
    print("Usage: python3 image_generator_embedding.py < mychars.txt [output_directory]")
    sys.exit(1)

output_root = sys.argv[1]
output_dir = os.path.join(output_root, "images")
missing_chars_file = os.path.join(output_root, "missing_chars.txt")
saving_chars_file = os.path.join(output_root, "saving_chars.txt")

os.makedirs(output_dir, exist_ok=True)

open(missing_chars_file, "w", encoding="utf-8").close()
open(saving_chars_file, "w", encoding="utf-8").close()

font_files = [os.path.join(font_dir, f) for f in os.listdir(font_dir) if f.endswith((".ttf", ".otf"))]
if not font_files:
    raise RuntimeError(f"No font files found in directory: {font_dir}")

def is_char_supported(font_path, char):
    txt_path = f"{os.path.splitext(font_path)[0]}.txt"
    if not os.path.exists(txt_path):
        raise FileNotFoundError(f"Glyph file not found: {txt_path}")
    
    with open(txt_path, "r", encoding="utf-8") as file:
        glyphs = file.read()
    
    return char in glyphs

for line in sys.stdin:
    unicode_text = line.strip()
    if not unicode_text:
        continue

    codepoint = ord(unicode_text)
    print(f"Processing character: {unicode_text} (U+{codepoint:X})")

    selected_font = None
    for font_path in font_files:
        if is_char_supported(font_path, unicode_text):
            selected_font = font_path
            break

    if not selected_font:
        with open(missing_chars_file, "a", encoding="utf-8") as missing_file:
            missing_file.write(f"{unicode_text} U+{codepoint:X}\n")
        print(f"Character not supported: {unicode_text} (U+{codepoint:X})")
        continue

    im = Image.new("RGB", (width, height), "black")
    draw = ImageDraw.Draw(im)
    try:
        unicode_font = ImageFont.truetype(selected_font, font_size)
        bbox = draw.textbbox((0, 0), unicode_text, font=unicode_font)
        text_width = bbox[2] - bbox[0]
        text_height = bbox[3] - bbox[1]
        ascent, descent = unicode_font.getmetrics()
        x_offset = (width - text_width) // 2
        bbox_center_y = (bbox[1] + bbox[3]) / 2
        image_center_y = height / 2
        y_offset = int(image_center_y - bbox_center_y)

        draw.text((x_offset, y_offset), unicode_text, font=unicode_font, fill=font_color)

        image_name = f"text-other_{codepoint:X}.png"
        image_path = os.path.join(output_dir, image_name)
        im.save(image_path)
        print(f"Saved image: {image_path}")

        with open(saving_chars_file, "a", encoding="utf-8") as saving_file:
            saving_file.write(f"{unicode_text} U+{codepoint:X} {image_name}\n")

    except Exception as e:
        print(f"Error rendering {unicode_text} with {selected_font}: {e}")
        with open(missing_chars_file, "a", encoding="utf-8") as missing_file:
            missing_file.write(f"{unicode_text} U+{codepoint:X} Error: {str(e)}\n")

def count_lines(file_path):
    if not os.path.exists(file_path):
        return 0
    with open(file_path, "r", encoding="utf-8") as f:
        return sum(1 for _ in f)

missing_count = count_lines(missing_chars_file)
saving_count = count_lines(saving_chars_file)

print(f"Images have been saved in '{output_dir}' directory.")
print(f"Characters not supported are logged in '{missing_chars_file}' ({missing_count} characters).")
print(f"Characters supported and saved are logged in '{saving_chars_file}' ({saving_count} characters).")