/* LUT-GEMM
 * Copyright (c) 2024-present NAVER Cloud Corp. All rights reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *     http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#ifndef KERNELS_AQLM_HPP
#define KERNELS_AQLM_HPP


namespace kernel{

#include <stdio.h>

template<bool use_bfloat16, size_t group_size>
__global__ void Code1x16MatVec(
  const int4* __restrict__ A,
  const int4* __restrict__ B,
        int4* __restrict__ C,
  const int4* __restrict__ codebook,
  int prob_m,
  int prob_k
) {
  int a_gl_stride = prob_k / 8 / group_size;
  int a_gl_rd = (blockDim.x / 32) * blockIdx.x + (threadIdx.x / 32);
  bool pred = a_gl_rd < prob_m;
  int b_gl_rd = 0;
  int c_gl_wr = a_gl_rd;
  a_gl_rd = a_gl_stride * a_gl_rd + threadIdx.x % 32;
  int a_gl_end = a_gl_rd + a_gl_stride - threadIdx.x % 32;

  __shared__ int4 sh_b[32 * (group_size + 1)];
  float res = 0;

  int iters = (prob_k / group_size + group_size * 32 - 1) / (group_size * 32);
  while (iters--) {
    // We pad shared memory to avoid bank conflicts during reads
    __syncthreads();
    for (int i = threadIdx.x; i < 32 * group_size; i += blockDim.x) {
      if (8 * (b_gl_rd + i) < prob_k)
        sh_b[(group_size + 1) * (i / group_size) + i % group_size] = B[b_gl_rd + i];
    }
    __syncthreads();
    b_gl_rd += 32 * group_size;

    int b_sh_rd = (group_size + 1) * (threadIdx.x % 32);
    if (pred && a_gl_rd < a_gl_end) {
      const uint16_t* enc = reinterpret_cast<const uint16_t*>(&A[a_gl_rd]);
      #pragma unroll
      for (int i = 0; i < 8; i++) {
        uint32_t dec[group_size / 2];
        // We bypass the L1 cache to avoid massive amounts of memory streaming that doesn't
        // actually help us; this brings > 2x speedup.
        asm volatile (
          "ld.cg.global.v4.u32 {%0, %1, %2, %3}, [%4];"
          : "=r"(dec[0]), "=r"(dec[1]), "=r"(dec[2]), "=r"(dec[3])
          : "l"((void*) &codebook[(group_size / 8) * enc[i]])
        );
        if constexpr (group_size == 16) {
          asm volatile (
            "ld.cg.global.v4.u32 {%0, %1, %2, %3}, [%4];"
            : "=r"(dec[4]), "=r"(dec[5]), "=r"(dec[6]), "=r"(dec[7])
            : "l"((void*) &codebook[(group_size / 8) * enc[i] + 1])
          );
        }
        if constexpr (use_bfloat16) {
        #if !defined(__CUDA_ARCH__) || (__CUDA_ARCH__ >= 800)
          nv_bfloat162* a = reinterpret_cast<nv_bfloat162*>(&dec);
          nv_bfloat162* b = reinterpret_cast<nv_bfloat162*>(&sh_b[b_sh_rd]);
          nv_bfloat162 res2 = {};
          #pragma unroll
          for (int j = 0; j < group_size / 2; j++)
            res2 = __hfma2(a[j], b[j], res2);
          res += __bfloat162float(res2.x) + __bfloat162float(res2.y);
        #endif
        } else {
          half2* a = reinterpret_cast<half2*>(&dec);
          half2* b = reinterpret_cast<half2*>(&sh_b[b_sh_rd]);
          half2 res2 = {};
          #pragma unroll
          for (int j = 0; j < group_size / 2; j++)
            res2 = __hfma2(a[j], b[j], res2);
          res += __half2float(res2.x) + __half2float(res2.y);
        }
        b_sh_rd += group_size / 8;
      }
      a_gl_rd += 32;
    }
  }

  if (pred) {
    #pragma unroll
    for (int i = 16; i > 0; i /= 2)
      res += __shfl_down_sync(0xffffffff, res, i);
    if (threadIdx.x % 32 == 0) {
      if constexpr (use_bfloat16) {
        reinterpret_cast<__nv_bfloat16*>(C)[c_gl_wr] = __float2bfloat16(res);
      } else {
        reinterpret_cast<__half*>(C)[c_gl_wr] = __float2half(res);
      }
    }
  }
}


template<bool use_bfloat16>
__global__ void Code2x8MatVec(
  const int4* __restrict__ A,
  const int4* __restrict__ B,
        int4* __restrict__ C,
  const int4* __restrict__ codebook,
  int prob_m,
  int prob_k
) {
  int a_gl_stride = prob_k / 8 / 8;
  int a_gl_rd = (blockDim.x / 32) * blockIdx.x + (threadIdx.x / 32);
  bool pred = a_gl_rd < prob_m;
  int b_gl_rd = 0;
  int c_gl_wr = a_gl_rd;
  a_gl_rd = a_gl_stride * a_gl_rd + threadIdx.x % 32;
  int a_gl_end = a_gl_rd + a_gl_stride - threadIdx.x % 32;
  int lane = threadIdx.x % 8;

  extern __shared__ int4 sh[];
  int4* sh_b = sh;
  int4* sh_code = sh_b + 32 * 9;
  int4* sh_code0 = sh_code;
  int4* sh_code1 = sh_code + 256 * 8;

  for (int i = threadIdx.x; i < 2 * 256; i += blockDim.x) {
    int4 dec = codebook[i];
    #pragma unroll
    for (int j = 0; j < 8; j++)
      sh_code[8 * i + (j + lane) % 8] = dec;
  }
  __syncthreads();

  float res = 0;

  int iters = (prob_k / 8 + 8 * 32 - 1) / (8 * 32);
  while (iters--) {
    // We pad shared memory to avoid bank conflicts during reads
    __syncthreads();
    for (int i = threadIdx.x; i < 32 * 8; i += blockDim.x) {
      if (b_gl_rd + i < prob_k / 8)
        sh_b[9 * (i / 8) + i % 8] = B[b_gl_rd + i];
    }
    __syncthreads();
    b_gl_rd += 32 * 8;

    int b_sh_rd = 9 * (threadIdx.x % 32);
    if (pred && a_gl_rd < a_gl_end) {
      const uint8_t* enc = reinterpret_cast<const uint8_t*>(&A[a_gl_rd]);
      #pragma unroll
      for (int i = 0; i < 8; i++) {
        if constexpr (use_bfloat16) {
        #if !defined(__CUDA_ARCH__) || (__CUDA_ARCH__ >= 800)
          nv_bfloat162* a0 = reinterpret_cast<nv_bfloat162*>(&sh_code0[8 * enc[2 * i + 0] + lane]);
          nv_bfloat162* a1 = reinterpret_cast<nv_bfloat162*>(&sh_code1[8 * enc[2 * i + 1] + lane]);
          nv_bfloat162*  b = reinterpret_cast<nv_bfloat162*>(&sh_b[b_sh_rd]);
          nv_bfloat162 res2 = {};
          #pragma unroll
          for (int j = 0; j < 4; j++)
            res2 = __hfma2(__hadd2(a0[j], a1[j]), b[j], res2);
          res += __bfloat162float(res2.x) + __bfloat162float(res2.y);
        #endif
        } else {
          half2* a0 = reinterpret_cast<half2*>(&sh_code0[8 * enc[2 * i + 0] + lane]);
          half2* a1 = reinterpret_cast<half2*>(&sh_code1[8 * enc[2 * i + 1] + lane]);
          half2*  b = reinterpret_cast<half2*>(&sh_b[b_sh_rd]);
          half2 res2 = {};
          #pragma unroll
          for (int j = 0; j < 4; j++)
            res2 = __hfma2(__hadd2(a0[j], a1[j]), b[j], res2);
          res += __half2float(res2.x) + __half2float(res2.y);
        }
        b_sh_rd++;
      }
      a_gl_rd += 32;
    }
  }

  if (pred) {
    #pragma unroll
    for (int i = 16; i > 0; i /= 2)
      res += __shfl_down_sync(0xffffffff, res, i);
    if (threadIdx.x % 32 == 0) {
      if constexpr (use_bfloat16) {
        reinterpret_cast<__nv_bfloat16*>(C)[c_gl_wr] = __float2bfloat16(res);
      } else {
        reinterpret_cast<__half*>(C)[c_gl_wr] = __float2half(res);
      }
    }
  }
}

template<bool use_bfloat16>
__global__ void Code2x8Dequant(
  const int4* __restrict__ A,
        int4* __restrict__ C,
  const int4* __restrict__ codebook,
  int prob_m,
  int prob_k
) {
  int a_gl_stride = prob_k / 8 / 8;
  int a_gl_rd = (blockDim.x / 32) * blockIdx.x + (threadIdx.x / 32);
  bool pred = a_gl_rd < prob_m;
  a_gl_rd = a_gl_stride * a_gl_rd + threadIdx.x % 32;
  int a_gl_end = a_gl_rd + a_gl_stride - threadIdx.x % 32;
  int lane = threadIdx.x % 8;

  int c_gl_stride = prob_k / 8;
  int c_gl_wr = (blockDim.x / 32) * blockIdx.x + (threadIdx.x / 32);
  c_gl_wr = c_gl_stride * c_gl_wr + (threadIdx.x % 32) * 8;

  extern __shared__ int4 sh[];
  int4* sh_code = sh;
  int4* sh_code0 = sh_code;
  int4* sh_code1 = sh_code + 256 * 8;

  for (int i = threadIdx.x; i < 2 * 256; i += blockDim.x) {
    int4 dec = codebook[i];
    #pragma unroll
    for (int j = 0; j < 8; j++)
      sh_code[8 * i + (j + lane) % 8] = dec;
  }
  __syncthreads();

  int iters = (prob_k / 8 - 1) / (8 * 32) + 1;
  while (iters--) {
    if (pred && a_gl_rd < a_gl_end) {
      const uint8_t* enc = reinterpret_cast<const uint8_t*>(&A[a_gl_rd]);
      #pragma unroll
      for (int i = 0; i < 8; i++) {
        int4 chunk;
        if constexpr (use_bfloat16) {
          #if !defined(__CUDA_ARCH__) || (__CUDA_ARCH__ >= 800)
          nv_bfloat162* a0 = reinterpret_cast<nv_bfloat162*>(&sh_code0[8 * enc[2 * i + 0] + lane]);
          nv_bfloat162* a1 = reinterpret_cast<nv_bfloat162*>(&sh_code1[8 * enc[2 * i + 1] + lane]);
          #pragma unroll
          for (int j = 0; j < 4; j++)
            reinterpret_cast<nv_bfloat162*>(&chunk)[j] = __hadd2(a0[j], a1[j]);
          #endif
        } else {
          half2* a0 = reinterpret_cast<half2*>(&sh_code0[8 * enc[2 * i + 0] + lane]);
          half2* a1 = reinterpret_cast<half2*>(&sh_code1[8 * enc[2 * i + 1] + lane]);
          #pragma unroll
          for (int j = 0; j < 4; j++)
            reinterpret_cast<half2*>(&chunk)[j] = __hadd2(a0[j], a1[j]);
        }
        C[a_gl_rd * 8 + i] = chunk;
      }
    }
    a_gl_rd += 32;
  }
}


__global__ void Code1x4Dequant(
  const uint8_t* __restrict__ codes,
        __half*    __restrict__ output,
  const __half* __restrict__ codebook,
  int prob_m,
  int prob_k  // = code_len * 4
) {

  extern __shared__ __half2 sh_cb[];

  const __half2* codebook2 = reinterpret_cast<const __half2*>(codebook);
  int total_pairs = 256 * 2;
  for (int idx = threadIdx.x; idx < total_pairs; idx += blockDim.x) {
      sh_cb[idx] = codebook2[idx];
  }
  __syncthreads();


  const int WARP = 32;
  int warp_id = (blockIdx.x * (blockDim.x / WARP)) + (threadIdx.x / WARP);
  if (warp_id >= prob_m) return;

  int lane = threadIdx.x & (WARP - 1);
  // codes: [prob_m × (prob_k/4)] uint8_t
  const uint8_t* row_codes = codes + warp_id * (prob_k / 4);
  // output: [prob_m × prob_k] __half → [prob_m × prob_k/2] __half2
  __half2* row_out2 = reinterpret_cast<__half2*>(
                          output + warp_id * prob_k);

  int codes_per_row = prob_k / 4;

  for (int i = lane; i < codes_per_row; i += WARP) {
      uint8_t c = row_codes[i];
      int base = c * 2; 
      row_out2[2*i + 0] = sh_cb[base + 0];
      row_out2[2*i + 1] = sh_cb[base + 1];
  }
}


}

#endif //KERNELS_AQLM_HPP

