#!/usr/bin/env bash

# Bash settings
set -o errexit    # abort on nonzero exitstatus
set -o pipefail   # don't hide errors within pipes
set -o nounset    # abort on unbound variable

# Variables
readonly script_name=$(basename "${0}")                                   # Name of the script
readonly script_dir=$( cd "$( dirname "${BASH_SOURCE[0]}" )" && pwd )     # Directory of the script
IFS=$'\t\n'                                                               # Split on newlines and tabs (but not on spaces)
NCPU="${1:-4}"                                                            # Number of CPUs used by Julia

# Description of the script
echo -n "Running the script ${script_name} in ${script_dir}."
printf "\n"
echo -n "Each Julia job will use ${NCPU} CPUs."
printf "\n\n"

##
##  Benchmark EB-TCϵ and other algorithms
##

# Random Gaussian Instances: K ∈ (5,10,20) for ϵ ∈ (0.05, 0.1)
for eps in 0.05 0.1
do
  for K in 5 10 20
  do
    echo -n "Running: julia -O3 -p${NCPU} epstt_randoms.jl --expe 'epsFull' --inst 'uniform' --K ${K} --eps ${eps} --reps 0.25 --opt 'add' --seed 42 --Nruns 1000 --history 'partial' --freqHist 50"
    printf "\n\n"
    julia -O3 -p"${NCPU}" epstt_randoms.jl --expe "epsFull" --inst "uniform" --K "${K}" --eps "${eps}" --reps 0.25 --opt "add" --seed 42 --Nruns 1000 --history "partial" --freqHist 50
    printf "\n\n"
  done
done

# Alpha 0.3 benchmark
for K in 10 50 100 250 500 750 1000
do
  echo -n "Running: julia -O3 -p${NCPU} epstt_experiments.jl --expe 'epsFast' --inst 'alpha3' --K ${K} --eps 0.1 --opt 'add' --seed 42 --Nruns 100 --history 'partial' --freqHist 50"
  printf "\n\n"
  julia -O3 -p"${NCPU}" epstt_experiments.jl --expe "epsFast" --inst "alpha3" --K "${K}" --eps 0.1 --opt "add" --seed 42 --Nruns 100 --history "partial" --freqHist 50
  printf "\n\n"
done

# Alpha 0.6 benchmark
for K in 10 50 100 250 500
do
  echo -n "Running: julia -O3 -p${NCPU} epstt_experiments.jl --expe 'epsFast' --inst 'alpha6' --K ${K} --eps 0.1 --opt 'add' --seed 42 --Nruns 100 --history 'partial' --freqHist 50"
  printf "\n\n"
  julia -O3 -p"${NCPU}" epstt_experiments.jl --expe "epsFast" --inst "alpha6" --K "${K}" --eps 0.1 --opt "add" --seed 42 --Nruns 100 --history "partial" --freqHist 50
  printf "\n\n"
done

# Sparse benchmark
for K in 10 50 100 250 500 750 1000
do
  echo -n "Running: julia -O3 -p${NCPU} epstt_experiments.jl --expe 'epsFast' --inst 'sparse' --K ${K} --eps 0.1 --opt 'add' --seed 42 --Nruns 100 --history 'partial' --freqHist 50"
  printf "\n\n"
  julia -O3 -p"${NCPU}" epstt_experiments.jl --expe "epsFast" --inst "sparse" --K "${K}" --eps 0.1 --opt "add" --seed 42 --Nruns 100 --history "partial" --freqHist 50
  printf "\n\n"
done

# GK21 Gaussian Instance 1
echo -n "Running: julia -O3 -p${NCPU} epstt_experiments.jl --expe 'epsFull' --inst 'GK21_1' --K 5 --eps 0.1 --opt 'add' --seed 42 --Nruns 1000 --history 'partial' --freqHist 50"
printf "\n\n"
julia -O3 -p"${NCPU}" epstt_experiments.jl --expe "epsFull" --inst "GK21_1" --K 5 --eps 0.1 --opt "add" --seed 42 --Nruns 1000 --history "partial" --freqHist 50
printf "\n\n"

# GK21 Gaussian Instance 2
echo -n "Running: julia -O3 -p${NCPU} epstt_experiments.jl --expe 'epsFull' --inst 'GK21_2' --K 6 --eps 0.15 --opt 'add' --seed 42 --Nruns 1000 --history 'partial' --freqHist 50"
printf "\n\n"
julia -O3 -p"${NCPU}" epstt_experiments.jl --expe "epsFull" --inst "GK21_2" --K 6 --eps 0.15 --opt "add" --seed 42 --Nruns 1000 --history "partial" --freqHist 50
printf "\n\n"

# GK21 Gaussian Instance 3
echo -n "Running: julia -O3 -p${NCPU} epstt_experiments.jl --expe 'epsFull' --inst 'GK21_3' --K 6 --eps 0.1 --opt 'add' --seed 42 --Nruns 1000 --history 'partial' --freqHist 50"
printf "\n\n"
julia -O3 -p"${NCPU}" epstt_experiments.jl --expe "epsFull" --inst "GK21_3" --K 6 --eps 0.1 --opt "add" --seed 42 --Nruns 1000 --history "partial" --freqHist 50
printf "\n\n"

# 2G varying number good arm
for Ie in 1 2 3 4 5 6 7 8
do
  echo -n "Running: julia -O3 -p${NCPU} epstt_experiments.jl --expe 'eps2G' --inst '2G' --K 10 --eps 0.1 --Ie ${Ie} --opt 'add' --seed 42 --Nruns 1000 --history 'partial' --freqHist 50"
  printf "\n\n"
  julia -O3 -p"${NCPU}" epstt_experiments.jl --expe "eps2G" --inst "2G" --K 10 --eps 0.1 --Ie "${Ie}" --opt "add" --seed 42 --Nruns 1000 --history "partial" --freqHist 50
  printf "\n\n"
done


##
## Detailed Simple Regret, benchmark DSR and DSH
##

# GK21 Gaussian Instance 1
echo -n "Running: julia -O3 -p${NCPU} epstt_experiments.jl --expe 'epsPoE' --inst 'GK21_1' --K 5 --eps 0.1 --opt 'add' --seed 42 --Nruns 10000 --history 'partial' --freqHist 4 --Tmax 500"
printf "\n\n"
julia -O3 -p"${NCPU}" epstt_experiments.jl --expe "epsPoE" --inst "GK21_1" --K 5 --eps 0.1 --opt "add" --seed 42 --Nruns 10000 --history "partial" --freqHist 4 --Tmax 500
printf "\n\n"

# GK21 Gaussian Instance 2
echo -n "Running: julia -O3 -p${NCPU} epstt_experiments.jl --expe 'epsPoE' --inst 'GK21_2' --K 6 --eps 0.15 --opt 'add' --seed 42 --Nruns 10000 --history 'partial' --freqHist 4 --Tmax 1000"
printf "\n\n"
julia -O3 -p"${NCPU}" epstt_experiments.jl --expe "epsPoE" --inst "GK21_2" --K 6 --eps 0.15 --opt "add" --seed 42 --Nruns 10000 --history "partial" --freqHist 4 --Tmax 1000
printf "\n\n"

# GK21 Gaussian Instance 3
echo -n "Running: julia -O3 -p${NCPU} epstt_experiments.jl --expe 'epsPoE' --inst 'GK21_3' --K 6 --eps 0.1 --opt 'add' --seed 42 --Nruns 10000 --history 'partial' --freqHist 4 --Tmax 2000"
printf "\n\n"
julia -O3 -p"${NCPU}" epstt_experiments.jl --expe "epsPoE" --inst "GK21_3" --K 6 --eps 0.1 --opt "add" --seed 42 --Nruns 10000 --history "partial" --freqHist 4 --Tmax 2000
printf "\n\n"

# 2G varying number good arm
for Ie in 1 2 3 4 5 6
do
  echo -n "Running: julia -O3 -p${NCPU} epstt_experiments.jl --expe 'epsPoE' --inst '2G' --K 10 --eps 0.1 --Ie ${Ie} --opt 'add' --seed 42 --Nruns 10000 --history 'partial' --freqHist 4 --Tmax 1000"
  printf "\n\n"
  julia -O3 -p"${NCPU}" epstt_experiments.jl --expe "epsPoE" --inst "2G" --K 10 --eps 0.1 --Ie "${Ie}" --opt "add" --seed 42 --Nruns 10000 --history "partial" --freqHist 4 --Tmax 1000
  printf "\n\n"
done

##
## Benchmarking BAI problems with EB-TCϵ and EB-TCϵ with varying slack
##

# Random Gaussian Instances: K ∈ (5, 10, 20)
for K in 5 10 20
do
  echo -n "Running: julia -O3 -p${NCPU} epstt_randoms.jl --expe 'epsVSBench' --inst 'BAIuniform' --K ${K} --eps 0.0 --reps 0.001 --opt 'add' --seed 42 --Nruns 1000 --history 'partial' --freqHist 50"
  printf "\n\n"
  julia -O3 -p"${NCPU}" epstt_randoms.jl --expe "epsVSBench" --inst "BAIuniform" --K "${K}" --eps 0.0 --reps 0.001 --opt "add" --seed 42 --Nruns 1000 --history "partial" --freqHist 50
  printf "\n\n"
done

# BAI GK21 Gaussian Instance 1
echo -n "Running: julia -O3 -p${NCPU} epstt_experiments.jl --expe 'epsVSBench' --inst 'GK21_1' --K 5 --eps 0.0 --opt 'add' --seed 42 --Nruns 1000 --history 'partial' --freqHist 50"
printf "\n\n"
julia -O3 -p"${NCPU}" epstt_experiments.jl --expe "epsVSBench" --inst "GK21_1" --K 5 --eps 0.0 --opt "add" --seed 42 --Nruns 1000 --history "partial" --freqHist 50
printf "\n\n"

# BAI GK21 Gaussian Instance 2
echo -n "Running: julia -O3 -p${NCPU} epstt_experiments.jl --expe 'epsVSBench' --inst 'GK21_2' --K 6 --eps 0.0 --opt 'add' --seed 42 --Nruns 1000 --history 'partial' --freqHist 50"
printf "\n\n"
julia -O3 -p"${NCPU}" epstt_experiments.jl --expe "epsVSBench" --inst "GK21_2" --K 6 --eps 0.0 --opt "add" --seed 42 --Nruns 1000 --history "partial" --freqHist 50
printf "\n\n"
