##### based on torchtune's full_ft_distributed recipe for llama2

# Copyright (c) Meta Platforms, Inc. and affiliates.
# All rights reserved.
#
# This source code is licensed under the BSD-style license found in the
# LICENSE file in the root directory of this source tree.

DISABLE_WANDB = False

import types
import math
import sys
import time
import gc
import wandb

from functools import partial
import torch.nn.functional as F
from typing import Any, Dict, List, Optional, Union, Callable, Tuple
import torch
from omegaconf import DictConfig

from torch import nn
from torch.distributed import (
    destroy_process_group,
    init_device_mesh,
    init_process_group,
)
from torch.distributed._tensor import DTensor
from torch.distributed.tensor.parallel import parallelize_module
from torch.optim import Optimizer
from torchtune import config, modules, training, utils
from torchtune.modules.attention_utils import _MaskType
from torchtune.recipe_interfaces import FTRecipeInterface
from torchtune.training import DummyProfiler, PROFILER_KEY
from torchtune.training.activations import apply_selective_activation_checkpointing
from torchtune.training.checkpointing._checkpoint_client import (
    CheckpointClient,
    TrainingProgress,
)
from torchtune.training.lr_schedulers import get_lr
from torch.nn.attention.flex_attention import flex_attention, BlockMask, create_block_mask
import torch._dynamo

from torch.utils.data import Dataset
from torchtune.modules.tokenizers import ModelTokenizer
from datasets import load_dataset
from torchtune.modules.kv_cache import KVCache



# Constants
MAX_SEQ_LEN = 64*1024
VAL_TOKENS = 4_000_000
VAL_EVERY = 25
_10M = 10_000_000
_50M = 50_000_000

# Configure torch._dynamo
torch._dynamo.config.suppress_errors = True

# Compile flex_attention w/ max-autotune
flex_attention_compiled = torch.compile(flex_attention, dynamic=False, mode='max-autotune')
create_block_mask_compiled = torch.compile(create_block_mask, dynamic=False)

@torch.compiler.disable(recursive=False)
def compile_friendly_flex_attention(
    q: torch.Tensor,
    k: torch.Tensor,
    v: torch.Tensor,
    block_mask: BlockMask,
    score_mod: Callable,
) -> torch.Tensor:
    return flex_attention_compiled(q, k, v,
    block_mask=block_mask,
    score_mod=score_mod)

def scale_free_score_mod(score, b, h, q_idx, kv_idx):
    ix = q_idx - kv_idx
    lttau = (ix/10.).log1p()
    a_t = (2*lttau + 1).sqrt()
    m_t = -2*lttau
    return a_t * score + m_t

def alibi_score_mod(score, b, h, q_idx, kv_idx):
    scale = (-((h + 1) * 8.0 / 32)).exp2() # hardcode nheads as 32 for llama2
    bias = (kv_idx - q_idx) * scale
    return score + bias

log = utils.get_logger("DEBUG")
import logging
logger = logging.getLogger(__name__)

"""nousyarn"""
"""use https://huggingface.co/NousResearch/Yarn-Llama-2-13b-128k/blob/main/modeling_llama_together_yarn.py"""
from einops import rearrange, repeat

def rotate_half(x, interleaved=False):
    if not interleaved:
        x1, x2 = x.chunk(2, dim=-1)
        return torch.cat((-x2, x1), dim=-1)
    else:
        x1, x2 = x[..., ::2], x[..., 1::2]
        return rearrange(torch.stack((-x2, x1), dim=-1), "... d two -> ... (d two)", two=2)


def apply_rotary_emb_torch(x, cos, sin, interleaved=False):
    """
    x: (batch_size, seqlen, nheads, headdim)
    cos, sin: (seqlen, rotary_dim / 2) or (batch_size, seqlen, rotary_dim / 2)
    """
    ro_dim = cos.shape[-1] * 2
    assert ro_dim <= x.shape[-1]
    cos = repeat(cos, "... d -> ... 1 (2 d)" if not interleaved else "... d -> ... 1 (d 2)")
    sin = repeat(sin, "... d -> ... 1 (2 d)" if not interleaved else "... d -> ... 1 (d 2)")
    return torch.cat(
        [x[..., :ro_dim] * cos + rotate_half(x[..., :ro_dim], interleaved) * sin, x[..., ro_dim:]],
        dim=-1,
    )


# Inverse dim formula to find dim based on number of rotations
def _yarn_find_correction_dim(num_rotations, dim, base=10_000, max_position_embeddings=4*1024):
    return (dim * math.log(max_position_embeddings/(num_rotations * 2 * math.pi)))/(2 * math.log(base))

# Find dim range bounds based on rotations
def _yarn_find_correction_range(low_rot, high_rot, dim, base=10_000, max_position_embeddings=4*1024):
    low = math.floor(_yarn_find_correction_dim(
        low_rot, dim, base, max_position_embeddings))
    high = math.ceil(_yarn_find_correction_dim(
        high_rot, dim, base, max_position_embeddings))
    return max(low, 0), min(high, dim-1)  # Clamp values just in case

def _yarn_linear_ramp_mask(min, max, dim):
    if min == max:
        max += 0.001  # Prevent singularity

    linear_func = (torch.arange(dim, dtype=torch.float32) - min) / (max - min)
    ramp_func = torch.clamp(linear_func, 0, 1)
    return ramp_func

def _yarn_get_mscale(scale=1):
    if scale <= 1:
        return 1.0
    return 0.1 * math.log(scale) + 1.0

class NousYaRN(torch.nn.Module):
    """
    The rotary position embeddings from RoFormer_ (Su et. al).
    A crucial insight from the method is that the query and keys are
    transformed by rotation matrices which depend on the relative positions.
    Other implementations are available in the Rotary Transformer repo_ and in
    GPT-NeoX_, GPT-NeoX was an inspiration
    .. _RoFormer: https://arxiv.org/abs/2104.09864
    .. _repo: https://github.com/ZhuiyiTechnology/roformer
    .. _GPT-NeoX: https://github.com/EleutherAI/gpt-neox
    This implements the YaRN extension method.
    """

    def __init__(self, dim: int, base=10_000, interleaved=False,
                 scaling_factor=1.0, pos_idx_in_fp32=True,
                 max_position_embeddings=64*1024,
                 original_max_position_embeddings=4*1024, extrapolation_factor=1,
                 attn_factor=1, beta_fast=32, beta_slow=1,
                 dynamic=True, finetuned=False, device=None):
        """
            interleaved: if True, rotate pairs of even and odd dimensions (GPT-J style) instead
                of 1st half and 2nd half (GPT-NeoX style).
            pos_idx_in_fp32: if True, the position indices [0.0, ..., seqlen - 1] are in fp32,
                otherwise they might be in lower precision.
                This option was added because previously (before 2023-07-02), when we construct
                the position indices, we use the dtype of self.inv_freq. In most cases this would
                be fp32, but if the model is trained in pure bf16 (not mixed precision), then
                self.inv_freq would be bf16, and the position indices are also in bf16.
                Because of the limited precision of bf16 (e.g. 1995.0 is rounded to 2000.0), the
                embeddings for some positions will coincide.
                To maintain compatibility with models previously trained in pure bf16,
                we add this option.
            scaling_factor: RotaryEmbedding extended with YaRN scaling.
        """
        super().__init__()

        self.dim = dim
        self.base = float(base)
        self.interleaved = interleaved
        self.scaling_factor = scaling_factor
        self.max_position_embeddings = max_position_embeddings
        self.original_max_position_embeddings = original_max_position_embeddings if original_max_position_embeddings else max_position_embeddings
        self.extrapolation_factor = extrapolation_factor
        self.attn_factor = attn_factor
        self.beta_fast = beta_fast
        self.beta_slow = beta_slow
        self.pos_idx_in_fp32 = pos_idx_in_fp32
        self.mscale = float(_yarn_get_mscale(self.scaling_factor) * attn_factor) # Get n-d magnitude scaling corrected for interpolation
        self.dynamic = dynamic
        self.finetuned = finetuned

        # Generate and save the inverse frequency buffer (non trainable)
        if not dynamic:
            self._compute_inv_freq(scaling_factor, device)

        self._seq_len_cached = 0
        self._cos_cached = None
        self._sin_cached = None

    def _compute_inv_freq(self, scaling_factor, device=None):
        pos_freqs = self.base ** (torch.arange(0, self.dim, 2).float().to(device) / self.dim)
        inv_freq_extrapolation = 1.0 / pos_freqs
        inv_freq_interpolation = 1.0 / (scaling_factor * pos_freqs)

        low, high = _yarn_find_correction_range(self.beta_fast, self.beta_slow, self.dim, self.base, self.original_max_position_embeddings)
        inv_freq_mask = (1 - _yarn_linear_ramp_mask(low, high, self.dim // 2).float().to(device)) * self.extrapolation_factor # Get n-d rotational scaling corrected for extrapolation
        inv_freq = inv_freq_interpolation * (1 - inv_freq_mask) + inv_freq_extrapolation * inv_freq_mask
        self.register_buffer("inv_freq", inv_freq, persistent=False)

    def _compute_inv_freq_original(self, device=None):
        inv_freq = 1 / (self.base ** (torch.arange(0, self.dim, 2, device=device,
                                                 dtype=torch.float32) / self.dim))
        self.register_buffer("inv_freq", inv_freq, persistent=False)

    def _update_cos_sin_cache(self, seqlen, device=None, dtype=None):
        # Reset the tables if the sequence length has changed,
        # if we're on a new device (possibly due to tracing for instance),
        # or if we're switching from inference mode to training
        if (seqlen != self._seq_len_cached or self._cos_cached.device != device
            or self._cos_cached.dtype != dtype
            or (self.training and self._cos_cached.is_inference())):
            self._seq_len_cached = seqlen

            if self.dynamic:
                scaling_factor = None
                if seqlen <= self.max_position_embeddings:
                    if self.finetuned:
                        scaling_factor = self.scaling_factor
                else:
                    scaling_factor = seqlen / self.original_max_position_embeddings
                if scaling_factor:
                    self._compute_inv_freq(scaling_factor, device)
                    self.mscale = float(_yarn_get_mscale(scaling_factor) * self.attn_factor)
                else:
                    self._compute_inv_freq_original(device)

            # We want fp32 here, not self.inv_freq.dtype, since the model could be loaded in bf16
            # And the output of arange can be quite large, so bf16 would lose a lot of precision.
            # However, for compatibility reason, we add an option to use the dtype of self.inv_freq.
            if self.pos_idx_in_fp32:
                t = torch.arange(seqlen, device=device, dtype=torch.float32)
                # We want fp32 here as well since inv_freq will be multiplied with t, and the output
                # will be large. Having it in bf16 will lose a lot of precision and cause the
                # cos & sin output to change significantly.
                # We want to recompute self.inv_freq if it was not loaded in fp32
                if self.inv_freq.dtype != torch.float32:
                    inv_freq = self.inv_freq.to(torch.float32)
                else:
                    inv_freq = self.inv_freq
            else:
                t = torch.arange(seqlen, device=device, dtype=self.inv_freq.dtype)
                inv_freq = self.inv_freq
            # Don't do einsum, it converts fp32 to fp16 under AMP
            # freqs = torch.einsum("i,j->ij", t, self.inv_freq)
            freqs = torch.outer(t, inv_freq)
            self._cos_cached = (torch.cos(freqs) * self.mscale).to(dtype)
            self._sin_cached = (torch.sin(freqs) * self.mscale).to(dtype)


    def forward(self, x: torch.Tensor, seqlen_offset: int = 0, **kwargs) -> Tuple[torch.Tensor, torch.Tensor]:
        """
        q: (batch, seqlen, nheads, headdim)
        k: (batch, seqlen, nheads, headdim)
        seqlen_offset: can be used in generation where the qkv being passed in is only the last
        token in the batch.
        """
        assert x.shape[1] in [1024, 4096, 16*1024, 64*1024]
        self._update_cos_sin_cache(x.shape[1] + seqlen_offset, device=x.device, dtype=x.dtype)
        return apply_rotary_emb_torch(x, self._cos_cached[seqlen_offset:], self._sin_cached[seqlen_offset:], True)
    # def forward(self, q: torch.Tensor, k: torch.Tensor, seqlen_offset: int = 0, **kwargs) -> Tuple[torch.Tensor, torch.Tensor]:
    #     """
    #     q: (batch, seqlen, nheads, headdim)
    #     k: (batch, seqlen, nheads, headdim)
    #     seqlen_offset: can be used in generation where the qkv being passed in is only the last
    #     token in the batch.
    #     """
    #     assert q.shape[1] in [1024, 4096, 16*1024, 64*1024]
    #     self._update_cos_sin_cache(q.shape[1] + seqlen_offset, device=q.device, dtype=q.dtype)
    #     return apply_rotary_emb_torch(
    #         q, self._cos_cached[seqlen_offset:], self._sin_cached[seqlen_offset:], True
    #     ), apply_rotary_emb_torch(
    #         k, self._cos_cached[seqlen_offset:], self._sin_cached[seqlen_offset:], True
    #     )
        # return apply_rotary_emb_func(
        #     q, self._cos_cached[seqlen_offset:], self._sin_cached[seqlen_offset:],
        #     self.interleaved, True # inplace=True
        # ), apply_rotary_emb_func(
        #     k, self._cos_cached[seqlen_offset:], self._sin_cached[seqlen_offset:],
        #     self.interleaved, True # inplace=True
        # )
"""--------------------------------------------------------------------------"""
"""nousyarn end"""


class RotaryPositionalEmbeddings(nn.Module):
    """
    This class implements Rotary Positional Embeddings (RoPE)
    proposed in https://arxiv.org/abs/2104.09864.

    Reference implementation (used for correctness verfication)
    can be found here:
    https://github.com/meta-llama/llama/blob/main/llama/model.py#L80

    In this implementation we cache the embeddings for each position upto
    ``max_seq_len`` by computing this during init.

    Args:
        dim (int): Embedding dimension. This is usually set to the dim of each
            head in the attention module computed as ``embed_dim // num_heads``
        max_seq_len (int): Maximum expected sequence length for the
            model, if exceeded the cached freqs will be recomputed
        base (int): The base for the geometric progression used to compute
            the rotation angles
    """

    def __init__(
        self,
        dim: int,
        max_seq_len: int = 64*1024,
        base: int = 10_000,
    ) -> None:
        super().__init__()
        self.dim = dim
        self.base = base
        self.max_seq_len = max_seq_len
        self.rope_init()

    def rope_init(self):
        theta = 1.0 / (
            self.base
            ** (torch.arange(0, self.dim, 2)[: (self.dim // 2)].float() / self.dim)
        )
        self.register_buffer("theta", theta, persistent=False)
        self.build_rope_cache(self.max_seq_len)

    def build_rope_cache(self, max_seq_len: int = MAX_SEQ_LEN) -> None:
        # Create position indexes `[0, 1, ..., max_seq_len - 1]`
        seq_idx = torch.arange(
            max_seq_len, dtype=self.theta.dtype, device=self.theta.device
        )

        # Outer product of theta and position index; output tensor has
        # a shape of [max_seq_len, dim // 2]
        idx_theta = torch.einsum("i, j -> ij", seq_idx, self.theta).float()

        # cache includes both the cos and sin components and so the output shape is
        # [max_seq_len, dim // 2, 2]
        cache = torch.stack([torch.cos(idx_theta), torch.sin(idx_theta)], dim=-1)
        self.register_buffer("cache", cache, persistent=False)

    def forward(
        self, x: torch.Tensor, *, input_pos: Optional[torch.Tensor] = None
    ) -> torch.Tensor:
        """
        Args:
            x (torch.Tensor): input tensor with shape
                ``[b, s, n_h, h_d]``
            input_pos (Optional[torch.Tensor]): Optional tensor which contains the position ids
                of each token. During training, this is used to indicate the positions
                of each token relative to its sample when packed, shape [b, s].
                During inference, this indicates the position of the current token.
                If none, assume the index of the token is its position id. Default is None.

        Returns:
            torch.Tensor: output tensor with shape ``[b, s, n_h, h_d]``

        Notation used for tensor shapes:
            - b: batch size
            - s: sequence length
            - n_h: num heads
            - h_d: head dim
        """
        # input tensor has shape [b, s, n_h, h_d]
        seq_len = x.size(1)

        # extract the values based on whether input_pos is set or not
        rope_cache = (
            self.cache[:seq_len] if input_pos is None else self.cache[input_pos]
        )

        # reshape input; the last dimension is used for computing the output.
        # Cast to float to match the reference implementation
        # tensor has shape [b, s, n_h, h_d // 2, 2]
        xshaped = x.float().reshape(*x.shape[:-1], -1, 2)

        # reshape the cache for broadcasting
        # tensor has shape [b, s, 1, h_d // 2, 2] if packed samples,
        # otherwise has shape [1, s, 1, h_d // 2, 2]
        rope_cache = rope_cache.view(-1, xshaped.size(1), 1, xshaped.size(3), 2)

        # tensor has shape [b, s, n_h, h_d // 2, 2]
        x_out = torch.stack(
            [
                xshaped[..., 0] * rope_cache[..., 0]
                - xshaped[..., 1] * rope_cache[..., 1],
                xshaped[..., 1] * rope_cache[..., 0]
                + xshaped[..., 0] * rope_cache[..., 1],
            ],
            -1,
        )

        # tensor has shape [b, s, n_h, h_d]
        x_out = x_out.flatten(3)
        return x_out.type_as(x)

class NoPE(nn.Module):
    def __init__(
        self,
        *args,
        **kwargs,
    ) -> None:
        super().__init__()

    def forward(
        self, x: torch.Tensor, *, input_pos: Optional[torch.Tensor] = None
    ) -> torch.Tensor: # identity
        return x

class pRotaryPositionalEmbeddings(nn.Module):
    def __init__(
        self,
        dim: int,
        max_seq_len: int = 64*1024,
        base: int = 1024,
    ) -> None:
        super().__init__()
        self.dim = dim
        self.base = base
        self.max_seq_len = max_seq_len
        self.rope_init()

    def rope_init(self):
        theta = (1 / self.base) ** torch.linspace(0, 1, steps=self.dim//4, dtype=torch.float32)
        theta = torch.cat([theta, theta.new_zeros(self.dim//4)])
        self.register_buffer("theta", theta, persistent=False)
        self.build_rope_cache(self.max_seq_len)

    def build_rope_cache(self, max_seq_len: int = MAX_SEQ_LEN) -> None:
        # Create position indexes `[0, 1, ..., max_seq_len - 1]`
        seq_idx = torch.arange(
            max_seq_len, dtype=self.theta.dtype, device=self.theta.device
        )

        # Outer product of theta and position index; output tensor has
        # a shape of [max_seq_len, dim // 2]
        idx_theta = torch.einsum("i, j -> ij", seq_idx, self.theta).float()

        # cache includes both the cos and sin components and so the output shape is
        # [max_seq_len, dim // 2, 2]
        cache = torch.stack([torch.cos(idx_theta), torch.sin(idx_theta)], dim=-1)
        self.register_buffer("cache", cache, persistent=False)

    def forward(
        self, x: torch.Tensor, *, input_pos: Optional[torch.Tensor] = None
    ) -> torch.Tensor:
        # input tensor has shape [b, s, n_h, h_d]
        seq_len = x.size(1)

        # extract the values based on whether input_pos is set or not
        rope_cache = (
            self.cache[:seq_len] if input_pos is None else self.cache[input_pos]
        )

        # reshape input; the last dimension is used for computing the output.
        # Cast to float to match the reference implementation
        # tensor has shape [b, s, n_h, h_d // 2, 2]
        xshaped = x.float().reshape(*x.shape[:-1], -1, 2)

        # reshape the cache for broadcasting
        # tensor has shape [b, s, 1, h_d // 2, 2] if packed samples,
        # otherwise has shape [1, s, 1, h_d // 2, 2]
        rope_cache = rope_cache.view(-1, xshaped.size(1), 1, xshaped.size(3), 2)

        # tensor has shape [b, s, n_h, h_d // 2, 2]
        x_out = torch.stack(
            [
                xshaped[..., 0] * rope_cache[..., 0]
                - xshaped[..., 1] * rope_cache[..., 1],
                xshaped[..., 1] * rope_cache[..., 0]
                + xshaped[..., 0] * rope_cache[..., 1],
            ],
            -1,
        )

        # tensor has shape [b, s, n_h, h_d]
        x_out = x_out.flatten(3)
        return x_out.type_as(x)

class NTKAwareRotaryPositionalEmbeddings(nn.Module):
    """
    basically the same as rope, except if the input sequence is longer than the training sequence
    we tweak theta
    """
    def __init__(
        self,
        dim: int,
        max_seq_len: int = 64*1024,
        base: int = 10_000,
    ) -> None:
        super().__init__()
        self.dim = dim
        self.base = base
        self.max_seq_len = max_seq_len
        self.train_seq_len = 4096 # hardcode for llama2
        self.rope_init()

    def rope_init(self):
        theta = 1.0 / (
            self.base
            ** (torch.arange(0, self.dim, 2)[: (self.dim // 2)].float() / self.dim)
        )
        self.register_buffer("theta", theta, persistent=False)
        self.build_rope_cache(self.max_seq_len, seq_len=self.train_seq_len)

    def build_rope_cache(self, max_seq_len: int = MAX_SEQ_LEN, seq_len: int=None) -> None:
        assert seq_len is not None, "seq_len must be provided"
        # Create position indexes `[0, 1, ..., max_seq_len - 1]`
        seq_idx = torch.arange(
            max_seq_len, dtype=self.theta.dtype, device=self.theta.device
        )

        if seq_len > self.train_seq_len:
            s = seq_len / self.train_seq_len
            scaled_base = self.base * (s ** (self.dim / (self.dim - 2)))
            theta = 1.0 / (
                scaled_base
                ** (torch.arange(0, self.dim, 2)[: (self.dim // 2)].float() / self.dim)
            ).to(self.theta.device, dtype=self.theta.dtype)
        else:
            theta = self.theta

        # Outer product of theta and position index; output tensor has
        # a shape of [max_seq_len, dim // 2]
        idx_theta = torch.einsum("i, j -> ij", seq_idx, theta).float()

        # cache includes both the cos and sin components and so the output shape is
        # [max_seq_len, dim // 2, 2]
        cache = torch.stack([torch.cos(idx_theta), torch.sin(idx_theta)], dim=-1)
        self.register_buffer("cache", cache, persistent=False)

    def forward(
        self, x: torch.Tensor, *, input_pos: Optional[torch.Tensor] = None
    ) -> torch.Tensor:
        seq_len = x.size(1)
        if self.cache is None or seq_len != self.cache.size(0):
            ## rebuild if sequence length changed
            self.build_rope_cache(max_seq_len=self.max_seq_len, seq_len=seq_len)

        # extract the values based on whether input_pos is set or not
        rope_cache = (
            self.cache[:seq_len] if input_pos is None else self.cache[input_pos]
        )

        # reshape input; the last dimension is used for computing the output.
        # Cast to float to match the reference implementation
        # tensor has shape [b, s, n_h, h_d // 2, 2]
        xshaped = x.float().reshape(*x.shape[:-1], -1, 2)

        # reshape the cache for broadcasting
        # tensor has shape [b, s, 1, h_d // 2, 2] if packed samples,
        # otherwise has shape [1, s, 1, h_d // 2, 2]
        rope_cache = rope_cache.view(-1, xshaped.size(1), 1, xshaped.size(3), 2)

        # tensor has shape [b, s, n_h, h_d // 2, 2]
        x_out = torch.stack(
            [
                xshaped[..., 0] * rope_cache[..., 0]
                - xshaped[..., 1] * rope_cache[..., 1],
                xshaped[..., 1] * rope_cache[..., 0]
                + xshaped[..., 0] * rope_cache[..., 1],
            ],
            -1,
        )

        # tensor has shape [b, s, n_h, h_d]
        x_out = x_out.flatten(3)
        return x_out.type_as(x)


class RMSNorm(nn.Module):
    def forward(self, x): return F.rms_norm(x, (x.size(-1),))
class LogNAttn(nn.Module):
    def __init__(self, num_heads, special_s=False):
        super().__init__()
        self.logn_s = torch.nn.Parameter(torch.ones(num_heads))
        if special_s: # try to initialize s more sensibly. see https://arxiv.org/pdf/2501.19399
            # try to make the scalar s = tr_len / sum(log(n) for n=1,...,tr_len)
            # this is equivalent multiplying logn_list by a number
            # >>> import math
            # >>> 4096/sum(math.log(n+1) for n in range(4096))
            # 0.13663057759265326
            logn_list = [math.log(i+1, math.e)*0.137 for i in range(MAX_SEQ_LEN)] # (T,)
        else:
            logn_list = [math.log(i+1, math.e) for i in range(MAX_SEQ_LEN)] # (T,)
        self.register_buffer("logn", torch.tensor(logn_list))
    def state_dict(self, *args, **kwargs):
        return {
            'logn_s': self.logn_s,
            'logn': self.logn
        }
    def load_state_dict(self, state_dict, *args, **kwargs):
        self.logn_s.data.copy_(state_dict['logn_s'])
        self.logn.copy_(state_dict['logn'])
    def forward(self, q, s_x):
        ## update q by multiplying by logn factor
        # q shape: [b, n_h, s_x, h_d]
        # logn shape: s_x
        # logn_s shape: n_h
        q = self.logn[None,None,:s_x,None] * q * self.logn_s[None,:,None,None]
        return q

def logn_attn_forward(
    self,
    x: torch.Tensor,
    y: Optional[torch.Tensor] = None,
    *,
    mask: Optional[_MaskType] = None,
    input_pos: Optional[torch.Tensor] = None,
) -> torch.Tensor:
    """
    Args:
        x (torch.Tensor): input tensor with shape [b x s_x x d] for the query
        y (Optional[torch.Tensor]): second input tensor with shape [b x s_y x d], is the input
            for k and v. For self attention, x=y. Optional only with kv_cache enabled.
        mask (Optional[_MaskType]): Used to mask the scores after the query-key multiplication
            and before the softmax. Either:

            A boolean tensor with shape ``[b x s x s]``, ``[b x s x self.encoder_max_cache_seq_len]``,
            or ``[b x s x self.decoder_max_cache_seq_len]`` if using KV-cacheing with encoder/decoder layers.
            A value of True in row ``i`` and column ``j`` means token ``i`` attends to token ``j``. A value of False means
            token ``i`` does not attend to token ``j``. If no mask is specified, a causal mask
            is used by default.

            A :class:`~torch.nn.attention.flex_attention.BlockMask` for document masking in a packed sequence
            created via `create_block_mask <https://pytorch.org/blog/flexattention/#mask-mods>`_. We  use
            :func:`~torch.nn.attention.flex_attention.flex_attention` when computing attention with block masks.
            Default is None.
        input_pos (Optional[torch.Tensor]): Optional tensor which contains the position ids
            of each token. During training, this is used to indicate the positions
            of each token relative to its sample when packed, shape [b x s].
            During inference, this indicates the position of the current token.
            If none, assume the index of the token is its position id. Default is None.

    Raises:
        ValueError: If no ``y`` input and ``kv_cache`` is not enabled.

    Returns:
        torch.Tensor: output tensor with attention applied

    Notation used for tensor shapes:
        - b: batch size
        - s_x: sequence length for x
        - s_y: sequence length for y
        - n_h: num heads
        - n_kv: num kv heads
        - d: embed dim
        - h_d: head dim
    """
    # x has shape [b, s_x, d]
    # y has shape [b, s_y, d]
    b, s_x, _ = x.shape
    s_y = y.shape[1] if y is not None else 0

    # q has shape [b, s_x, num_heads * head_dim]
    q = self.q_proj(x)

    # number of queries per key/value
    q_per_kv = self.num_heads // self.num_kv_heads
    q = q.view(b, s_x, self.num_kv_heads * q_per_kv, self.head_dim)

    # Apply positional embeddings
    if self.pos_embeddings is not None:
        q = self.pos_embeddings(q, input_pos=input_pos)

    # [b, n_h, s_x, h_d]
    q = q.transpose(1, 2)

    # Normalize q
    if self.q_norm is not None:
        q = self.q_norm(q)

    if y is None:
        if self.kv_cache is None or not self.cache_enabled:
            raise ValueError(
                "Must provide y input or use kv_cache to enable streaming decoding"
            )
        k = self.kv_cache.k_cache
        v = self.kv_cache.v_cache
    else:
        # Update k and v shape, positional embeddings, and normalization

        # k,v shape [b, s_y, num_kv_heads * head_dim]
        k = self.k_proj(y)
        v = self.v_proj(y)

        # Apply positional embeddings
        # k,v shape: [b, s_y, n_kv, h_d]
        k = k.view(b, s_y, -1, self.head_dim)
        v = v.view(b, s_y, -1, self.head_dim)
        if self.pos_embeddings is not None:
            k = self.pos_embeddings(k, input_pos=input_pos)

        # k,v shape: [b, n_kv, s_y, h_d]
        k = k.transpose(1, 2)
        v = v.transpose(1, 2)

        # Normalize k
        if self.k_norm is not None:
            k = self.k_norm(k)

        # Update key-value cache
        if self.kv_cache is not None and self.cache_enabled:
            k, v = self.kv_cache.update(k, v)

    # If needed, expand the key and value tensors to have the same shape
    # as the query tensor by copying values across the relevant dim
    # k,v shape: [b, n_kv, s, h_d] -> [b, n_h, s, h_d]
    if self.num_heads != self.num_kv_heads:
        expand_shape = (b, self.num_kv_heads, q_per_kv, -1, self.head_dim)
        k = k.unsqueeze(2).expand(expand_shape).flatten(1, 2)
        v = v.unsqueeze(2).expand(expand_shape).flatten(1, 2)

    q = self.logn_attn_module(q, s_x)
    output = self._attention_call(
        q,
        k,
        v,
        mask=mask,
        dropout_p=self.attn_dropout if self.training else 0.0,
        is_causal=self.kv_cache is None and mask is None and self.is_causal,
    )
    # reshape the output to be the same shape as the input
    output = output.transpose(1, 2).contiguous().view(b, s_x, -1)
    return self.output_proj(output)

def alibi_forward(
    self,
    x: torch.Tensor,
    y: Optional[torch.Tensor] = None,
    *,
    mask: Optional[_MaskType] = None,
    input_pos: Optional[torch.Tensor] = None,
) -> torch.Tensor:
    # x has shape [b, s_x, d]
    # y has shape [b, s_y, d]
    b, s_x, _ = x.shape
    s_y = y.shape[1] if y is not None else 0

    # q has shape [b, s_x, num_heads * head_dim]
    q = self.q_proj(x)

    # number of queries per key/value
    q_per_kv = self.num_heads // self.num_kv_heads
    q = q.view(b, s_x, self.num_kv_heads * q_per_kv, self.head_dim)

    ## alibi doesn't use rope
    # if self.pos_embeddings is not None:
    #     q = self.pos_embeddings(q, input_pos=input_pos)

    # [b, n_h, s_x, h_d]
    q = q.transpose(1, 2)

    # Normalize q
    if self.q_norm is not None:
        q = self.q_norm(q)

    if y is None:
        if self.kv_cache is None or not self.cache_enabled:
            raise ValueError(
                "Must provide y input or use kv_cache to enable streaming decoding"
            )
        k = self.kv_cache.k_cache
        v = self.kv_cache.v_cache
    else:
        # Update k and v shape, positional embeddings, and normalization

        # k,v shape [b, s_y, num_kv_heads * head_dim]
        k = self.k_proj(y)
        v = self.v_proj(y)

        # Apply positional embeddings
        # k,v shape: [b, s_y, n_kv, h_d]
        k = k.view(b, s_y, -1, self.head_dim)
        v = v.view(b, s_y, -1, self.head_dim)

        ## alibi doesn't use roe
        # if self.pos_embeddings is not None:
        #     k = self.pos_embeddings(k, input_pos=input_pos)

        # k,v shape: [b, n_kv, s_y, h_d]
        k = k.transpose(1, 2)
        v = v.transpose(1, 2)

        # Normalize k
        if self.k_norm is not None:
            k = self.k_norm(k)

        # Update key-value cache
        if self.kv_cache is not None and self.cache_enabled:
            k, v = self.kv_cache.update(k, v)

    # If needed, expand the key and value tensors to have the same shape
    # as the query tensor by copying values across the relevant dim
    # k,v shape: [b, n_kv, s, h_d] -> [b, n_h, s, h_d]
    if self.num_heads != self.num_kv_heads:
        expand_shape = (b, self.num_kv_heads, q_per_kv, -1, self.head_dim)
        k = k.unsqueeze(2).expand(expand_shape).flatten(1, 2)
        v = v.unsqueeze(2).expand(expand_shape).flatten(1, 2)

    output = compile_friendly_flex_attention(
        q,
        k,
        v,
        block_mask=mask,
        score_mod=alibi_score_mod
    )

    # reshape the output to be the same shape as the input
    output = output.transpose(1, 2).contiguous().view(b, s_x, -1).bfloat16()
    return self.output_proj(output)

def scale_invariant_forward(
    self,
    x: torch.Tensor,
    y: Optional[torch.Tensor] = None,
    *,
    mask: Optional[_MaskType] = None,
    input_pos: Optional[torch.Tensor] = None,
) -> torch.Tensor:
    """
    Args:
        x (torch.Tensor): input tensor with shape [b x s_x x d] for the query
        y (Optional[torch.Tensor]): second input tensor with shape [b x s_y x d], is the input
            for k and v. For self attention, x=y. Optional only with kv_cache enabled.
        mask (Optional[_MaskType]): Used to mask the scores after the query-key multiplication
            and before the softmax. Either:

            A boolean tensor with shape ``[b x s x s]``, ``[b x s x self.encoder_max_cache_seq_len]``,
            or ``[b x s x self.decoder_max_cache_seq_len]`` if using KV-cacheing with encoder/decoder layers.
            A value of True in row ``i`` and column ``j`` means token ``i`` attends to token ``j``. A value of False means
            token ``i`` does not attend to token ``j``. If no mask is specified, a causal mask
            is used by default.

            A :class:`~torch.nn.attention.flex_attention.BlockMask` for document masking in a packed sequence
            created via `create_block_mask <https://pytorch.org/blog/flexattention/#mask-mods>`_. We  use
            :func:`~torch.nn.attention.flex_attention.flex_attention` when computing attention with block masks.
            Default is None.
        input_pos (Optional[torch.Tensor]): Optional tensor which contains the position ids
            of each token. During training, this is used to indicate the positions
            of each token relative to its sample when packed, shape [b x s].
            During inference, this indicates the position of the current token.
            If none, assume the index of the token is its position id. Default is None.

    Raises:
        ValueError: If no ``y`` input and ``kv_cache`` is not enabled.

    Returns:
        torch.Tensor: output tensor with attention applied

    Notation used for tensor shapes:
        - b: batch size
        - s_x: sequence length for x
        - s_y: sequence length for y
        - n_h: num heads
        - n_kv: num kv heads
        - d: embed dim
        - h_d: head dim
    """
    # x has shape [b, s_x, d]
    # y has shape [b, s_y, d]
    b, s_x, _ = x.shape
    s_y = y.shape[1] if y is not None else 0

    # q has shape [b, s_x, num_heads * head_dim]
    q = self.q_proj(x)

    # number of queries per key/value
    q_per_kv = self.num_heads // self.num_kv_heads
    q = q.view(b, s_x, self.num_kv_heads * q_per_kv, self.head_dim)

    # Apply positional embeddings
    if self.pos_embeddings is not None:
        q = self.pos_embeddings(q, input_pos=input_pos)

    # [b, n_h, s_x, h_d]
    q = q.transpose(1, 2)

    # Normalize q
    if self.q_norm is not None:
        q = self.q_norm(q)

    if y is None:
        if self.kv_cache is None or not self.cache_enabled:
            raise ValueError(
                "Must provide y input or use kv_cache to enable streaming decoding"
            )
        k = self.kv_cache.k_cache
        v = self.kv_cache.v_cache
    else:
        # Update k and v shape, positional embeddings, and normalization

        # k,v shape [b, s_y, num_kv_heads * head_dim]
        k = self.k_proj(y)
        v = self.v_proj(y)

        # Apply positional embeddings
        # k,v shape: [b, s_y, n_kv, h_d]
        k = k.view(b, s_y, -1, self.head_dim)
        v = v.view(b, s_y, -1, self.head_dim)
        if self.pos_embeddings is not None:
            k = self.pos_embeddings(k, input_pos=input_pos)

        # k,v shape: [b, n_kv, s_y, h_d]
        k = k.transpose(1, 2)
        v = v.transpose(1, 2)

        # Normalize k
        if self.k_norm is not None:
            k = self.k_norm(k)

        # Update key-value cache
        if self.kv_cache is not None and self.cache_enabled:
            k, v = self.kv_cache.update(k, v)

    # If needed, expand the key and value tensors to have the same shape
    # as the query tensor by copying values across the relevant dim
    # k,v shape: [b, n_kv, s, h_d] -> [b, n_h, s, h_d]
    if self.num_heads != self.num_kv_heads:
        expand_shape = (b, self.num_kv_heads, q_per_kv, -1, self.head_dim)
        k = k.unsqueeze(2).expand(expand_shape).flatten(1, 2)
        v = v.unsqueeze(2).expand(expand_shape).flatten(1, 2)

    output = compile_friendly_flex_attention(
        q,
        k,
        v,
        block_mask=mask,
        score_mod=scale_free_score_mod
    )

    # reshape the output to be the same shape as the input
    output = output.transpose(1, 2).contiguous().view(b, s_x, -1).bfloat16()
    return self.output_proj(output)

class FineWebDataset(Dataset):
    def __init__(
        self,
        tokenizer: ModelTokenizer,
        max_seq_len: int = MAX_SEQ_LEN,
        train: bool = True,
        **load_dataset_kwargs
    ):
        self.max_seq_len = max_seq_len
        self._tokenizer = tokenizer
        self._data = load_dataset("parquet",
                                data_files="./fineweb_local/sample/10BT/*.parquet",
                                split="train",
                                streaming=True,
                                )
        self._column = "text"
        self._token_stream = []
        self.n_val = VAL_TOKENS
        self._val = not train
        self._train = train

        self._data_iter = iter(self._data)
        if train: # the first few tokens belong to val -- so skip these!
            token_count = 0
            i = 0
            gen_tokens = self._generate_tokens()
            while token_count < max(self.n_val, _10M):
                i += 1
                sample = next(gen_tokens)
                tokens = sample["tokens"]
                token_count += len(tokens)
                if token_count >= self.n_val:
                    break

    def __len__(self):
        raise NotImplementedError("Not implemented since this is a streamed dataset")

    def __iter__(self):
        return self._generate_tokens()

    def _generate_tokens(self):
        tokens_exhausted = False
        ntokens = 0
        while not tokens_exhausted:
            try:
                sample = next(self._data_iter)
            except StopIteration:
                tokens_exhausted = True
            else:
                tokens = self._tokenizer.encode(text=sample[self._column], add_bos=True, add_eos=True)
                self._token_stream.extend(tokens)
                ntokens += len(tokens)
            if len(self._token_stream) >= self.max_seq_len:
                tokens_to_return = self._token_stream[:self.max_seq_len]; self._token_stream = self._token_stream[self.max_seq_len:]
                yield {"tokens": tokens_to_return, "labels": tokens_to_return.copy()}
                if ntokens >= self.n_val and self._val:
                    tokens_exhausted = True

class StreamDataLoader:
    def __init__(self, dataset, batch_size, max_tokens=None):
        self.dataset = dataset
        self.batch_size = batch_size
        self.max_tokens = max_tokens
        self.iterator = iter(dataset)
        self._len = self.max_tokens // (self.batch_size * self.dataset.max_seq_len)
        self._ix = 0  # Track current position
        print(f"StreamDataLoader initialized with {self._len} batches: max_tokens={self.max_tokens}, batch_size={self.batch_size}, max_seq_len={self.dataset.max_seq_len}")

    def reset(self):
        """Reset the iterator to start from the beginning."""
        self.iterator = iter(self.dataset)
        self._ix = 0

    def __len__(self): return self._len

    def __iter__(self):
        return self._batch_generator()

    def _batch_generator(self):
        tokens_exhausted = False
        while not tokens_exhausted:
            batch = []
            try:
                for _ in range(self.batch_size):
                    batch.append(next(self.iterator))
            except StopIteration:
                tokens_exhausted = True

            self._ix += 1
            if self._ix > self._len:
                return
            if len(batch) > 0:
                yield self._to_tensor_batch(batch)

    def _to_tensor_batch(self, batch):
        tokens = torch.tensor([x["tokens"] for x in batch], dtype=torch.long)
        labels = torch.tensor([x["labels"] for x in batch], dtype=torch.long)
        return {"tokens": tokens, "labels": labels}

    def state_dict(self):
        """Get the current state of the dataloader."""
        return {
            'ix': self._ix,
        }

    def load_state_dict(self, state_dict):
        """Load the state of the dataloader."""
        self._ix = state_dict['ix']
        self.iterator = iter(self.dataset)
        # Skip to the correct position
        for _ in range(self._ix * self.batch_size):
            try:
                next(self.iterator)
            except StopIteration:
                break
class FullFinetuneRecipeDistributed(FTRecipeInterface):
    """
    Full finetuning recipe for dense transformer-based LLMs such as Llama2. This recipe supports
    distributed training and can be run on a single node (1 to 8 GPUs).

    Features:
        - FSDP. Supported using PyTorch's FSDP APIs. CPU offload of parameters, gradients, and optimizer states
            is supported via ``fsdp_cpu_offload``. Resharding of parameters after the forward pass is
            done by default (corresponding to FULL_SHARD sharding strategy), but can be disabled by setting the config
            ``fsdp_reshard_after_forward`` to False (this corresponds to SHARD_GRAD_OP sharding strategy).
            DDP is currently not supported. Training on CPU is not supported.

        - Activation Checkpointing. This can be controlled using the ``enable_activation_checkpointing``
            flag. Activation checkpointing helps reduce the memory footprint since we no longer keep
            activations in memory and instead recompute them during the backward pass. This is especially
            helpful for larger batch sizes when you're memory constrained. But these savings in memory
            come at the cost of training performance. In most cases training can slow-down quite a bit as
            a result of this activation recomputation.

        - Activation Offloading. This can be controlled using the ``enable_activation_offloading``
            flag. Activation offloading is a technique similar to activations checkpointing that helps
            reduce the memory footprint to prevent OOMs on CUDA and enable bigger batches. Where activations
            checkpointing drops the activation in the forward to recompute it later in the backward,
            activations offloading will drop the activation in the forward to the CPU and bring it
            back during the backward pass. As always, there is a tradeoff--these savings in memory can
            come at the cost of training performance and CPU resources. To recover some runtime cost,
            we've added an option to enable offloading on a different stream to permit overlapping with
            the computation. This option is currently only available on PyTorch 2.5 or later and will
            be enabled by default if an acceptable torch version is found. Activation offloading can be
            used in conjunction with activation checkpointing.

        - Precision. Full fp32 and bf16 training are supported. Precision is controlled using the ``dtype``
            flag. When ``dtype=bf16``, all activations, gradients and optimizer states are in bfloat16. In
            most cases this should halve the memory footprint of full precision (fp32) training, without
            loss in model quality (will depend on the model, training data and other settings). For
            GPUs which do not support bfloat16, we fall back to fp32. Mixed precision training and fp16
            precision are currently not supported.

        - Gradient Accumulation. You can simulate larger batch sizes by accumulating gradients. This is
            controlled using the ``gradient_accumulation_steps`` flag.

                Total Batch Size = batch_size * number of GPUs * gradient accumulation steps.

            For example: with batch_size=1, nproc_per_node=2 and gradient_accumulation_steps=32 we get a
            total batch size of 64.

            Gradient accumulation is especially useful when you are memory constrained. In this case,
            accumulating gradients might give you better training speed than enabling activation
            checkpointing.

        - Checkpointing. Model weights are checkpointed both at the end of each epoch and at the end of
            training. Optimizer state and recipe state (seed, total_epochs, number of epochs run etc) are
            only saved at the end of a given epoch and used in case of resuming training.

            Resuming training is controlled by the ``resume_from_checkpoint`` flag. Mid-epoch checkpointing is
            currently not supported.

            For more details on the checkpointer, please take a look at
            our checkpointer deepdive (https://pytorch.org/torchtune/main/deep_dives/checkpointer.html).

        - Logging. Terminal, Disk, WandB and TensorBoard are all supported.

        - Gradient Clipping. Gradient clipping is supported using the ``clip_grad_norm`` flag. By default,
            ``clip_grad_norm`` is set to ``None``. If you only want to log the grad norm, you can set
            ``clip_grad_norm='inf'``.

    For a full list of example configs for this recipe, run ``tune ls`` on the command line. Each config
    has example commands for how to kick-off training.

    Args:
        cfg (DictConfig): OmegaConf object parsed from yaml file

    Raises:
        ValueError: If ``dtype`` is set to fp16.
        RuntimeError: If ``dtype`` is set to bf16 and the hardware does not support bf16.
        RuntimeError: If ``left_pad_sequence`` is set as the data collator.
        RuntimeError: If ``enable_activation_offloading`` is True and device is not CUDA.
        RuntimeError: If ``enable_activation_offloading`` is True and ``enable_activation_checkpointing`` is False.
    """

    def __init__(self, cfg: DictConfig) -> None:
        self.MODE = cfg.my_mode
        device_type = cfg.device
        self._device = utils.get_device(device=cfg.device)
        self._dtype = training.get_dtype(cfg.dtype, device=self._device)
        self.trained_tensors = cfg.get("trained_tensors", "all")

        if self._dtype == torch.float16:
            raise ValueError(
                "full fp16 training is not supported with this recipe. Please use bf16 or fp32 instead."
            )

        # Set up the backend for distributed training (NCCL, GLOO, etc.)
        self._enable_async_checkpointing = cfg.get("enable_async_checkpointing", False)
        self.fsdp_cpu_offload = cfg.get("fsdp_cpu_offload", False)
        self.distributed_backend = training.get_distributed_backend(
            device_type,
            offload_ops_to_cpu=self.fsdp_cpu_offload
            or self._enable_async_checkpointing,
        )
        init_process_group(self.distributed_backend)

        # Initialize distributed variables
        self.world_size, self.rank = utils.get_world_size_and_rank()
        self._is_rank_zero = self.rank == 0
        self.tensor_parallel_plan = config.instantiate(
            cfg.get("tensor_parallel_plan", None)
        )
        self.tensor_parallel_dim = cfg.get("tensor_parallel_dim", 1)
        if self.tensor_parallel_dim > 1 and self.tensor_parallel_plan is None:
            raise ValueError(
                "Tensor Parallel plan needs to be provided when tensor parallel is enabled."
            )
        if self.world_size % self.tensor_parallel_dim != 0:
            raise ValueError(
                f"world_size {self.world_size} must be divisible by tensor_parallel_dim {self.tensor_parallel_dim}"
            )
        if self.tensor_parallel_dim > 1 and cfg.optimizer.get("fused", False):
            raise ValueError(
                "Tensor parallelism is currently incompatible with fused optimizer."
            )

        self.data_parallel_dim = self.world_size // self.tensor_parallel_dim

        # Logging attributes
        self._output_dir = cfg.output_dir
        self._log_every_n_steps = cfg.get("log_every_n_steps", 1)
        self._log_peak_memory_stats = cfg.get("log_peak_memory_stats", False)
        if self._log_peak_memory_stats and device_type != "cuda":
            log.info(
                "log_peak_memory_stats was set to True, however, training does not use cuda. Setting log_peak_memory_stats=False."
            )
            self._log_peak_memory_stats = False

        # Training cfg
        self._resume_from_checkpoint = cfg.resume_from_checkpoint
        self._gradient_accumulation_steps = cfg.gradient_accumulation_steps
        self._optimizer_in_bwd = cfg.get("optimizer_in_bwd", False)
        self._clip_grad_norm = cfg.get("clip_grad_norm", None)
        self._checkpoint_client = CheckpointClient(cfg)

        # Optimizer in backward is not compatible with gradient accumulation or gradient clipping
        if self._optimizer_in_bwd:
            if self._clip_grad_norm is not None:
                raise RuntimeError(
                    "Gradient clipping is not supported with optimizer in bwd."
                    "Please set clip_grad_norm=None, or optimizer_in_bwd=False."
                )
            if self._gradient_accumulation_steps > 1:
                raise RuntimeError(
                    "Gradient accumulation is not supported with optimizer in bwd."
                    "Please set gradient_accumulation_steps=1, or optimizer_in_bwd=False."
                )

        # activation checkpointing/offloading
        self._enable_activation_checkpointing = cfg.get(
            "enable_activation_checkpointing", False
        )
        self._enable_activation_offloading = cfg.get(
            "enable_activation_offloading", False
        )
        if self._enable_activation_offloading:
            if device_type != "cuda":
                raise RuntimeError(
                    "enable_activation_offloading should only be True when training on CUDA"
                )
            if not self._enable_activation_checkpointing:
                raise RuntimeError(
                    "enable_activation_offloading should only be True when enable_activation_checkpointing is True"
                )
        elif (
            self._enable_activation_checkpointing
            and cfg.checkpointer.model_type != "LLAMA3_VISION"
        ):
            utils.log_rank_zero(
                log,
                "Hint: enable_activation_checkpointing is True, but enable_activation_offloading isn't. "
                "Enabling activation offloading should reduce memory further.",
            )

        # These are public properties which are updated by the checkpoint loader
        # when ``resume_from_checkpoint`` is `True` or validated in tests
        self.seed = training.set_seed(
            seed=cfg.seed, debug_mode=cfg.get("cudnn_deterministic_mode", None)
        )
        self.epochs_run = 0
        self.total_epochs = cfg.epochs
        self.max_steps_per_epoch = cfg.max_steps_per_epoch
        self.global_step = 0

    def _print0(self, *args, **kwargs):
        if self._is_rank_zero:
            print(*args, flush=True, **kwargs)

    def _save_metrics(self, metrics: Dict[str, float]) -> None:
        """Save metrics to a file."""
        import os
        metrics_file = os.path.join(self._output_dir, f"metrics_{self.MODE}.pt")
        try:
            if os.path.exists(metrics_file):
                existing_metrics = torch.load(metrics_file)
                existing_metrics.append(metrics)
                torch.save(existing_metrics, metrics_file)
            else:
                torch.save([metrics], metrics_file)
        except Exception as e:
            print(f"WARNING!!!!! Error saving metrics: {e}")

    def _setup_data_val(
        self,
        local_batch_size: int,
    ) -> Tuple:
        """
        All data related setup happens here. This recipe currently supports only
        map-style datasets. If a state_dict is provided (meaning we are resuming a training run),
        it is loaded into the dataloader.
        """
        ds_4k = FineWebDataset(tokenizer=self._tokenizer, max_seq_len=4*1024, train=False)
        ds_16k = FineWebDataset(tokenizer=self._tokenizer, max_seq_len=16*1024, train=False)
        ds_64k = FineWebDataset(tokenizer=self._tokenizer, max_seq_len=64*1024, train=False)

        dataloader_4k = StreamDataLoader(ds_4k, batch_size=local_batch_size, max_tokens=VAL_TOKENS)
        dataloader_16k = StreamDataLoader(ds_16k, batch_size=max(1, local_batch_size // 4), max_tokens=VAL_TOKENS)
        dataloader_64k = StreamDataLoader(ds_64k, batch_size=1, max_tokens=VAL_TOKENS)

        return dataloader_4k, dataloader_16k, dataloader_64k

    def _run_validation(self, dataloader, seq_length: int) -> float:
        """Run validation on a single dataloader and return the average loss."""
        running_loss = 0
        num_tokens = 0

        t0_val = time.perf_counter()

        if 'scale_invariant' in self.MODE or 'alibi' in self.MODE:
            def causal_mask(b, h, q_idx, kv_idx): return q_idx >= kv_idx
            B = dataloader.batch_size; H = 32
            mask = create_block_mask_compiled(causal_mask, B, H, seq_length, seq_length, device="cuda", _compile=True)
        else:
            mask = None

        with torch.no_grad():
            for batch_idx, batch in enumerate(iter(dataloader)):
                utils.batch_to_device(batch, self._device)

                # Calculate number of unmasked tokens
                current_num_tokens = (
                    batch["labels"] != self._loss_fn.ignore_index
                ).sum()
                num_tokens += current_num_tokens

                current_loss = self._val_step(batch, seq_length=seq_length, batch_size=dataloader.batch_size, mask=mask) * current_num_tokens
                running_loss += current_loss

                # Only evaluate on a few batches to save time
                if num_tokens > VAL_TOKENS:
                    break
        # This will ensure that the logged loss matches what we're optimizing
        torch.distributed.all_reduce(num_tokens)
        torch.distributed.all_reduce(running_loss)

        final_loss = running_loss.item() / num_tokens if num_tokens > 0 else float('inf')
        self._print0(f"Val @ {seq_length} complete: num_tokens={num_tokens}, final_loss={final_loss}|Time: {(time.perf_counter() - t0_val):.2f}")
        return final_loss

    def _val_step(self, batch: Dict[str, torch.Tensor], seq_length=None, batch_size=None, mask=None) -> torch.Tensor:
        # Shape [b, s], needed for the loss not the model
        labels = batch.pop("labels")

        with self.activations_handling_ctx:
            logits = self._model(**batch, mask=mask)
        # Shift labels to compute loss
        labels = torch.hstack(
            (labels[..., 1:], self.ignore_labels_cache[: labels.shape[0]])
        )
        if not isinstance(logits, list):
            labels = labels.reshape(-1)
            logits = logits.reshape(-1, logits.size(-1))
        # Compute loss
        loss = self._loss_fn(logits, labels)
        # free logits otherwise it peaks backward memory
        del logits

        return loss

    def setup(self, cfg: DictConfig) -> None:
        """
        Setup the recipe. This includes training state (if resume_from_checkpoint is True),
        model, tokenizer, loss, optimizer, lr scheduler, sampler, and dataloader.
        """
        if self.fsdp_cpu_offload:
            # Utilize all available CPU cores for intra-op parallelism. This provides ~2x
            # speed up when benchmarking fused AdamW on CPU
            training.set_torch_num_threads()

        if self._is_rank_zero:
            # Initialize wandb
            if not DISABLE_WANDB:
                wandb.init(project="llama_pretrain_dist_28may", config=cfg, name=self.MODE)
            self._metric_logger = config.instantiate(cfg.metric_logger)
            # log config with parameter override
            self._metric_logger.log_config(cfg)

        # Load the base model
        checkpoint_dict = self._checkpoint_client.load_base_checkpoint()

        self._compile = cfg.get("compile", False)
        self._model = self._setup_model(
            cfg_model=cfg.model,
            enable_activation_checkpointing=self._enable_activation_checkpointing,
            enable_activation_offloading=self._enable_activation_offloading,
            custom_sharded_layers=cfg.get("custom_sharded_layers", None),
            fsdp_cpu_offload=self.fsdp_cpu_offload,
            reshard_after_forward=cfg.get("fsdp_reshard_after_forward", True),
            model_state_dict=checkpoint_dict[training.MODEL_KEY],
            ac_mode=cfg.get("ac_mode", None),
            ac_option=cfg.get("ac_option", None),
        )
        n_params = sum(p.numel() for p in self._model.parameters() if p.requires_grad)
        print(f"Model has {n_params / 1e9:.2f}B parameters", flush=True)
        self._model.max_seq_len = cfg.model_max_seq_len
        self.max_train_tokens = cfg.get("max_train_tokens", _50M)
        self._tokenizer = config.instantiate(cfg.tokenizer)

        self._optimizer = self._setup_optimizer(
            cfg_optimizer=cfg.optimizer,
            optimizer_in_bwd=self._optimizer_in_bwd,
            opt_state_dict=(
                checkpoint_dict[training.OPT_KEY]
                if training.OPT_KEY in checkpoint_dict
                else None
            ),
        )

        if self._resume_from_checkpoint:
            # If async checkpointing is enabled, intermediate checkpoints are saved asynchronously
            # using the DistributedCheckpointer.
            # Therefore the recipe needs to load the distributed checkpoint to restore the training
            # progress.
            if self._enable_async_checkpointing:
                try:
                    checkpoint_dict = (
                        self._checkpoint_client.load_distributed_checkpoint(
                            self._model,
                            (
                                self._optim_ckpt_wrapper
                                if self._optimizer_in_bwd
                                else self._optimizer
                            ),
                        )
                    )
                except Exception as e:
                    log.warning(
                        f"Failed to load distributed checkpoint: {e}. Training will start from the base checkpoint."
                    )

            # Update the recipe state from the checkpoint state dict.
            self._update_recipe_state(checkpoint_dict)

        # initialize loss
        self._loss_fn = config.instantiate(cfg.loss)

        if self._compile:
            training.compile_loss(self._loss_fn, verbose=self._is_rank_zero)

        if self._loss_fn.__class__.__name__ == "CEWithChunkedOutputLoss":
            # set num_output_chunks for model
            self._model.set_num_output_chunks(self._loss_fn.num_output_chunks)

        utils.log_rank_zero(log, "Loss is initialized.")

        # sampler and dataloader depend on the tokenizer and loss_fn and should be
        # setup after both of these are initialized
        collate_name = cfg.get("collate_fn", "torchtune.data.padded_collate_sft")
        self.batch_size = cfg.batch_size
        self._dataloader = self._setup_data(
            batch_size=cfg.batch_size,
            dataloader_state_dict=(
                checkpoint_dict[training.DATALOADER_KEY]
                if self._resume_from_checkpoint and training.DATALOADER_KEY in checkpoint_dict
                else None
            ),
        )

        # Setup validation dataloaders
        self._val_dataloader_4k, self._val_dataloader_16k, self._val_dataloader_64k = self._setup_data_val(
            local_batch_size=cfg.batch_size,
        )

        # Finally update the recipe state which can only be correctly set after all of the
        # other components have been initialized and updated.
        #
        # Number of training steps in each epoch depends on the number of batches produced
        # by the dataloader, the max_steps_per_epoch param set by the user and the
        # gradient_accumulation_steps param. This value is used for logging and tracking
        # training state. The computation should happen after the dataloader has been setup
        self._steps_per_epoch = (
            len(self._dataloader) // self._gradient_accumulation_steps
        )
        if (
            self.max_steps_per_epoch is not None
            and self.max_steps_per_epoch < self._steps_per_epoch
        ):
            self._steps_per_epoch = self.max_steps_per_epoch
        self.global_step = self.epochs_run * self._steps_per_epoch
        self._print0(f"Steps per epoch: {self._steps_per_epoch}")
        self._print0(f"Total steps: {self.total_epochs * self._steps_per_epoch}")
        self._print0(f"Global step: {self.global_step}")

        # Setup lr scheduler
        self._lr_scheduler = self._setup_lr_scheduler(
            cfg_lr_scheduler=cfg.get("lr_scheduler", None),
            num_training_steps=self.total_epochs * self._steps_per_epoch,
            last_epoch=self.global_step - 1,
        )

        # Set up profiler, returns DummyProfiler (nullcontext object with no-op `step` method)
        # if cfg is missing profiler key or if `cfg.profiler.enabled = False`
        self._profiler = self._setup_profiler(cfg.get(PROFILER_KEY, None))

        # Used to ignore labels for loss computation
        self.ignore_labels_cache = torch.full(
            (cfg.batch_size, 1), self._loss_fn.ignore_index, device=self._device
        )

    def _setup_lr_scheduler(
        self,
        cfg_lr_scheduler: Optional[DictConfig],
        num_training_steps: int,
        last_epoch: int,
    ) -> Optional[Optimizer]:
        """
        Set up the learning rate scheduler based on the provided configuration.
        It supports both standard optimization and optimizer-in-backward cases.

        Args:
            cfg_lr_scheduler (Optional[DictConfig]): The learning rate scheduler configuration.
            num_training_steps (int): The total number of training steps.
            last_epoch (int): The index of the last epoch.

        Returns:
            lr_scheduler (Optional[Optimizer]): The learning rate scheduler.
        """
        if cfg_lr_scheduler is None:
            if self._is_rank_zero:
                log.info(
                    "No learning rate scheduler configured. Using constant learning rate."
                )
            return None

        if self._optimizer_in_bwd:
            # Use the first optimizer from the wrapper to represent the learning rate
            optimizer = next(iter(self._optim_ckpt_wrapper.optim_map.values()))
        else:
            # Standard case: use the single optimizer
            optimizer = self._optimizer

        # Instantiate the learning rate scheduler
        lr_scheduler = config.instantiate(
            cfg_lr_scheduler,
            optimizer,
            num_training_steps=num_training_steps,
            last_epoch=last_epoch,
        )

        if self._optimizer_in_bwd:
            # Modify the scheduler for optimizer_in_bwd case
            self._optim_ckpt_wrapper.set_lr_scheduler(lr_scheduler)

        if self._is_rank_zero:
            log.info("Learning rate scheduler is initialized.")

        return lr_scheduler

    def _setup_profiler(
        self, cfg_profiler: Optional[DictConfig] = None
    ) -> Union[torch.profiler.profile, DummyProfiler]:
        """
        Parses the `profiler` section of top-level `cfg` and sets up profiler

        Args:
            cfg_profiler (Optional[DictConfig]): ``profiler`` section of the top-level ``cfg`` (the main config passed to
                `recipe.main`). Default None.

        Returns:
            profiler: Union[torch.profiler.profile, DummyProfiler] - DummyProfiler is a nullcontext with no-op methods
            for `start`, `stop`, and `step` that can be used in place of `torch.profiler.profile` if profiler is not enabled such
            that the instrumented training loop does not need to be changed profiling is disabled.

        The profiler config can be provided in configs under the `profiler` key with the following layout:

        .. code-block:: yaml
            profiler:
                enabled: bool

                #Output directory of trace artifacts
                output_dir: str

            #`torch.profiler.ProfilerActivity` types to trace
            cpu: bool
            cuda: bool

                #Trace options
                profile_memory: bool
                with_stack: bool
                record_shapes: bool
                with_flops: bool

            # `torch.profiler.schedule` options:
            # wait_steps -> wait, warmup_steps -> warmup, active_steps -> active, num_cycles -> repeat
            wait_steps: int
            warmup_steps: int
            active_steps: int
            num_cycles: int
        """
        # Missing profiler section in config, assume disabled
        if cfg_profiler is None:
            cfg_profiler = DictConfig({"enabled": False})

        # Check that component is included and set correctly
        if cfg_profiler.get("_component_", None) is None:
            cfg_profiler["_component_"] = "torchtune.training.setup_torch_profiler"
        else:
            assert (
                cfg_profiler.get("_component_")
                == "torchtune.training.setup_torch_profiler"
            ), "Only torch profiler supported currently: component must be `torchtune.training.setup_torch_profiler`"

        profiler, profiler_cfg = config.instantiate(cfg_profiler)

        utils.log_rank_zero(
            log, f" Profiler config after instantiation: {profiler_cfg}"
        )
        if self._is_rank_zero:
            self.profiler_profile_memory = profiler_cfg.get("profile_memory", False)
            if profiler_cfg["enabled"]:
                self.profiler_wait_steps = profiler_cfg["wait_steps"]
                self.profiler_warmup_steps = profiler_cfg["warmup_steps"]
                self.profiler_active_steps = profiler_cfg["active_steps"]

        return profiler

    def _setup_model(
        self,
        cfg_model: DictConfig,
        enable_activation_checkpointing: bool,
        enable_activation_offloading: bool,
        fsdp_cpu_offload: bool,
        reshard_after_forward: bool,
        model_state_dict: Dict[str, Any],
        custom_sharded_layers: Optional[List[str]] = None,
        ac_mode: Optional[str] = None,
        ac_option: Optional[int] = None,
    ) -> nn.Module:
        """
        Model initialization has some important considerations:
           a. To minimize GPU peak memory, we initialize the model on meta device with
              the right dtype
           b. All ranks calls ``load_state_dict`` without peaking CPU RAMs since
              full state dicts are loaded with ``torch.load(mmap=True)``
        """

        utils.log_rank_zero(
            log,
            "Distributed training is enabled. Instantiating model and loading checkpoint on Rank 0 ...",
        )
        init_start = time.perf_counter()

        pos_emb = None; rms_norm = RMSNorm()
        self._print0(f"MODE: {self.MODE}")
        with training.set_default_dtype(self._dtype), torch.device("meta"):
            model = config.instantiate(cfg_model)
            for layer_ix, module in enumerate(model.modules()):
                if hasattr(module, "attn"):
                    if self.MODE == 'nope':
                        if pos_emb is None: pos_emb = NoPE()
                        module.attn.pos_embeddings = pos_emb
                    elif self.MODE == 'scale_invariant_p_rope':
                        head_dim = module.attn.head_dim
                        if pos_emb is None: pos_emb = pRotaryPositionalEmbeddings(head_dim)
                        module.attn.q_norm = rms_norm
                        module.attn.k_norm = rms_norm
                        module.attn.pos_embeddings = pos_emb
                        module.attn.forward = types.MethodType(scale_invariant_forward, module.attn)
                    elif self.MODE == 'scale_invariant_p_rope_no_qk_norm':
                        head_dim = module.attn.head_dim
                        if pos_emb is None: pos_emb = pRotaryPositionalEmbeddings(head_dim)
                        module.attn.pos_embeddings = pos_emb
                        module.attn.forward = types.MethodType(scale_invariant_forward, module.attn)
                    elif self.MODE == 'scale_invariant_nope_no_qk_norm':
                        head_dim = module.attn.head_dim
                        if pos_emb is None: pos_emb = NoPE()
                        module.attn.pos_embeddings = pos_emb
                        module.attn.forward = types.MethodType(scale_invariant_forward, module.attn)
                    elif self.MODE == 'alibi':
                        module.attn.pos_embeddings = None
                        module.attn.forward = types.MethodType(alibi_forward, module.attn)
                    elif self.MODE == 'p_rope':
                        head_dim = module.attn.head_dim
                        if pos_emb is None: pos_emb = pRotaryPositionalEmbeddings(head_dim, max_seq_len=MAX_SEQ_LEN)
                        module.attn.pos_embeddings = pos_emb
                    elif self.MODE == 'nousyarn':
                        # head_dim = module.attn.head_dim
                        # if pos_emb is None: pos_emb = YaRNPositionalEmbeddings(head_dim, max_seq_len=MAX_SEQ_LEN)
                        # module.attn.pos_embeddings = pos_emb
                        pass # do this outside of  meta ctx
                    elif 'logn_and_p_rope' in self.MODE:
                        head_dim = module.attn.head_dim
                        if pos_emb is None: pos_emb = pRotaryPositionalEmbeddings(head_dim)
                        module.attn.pos_embeddings = pos_emb
                        module.attn.forward = types.MethodType(logn_attn_forward, module.attn)
                    elif 'logn_and_rope' in self.MODE:
                        head_dim = module.attn.head_dim
                        if pos_emb is None: pos_emb = RotaryPositionalEmbeddings(head_dim, max_seq_len=MAX_SEQ_LEN)
                        module.attn.pos_embeddings = pos_emb
                        module.attn.forward = types.MethodType(logn_attn_forward, module.attn)
                    elif 'logn_and_ntk' in self.MODE:
                        head_dim = module.attn.head_dim
                        if pos_emb is None: pos_emb = NTKAwareRotaryPositionalEmbeddings(head_dim, max_seq_len=MAX_SEQ_LEN)
                        module.attn.pos_embeddings = pos_emb
                        module.attn.forward = types.MethodType(logn_attn_forward, module.attn)
                    elif self.MODE == 'rope':
                        head_dim = module.attn.head_dim
                        if pos_emb is None: pos_emb = RotaryPositionalEmbeddings(head_dim, max_seq_len=MAX_SEQ_LEN)
                        module.attn.pos_embeddings = pos_emb
                    elif self.MODE == 'ntk_aware_rope':
                        head_dim = module.attn.head_dim
                        if pos_emb is None: pos_emb = NTKAwareRotaryPositionalEmbeddings(head_dim, max_seq_len=MAX_SEQ_LEN)
                        module.attn.pos_embeddings = pos_emb
                    else:
                        raise ValueError(f"Invalid MODE: {self.MODE}")

        # set extra parameters -- these will NOT be sharded! so don't use device('meta')...
        if 'logn_and' in self.MODE: # need to set extra params
            for layer_ix, module in enumerate(model.modules()):
                if hasattr(module, "attn"):
                    head_dim = module.attn.head_dim
                    special_s = 'special_s' in self.MODE
                    logn_attn_module = LogNAttn(module.attn.num_heads, special_s=special_s).to(self._device, dtype=self._dtype)
                    module.attn.logn_attn_module = logn_attn_module

        if self.MODE == 'nousyarn': # need to set extra params
            for layer_ix, module in enumerate(model.modules()):
                if hasattr(module, "attn"):
                    head_dim = module.attn.head_dim
                    if pos_emb is None: pos_emb = NousYaRN(head_dim)
                    module.attn.pos_embeddings = pos_emb

        # Freeze parameters if needed
        if self.trained_tensors == "only_attn_and_norm":
            model.tok_embeddings.weight.requires_grad = False
            model.output.weight.requires_grad = False
            for layer in model.layers:
                layer.mlp.w1.weight.requires_grad = False
                layer.mlp.w2.weight.requires_grad = False
                layer.mlp.w3.weight.requires_grad = False


        if self._compile:
            training.compile_model(model, verbose=self._is_rank_zero)

        device_mesh = init_device_mesh(
            self._device.type,
            mesh_shape=(self.data_parallel_dim, self.tensor_parallel_dim),
            mesh_dim_names=("dp", "tp"),
        )
        self.dp_size = device_mesh["dp"].size()
        self.dp_rank = device_mesh["dp"].get_local_rank()

        # Apply tensor parallelism to the model
        if self.tensor_parallel_dim > 1:
            # Use the local number (num_heads, num_kv_heads, embed_dim) to account for tensor parallel
            model = training.prepare_mha_for_tp(model, device_mesh["tp"])
            parallelize_module(
                model,
                device_mesh["tp"],
                parallelize_plan=self.tensor_parallel_plan,
            )

        # We currently have two versions of activation checkpointing in this recipe
        # for testing and BC purposes. ``enable_activation_checkpointing`` controls
        # the older version of AC and this behavior is unchanged
        # ac_mode and ac_option together control selective AC. This is only enabled
        # when these are set AND ``enable_activation_checkpointing`` is set to False
        # We'll clean this up as soon as testing of AC is complete
        if (not enable_activation_checkpointing) and (ac_mode is not None):
            apply_selective_activation_checkpointing(
                model,
                ac_mode,
                ac_option,
            )

        # original activation checkpointing (full) - flip the condition above
        if enable_activation_checkpointing and ac_mode is None:
            training.set_activation_checkpointing(
                model, auto_wrap_policy={modules.TransformerSelfAttentionLayer}
            )


        # Apply Fully Sharded Data Parallelism to the model
        if self.data_parallel_dim > 1:
            fsdp_shard_conditions = [
                partial(
                    training.get_shard_conditions,
                    names_to_match=custom_sharded_layers,
                )
            ]
            training.shard_model(
                model=model,
                shard_conditions=fsdp_shard_conditions,
                cpu_offload=fsdp_cpu_offload,
                reshard_after_forward=reshard_after_forward,
                dp_mesh=device_mesh["dp"],
            )
        else:
            print("NOT USING FSDP", flush=True)
            raise ValueError("NOT USING FSDP")

        with training.set_default_dtype(self._dtype), self._device:
            for m in model.modules():
                # RoPE is not covered in state dict
                if hasattr(m, "rope_init"):
                    m.rope_init()

        # This method will convert the full model state dict into a sharded state
        # dict and load into the model
        training.load_from_full_model_state_dict(
            model,
            model_state_dict,
            self._device,
            strict=False,
            cpu_offload=fsdp_cpu_offload,
        )

        # activation offloading
        self.activations_handling_ctx = training.get_act_offloading_ctx_manager(
            model, enable_activation_offloading
        )

        # Ensure no params and buffers are on meta device
        training.validate_no_params_on_meta_device(model)

        utils.log_rank_zero(
            log,
            f"Instantiating model and loading checkpoint took {time.perf_counter() - init_start:.2f} secs",
        )

        if self._is_rank_zero:
            memory_stats = training.get_memory_stats(device=self._device)
            training.log_memory_stats(memory_stats)

        # synchronize before training begins
        torch.distributed.barrier()

        return model

    def _setup_optimizer(
        self,
        cfg_optimizer: DictConfig,
        optimizer_in_bwd: bool = False,
        opt_state_dict: Optional[Dict[str, Any]] = None,
    ) -> Optional[Optimizer]:
        if optimizer_in_bwd:
            # Maintain a dict of optims for every parameter.
            optim_dict = {
                param: config.instantiate(cfg_optimizer, [param])
                for param in self._model.parameters() if param.requires_grad
            }

            # Register optimizer step hooks on the model to run optimizer in backward.
            training.register_optim_in_bwd_hooks(
                model=self._model, optim_dict=optim_dict
            )
            # Create a wrapper for checkpoint save/load of optimizer states when running in backward.
            self._optim_ckpt_wrapper = training.create_optim_in_bwd_wrapper(
                model=self._model, optim_dict=optim_dict
            )
            # Load optimizer states for each param. If optimizer states are being restored in an optimizer in
            # backward run, these need to have been saved with the same setting. Cannot restore from runs that
            # did not use optimizer in backward.
            if opt_state_dict is not None:
                for param in opt_state_dict.keys():
                    try:
                        training.load_from_full_optimizer_state_dict(
                            self._model,
                            self._optim_ckpt_wrapper.optim_map[param],
                            opt_state_dict[param],
                            self._device,
                        )
                    except BaseException as e:
                        raise RuntimeError(
                            "Failed loading in-backward optimizer checkpoints."
                            "Please make sure run being restored from was using in-backward optimizer."
                        ) from e
            utils.log_rank_zero(log, "In-backward optimizers are set up.")
            return None
        else:
            optimizer = config.instantiate(cfg_optimizer, self._model.parameters())
            if opt_state_dict:
                training.load_from_full_optimizer_state_dict(
                    self._model,
                    optimizer,
                    opt_state_dict,
                    self._device,
                )

            utils.log_rank_zero(log, "Optimizer is initialized.")
            return optimizer

    def _setup_data(
        self,
        batch_size: int,
        dataloader_state_dict: Optional[Dict[str, Any]] = None,
    ):
        """
        All data related setup happens here. This recipe currently supports only
        map-style datasets. If a state_dict is provided (meaning we are resuming a training run),
        it is loaded into the dataloader.
        """
        ds = FineWebDataset(tokenizer=self._tokenizer, max_seq_len=4*1024, train=True)
        dataloader = StreamDataLoader(ds, batch_size=batch_size, max_tokens=self.max_train_tokens)

        if dataloader_state_dict is not None:
            dataloader.load_state_dict(dataloader_state_dict)

        return dataloader

    def val(self) -> None:
        training.cleanup_before_training()
        self._model.eval()
        val_dict = dict()

        # Run validation on different sequence lengths
        for seq_len, dataloader in [
            (4096, self._val_dataloader_4k),
            (16384, self._val_dataloader_16k),
            (65536, self._val_dataloader_64k),
        ]:
            if dataloader is not None:
                dataloader.reset()  # Reset the dataloader before validation
                val_loss = self._run_validation(dataloader, seq_len)
                val_dict[f"val_loss_{seq_len}"] = val_loss.item()

        # Log per-step metrics
        if self._is_rank_zero:
            # Log to wandb and save metrics
            if self._is_rank_zero:
                if not DISABLE_WANDB:
                    wandb.log(val_dict, step=0)
                self._save_metrics(val_dict)
                self._metric_logger.log_dict(
                    val_dict,
                    step=0,
                )

    def train(self) -> None:
        """
        The core training loop.
        """
        # clean up before training begins
        training.cleanup_before_training()

        # zero out the gradients before starting training
        if not self._optimizer_in_bwd:
            self._optimizer.zero_grad()
        else:
            for opt in self._optim_ckpt_wrapper.optim_map.values():
                opt.zero_grad()

        # Initialize tokens count and running loss (for grad accumulation)
        t0 = time.perf_counter()
        running_loss = 0
        num_tokens = 0

        self._profiler.start()

        if 'scale_invariant' in self.MODE or 'alibi' in self.MODE:
            def causal_mask(b, h, q_idx, kv_idx): return q_idx >= kv_idx
            train_seq_len = 4096 # always 4096!
            local_batch_size = self.batch_size; H = 32
            mask = create_block_mask_compiled(causal_mask, local_batch_size, H, train_seq_len, train_seq_len, device="cuda", _compile=True)
        else:
            mask = None
        # self.epochs_run should be non-zero when we're resuming from a checkpoint
        nbatches = len(self._dataloader)
        self._print0(f"TOTAL NBATCHES = {nbatches}")
        self._print0(f"TOTAL STEPS = {nbatches // self._gradient_accumulation_steps}")
        for curr_epoch in range(self.epochs_run, self.total_epochs):
            for idx, batch in enumerate(self._dataloader):
                # Clear memory before processing batch
                torch.cuda.empty_cache()
                gc.collect()

                # Start tracking CUDA memory for active steps for just the first epoch
                if (
                    self._is_rank_zero
                    and curr_epoch == 0
                    and self.profiler_profile_memory
                    and idx == self.profiler_wait_steps + self.profiler_warmup_steps
                    and self._device.type == "cuda"
                ):
                    torch.cuda.memory._record_memory_history()

                utils.batch_to_device(batch, self._device)

                # Calculate the number of unmasked tokens in the current batch
                current_num_tokens = (
                    batch["labels"] != self._loss_fn.ignore_index
                ).sum()
                num_tokens += current_num_tokens

                # Shape [b, s], needed for the loss not the model
                labels = batch.pop("labels")

                with self.activations_handling_ctx:
                    logits = self._model(**batch, mask=mask)
                # Shift labels to compute loss
                labels = torch.hstack(
                    (labels[..., 1:], self.ignore_labels_cache[: labels.shape[0]])
                )
                if not isinstance(logits, list):
                    labels = labels.reshape(-1)
                    logits = logits.reshape(-1, logits.size(-1))

                # Compute loss
                current_loss = self._loss_fn(logits, labels) * current_num_tokens

                # free logits otherwise it peaks backward memory
                del logits

                running_loss += current_loss

                # For optimizer in backward, we need to normalize before calling backward
                # This case and gradient accumulation are mutually exclusive
                if self._optimizer_in_bwd:
                    torch.distributed.all_reduce(num_tokens)
                    torch.distributed.all_reduce(running_loss)

                    # We multiply by world_size to undo FSDP2 gradient normalization.
                    current_loss = current_loss * (self.world_size / num_tokens)

                current_loss.backward()

                # Step with optimizer
                if (idx + 1) % self._gradient_accumulation_steps == 0:
                    if not self._optimizer_in_bwd:

                        # Get total number of tokens across all ranks to normalize gradients
                        torch.distributed.all_reduce(num_tokens)
                        # This will ensure that the logged loss matches what we're optimizing
                        torch.distributed.all_reduce(running_loss)
                        # Manually scale the gradients from unnormalized loss by total # of tokens
                        # We multiply by world_size to undo FSDP2 gradient normalization.
                        training.scale_grads(self._model, self.world_size / num_tokens)
                        if self._clip_grad_norm is not None:
                            grad_norm = torch.nn.utils.clip_grad_norm_(
                                self._model.parameters(),
                                max_norm=float(self._clip_grad_norm),
                            )
                            # If sharded, collect the DTensor here
                            if isinstance(grad_norm, DTensor):
                                grad_norm = grad_norm.full_tensor()
                        self._optimizer.step()
                        self._optimizer.zero_grad(set_to_none=True)




                    # Update the number of steps when the weights are updated
                    self.global_step += 1

                    # Step the learning rate scheduler
                    if self._lr_scheduler is not None:
                        self._lr_scheduler.step()

                    loss_to_log = running_loss.item() / num_tokens
                    # pbar.update(1)
                    # pbar.set_description(
                    #     f"{curr_epoch + 1}|{self.global_step}|Loss: {loss_to_log}"
                    # )
                    self._print0(f"{self.global_step}|batch {idx+1}/{nbatches}|Loss: {loss_to_log}|Time: {(time.perf_counter() - t0):.2f}")

                    # Run validation every VAL_EVERY steps
                    val_dict = dict()
                    if self.global_step % VAL_EVERY == 0:
                        self._print0("STARTING VAL")
                        self._model.eval()

                        # Run validation on different sequence lengths
                        for seq_len, dataloader in [
                            (4096, self._val_dataloader_4k),
                            (16384, self._val_dataloader_16k),
                            (65536, self._val_dataloader_64k),
                        ]:
                            if dataloader is not None:
                                dataloader.reset()  # Reset the dataloader before validation
                                val_loss = self._run_validation(dataloader, seq_len)
                                val_dict[f"val_loss_{seq_len}"] = val_loss.item()

                        self._model.train()
                        self._print0("ENDING VAL")

                    # Log per-step metrics
                    if (
                        self.global_step % self._log_every_n_steps == 0
                        and self._is_rank_zero
                    ):
                        time_per_step = time.perf_counter() - t0
                        log_dict = {
                            "loss": loss_to_log.item(),
                            "lr": get_lr(
                                (
                                    self._optimizer
                                    if not self._optimizer_in_bwd
                                    else self._optim_ckpt_wrapper
                                ),
                            ),
                            "tokens_per_second_per_gpu": num_tokens.item()
                            / (time_per_step * self.world_size),
                            'step': self.global_step,
                        }
                        if self._log_peak_memory_stats:
                            log_dict.update(
                                training.get_memory_stats(device=self._device)
                            )
                        if self._clip_grad_norm is not None:
                            log_dict.update({"grad_norm": grad_norm})

                        log_dict.update(val_dict)

                        # Log to wandb and save metrics
                        if self._is_rank_zero:
                            if not DISABLE_WANDB:
                                wandb.log(log_dict, step=self.global_step)
                            self._save_metrics(log_dict)
                            self._metric_logger.log_dict(
                                log_dict,
                                step=self.global_step,
                            )

                    # Reset running stats for the next step
                    running_loss = 0
                    num_tokens = 0
                    t0 = time.perf_counter()

                    # Stop tracking CUDA memory now that active steps are complete
                    if (
                        self._is_rank_zero
                        and curr_epoch == 0
                        and self.profiler_profile_memory
                        and idx
                        == self.profiler_wait_steps
                        + self.profiler_warmup_steps
                        + self.profiler_active_steps
                        and self._device.type == "cuda"
                    ):
                        torch.cuda.memory._record_memory_history(enabled=None)

                    # Step profiler
                    # Note that this is called within gradient accumulation block, hence
                    # will include multiple forward / backward passes if gradient accumulation > 1
                    self._profiler.step()

                if (
                    (idx + 1) // self._gradient_accumulation_steps
                ) == self.max_steps_per_epoch:
                    break

            self.epochs_run += 1
            if 'logn_and' in self.MODE:
                import datetime
                import os
                ## remove logn_attn_modules and save them separately
                logn_attn_dict = {}
                for layer_ix, module in enumerate(self._model.modules()):
                    if hasattr(module, "attn"):
                        if hasattr(module.attn, "logn_attn_module"):
                            logn_attn_dict[layer_ix] = module.attn.logn_attn_module.state_dict()
                        module.attn.logn_attn_module = None
                    # get datetime string
                dt_str = datetime.datetime.now().strftime("%Y%m%d_%H%M%S")
                logn_s_file = os.path.join(self._output_dir, f"logn_attn_dict_{curr_epoch}_{dt_str}.pt")
                torch.save(logn_attn_dict, logn_s_file)

            self._checkpoint_client.save_checkpoint(
                model=self._model,
                optimizer=(
                    self._optimizer
                    if not self._optimizer_in_bwd
                    else self._optim_ckpt_wrapper
                ),
                training_progress=TrainingProgress(
                    seed=self.seed,
                    epochs_run=self.epochs_run,
                    total_epochs=self.total_epochs,
                    max_steps_per_epoch=self.max_steps_per_epoch,
                    dataloader_state_dict=self._dataloader.state_dict(),
                ),
                epoch=curr_epoch,
            )

        self._profiler.stop()

    def cleanup(self) -> None:
        if self._is_rank_zero:
            self._metric_logger.close()
            if not DISABLE_WANDB:
                wandb.finish()
        destroy_process_group()


@config.parse
def recipe_main(cfg: DictConfig) -> None:
    """
    Entry point for the recipe.

    Configurable parameters are read in the following order:
        - Parameters specified in config (see available configs through ``tune ls``)
        - Overwritten by arguments from the command-line
    """
    config.log_config(recipe_name="FullFinetuneRecipeDistributed", cfg=cfg)
    recipe = FullFinetuneRecipeDistributed(cfg=cfg)
    recipe.setup(cfg=cfg)
    if recipe.MODE == 'rope' or recipe.MODE == 'ntk_aware_rope': # these methods don't need training!
        recipe.val()
    else:
        recipe.train()
    recipe.cleanup()


if __name__ == "__main__":
    sys.exit(recipe_main())