// SPDX-License-Identifier: GPL-2.0
/*
 * XDR support for nfsd
 *
 * Copyright (C) 1995, 1996 Olaf Kirch <okir@monad.swb.de>
 */

#include "vfs.h"
#include "xdr.h"
#include "auth.h"

/*
 * Mapping of S_IF* types to NFS file types
 */
static const u32 nfs_ftypes[] = {
	NFNON,  NFCHR,  NFCHR, NFBAD,
	NFDIR,  NFBAD,  NFBLK, NFBAD,
	NFREG,  NFBAD,  NFLNK, NFBAD,
	NFSOCK, NFBAD,  NFLNK, NFBAD,
};


/*
 * Basic NFSv2 data types (RFC 1094 Section 2.3)
 */

/**
 * svcxdr_encode_stat - Encode an NFSv2 status code
 * @xdr: XDR stream
 * @status: status value to encode
 *
 * Return values:
 *   %false: Send buffer space was exhausted
 *   %true: Success
 */
bool
svcxdr_encode_stat(struct xdr_stream *xdr, __be32 status)
{
	__be32 *p;

	p = xdr_reserve_space(xdr, sizeof(status));
	if (!p)
		return false;
	*p = status;

	return true;
}

/**
 * svcxdr_decode_fhandle - Decode an NFSv2 file handle
 * @xdr: XDR stream positioned at an encoded NFSv2 FH
 * @fhp: OUT: filled-in server file handle
 *
 * Return values:
 *  %false: The encoded file handle was not valid
 *  %true: @fhp has been initialized
 */
bool
svcxdr_decode_fhandle(struct xdr_stream *xdr, struct svc_fh *fhp)
{
	__be32 *p;

	p = xdr_inline_decode(xdr, NFS_FHSIZE);
	if (!p)
		return false;
	fh_init(fhp, NFS_FHSIZE);
	memcpy(&amp;fhp-&gt;fh_handle.fh_raw, p, NFS_FHSIZE);
	fhp-&gt;fh_handle.fh_size = NFS_FHSIZE;

	return true;
}

static bool
svcxdr_encode_fhandle(struct xdr_stream *xdr, const struct svc_fh *fhp)
{
	__be32 *p;

	p = xdr_reserve_space(xdr, NFS_FHSIZE);
	if (!p)
		return false;
	memcpy(p, &amp;fhp-&gt;fh_handle.fh_raw, NFS_FHSIZE);

	return true;
}

static __be32 *
encode_timeval(__be32 *p, const struct timespec64 *time)
{
	*p++ = cpu_to_be32((u32)time-&gt;tv_sec);
	if (time-&gt;tv_nsec)
		*p++ = cpu_to_be32(time-&gt;tv_nsec / NSEC_PER_USEC);
	else
		*p++ = xdr_zero;
	return p;
}

static bool
svcxdr_decode_filename(struct xdr_stream *xdr, char **name, unsigned int *len)
{
	u32 size, i;
	__be32 *p;
	char *c;

	if (xdr_stream_decode_u32(xdr, &amp;size) &lt; 0)
		return false;
	if (size == 0 || size &gt; NFS_MAXNAMLEN)
		return false;
	p = xdr_inline_decode(xdr, size);
	if (!p)
		return false;

	*len = size;
	*name = (char *)p;
	for (i = 0, c = *name; i &lt; size; i++, c++)
		if (*c == '\0' || *c == '/')
			return false;

	return true;
}

static bool
svcxdr_decode_diropargs(struct xdr_stream *xdr, struct svc_fh *fhp,
			char **name, unsigned int *len)
{
	return svcxdr_decode_fhandle(xdr, fhp) &amp;&amp;
		svcxdr_decode_filename(xdr, name, len);
}

static bool
svcxdr_decode_sattr(struct svc_rqst *rqstp, struct xdr_stream *xdr,
		    struct iattr *iap)
{
	u32 tmp1, tmp2;
	__be32 *p;

	p = xdr_inline_decode(xdr, XDR_UNIT * 8);
	if (!p)
		return false;

	iap-&gt;ia_valid = 0;

	/*
	 * Some Sun clients put 0xffff in the mode field when they
	 * mean 0xffffffff.
	 */
	tmp1 = be32_to_cpup(p++);
	if (tmp1 != (u32)-1 &amp;&amp; tmp1 != 0xffff) {
		iap-&gt;ia_valid |= ATTR_MODE;
		iap-&gt;ia_mode = tmp1;
	}

	tmp1 = be32_to_cpup(p++);
	if (tmp1 != (u32)-1) {
		iap-&gt;ia_uid = make_kuid(nfsd_user_namespace(rqstp), tmp1);
		if (uid_valid(iap-&gt;ia_uid))
			iap-&gt;ia_valid |= ATTR_UID;
	}

	tmp1 = be32_to_cpup(p++);
	if (tmp1 != (u32)-1) {
		iap-&gt;ia_gid = make_kgid(nfsd_user_namespace(rqstp), tmp1);
		if (gid_valid(iap-&gt;ia_gid))
			iap-&gt;ia_valid |= ATTR_GID;
	}

	tmp1 = be32_to_cpup(p++);
	if (tmp1 != (u32)-1) {
		iap-&gt;ia_valid |= ATTR_SIZE;
		iap-&gt;ia_size = tmp1;
	}

	tmp1 = be32_to_cpup(p++);
	tmp2 = be32_to_cpup(p++);
	if (tmp1 != (u32)-1 &amp;&amp; tmp2 != (u32)-1) {
		iap-&gt;ia_valid |= ATTR_ATIME | ATTR_ATIME_SET;
		iap-&gt;ia_atime.tv_sec = tmp1;
		iap-&gt;ia_atime.tv_nsec = tmp2 * NSEC_PER_USEC;
	}

	tmp1 = be32_to_cpup(p++);
	tmp2 = be32_to_cpup(p++);
	if (tmp1 != (u32)-1 &amp;&amp; tmp2 != (u32)-1) {
		iap-&gt;ia_valid |= ATTR_MTIME | ATTR_MTIME_SET;
		iap-&gt;ia_mtime.tv_sec = tmp1;
		iap-&gt;ia_mtime.tv_nsec = tmp2 * NSEC_PER_USEC;
		/*
		 * Passing the invalid value useconds=1000000 for mtime
		 * is a Sun convention for "set both mtime and atime to
		 * current server time".  It's needed to make permissions
		 * checks for the "touch" program across v2 mounts to
		 * Solaris and Irix boxes work correctly. See description of
		 * sattr in section 6.1 of "NFS Illustrated" by
		 * Brent Callaghan, Addison-Wesley, ISBN 0-201-32750-5
		 */
		if (tmp2 == 1000000)
			iap-&gt;ia_valid &amp;= ~(ATTR_ATIME_SET|ATTR_MTIME_SET);
	}

	return true;
}

/**
 * svcxdr_encode_fattr - Encode NFSv2 file attributes
 * @rqstp: Context of a completed RPC transaction
 * @xdr: XDR stream
 * @fhp: File handle to encode
 * @stat: Attributes to encode
 *
 * Return values:
 *   %false: Send buffer space was exhausted
 *   %true: Success
 */
bool
svcxdr_encode_fattr(struct svc_rqst *rqstp, struct xdr_stream *xdr,
		    const struct svc_fh *fhp, const struct kstat *stat)
{
	struct user_namespace *userns = nfsd_user_namespace(rqstp);
	struct dentry *dentry = fhp-&gt;fh_dentry;
	int type = stat-&gt;mode &amp; S_IFMT;
	struct timespec64 time;
	__be32 *p;
	u32 fsid;

	p = xdr_reserve_space(xdr, XDR_UNIT * 17);
	if (!p)
		return false;

	*p++ = cpu_to_be32(nfs_ftypes[type &gt;&gt; 12]);
	*p++ = cpu_to_be32((u32)stat-&gt;mode);
	*p++ = cpu_to_be32((u32)stat-&gt;nlink);
	*p++ = cpu_to_be32((u32)from_kuid_munged(userns, stat-&gt;uid));
	*p++ = cpu_to_be32((u32)from_kgid_munged(userns, stat-&gt;gid));

	if (S_ISLNK(type) &amp;&amp; stat-&gt;size &gt; NFS_MAXPATHLEN)
		*p++ = cpu_to_be32(NFS_MAXPATHLEN);
	else
		*p++ = cpu_to_be32((u32) stat-&gt;size);
	*p++ = cpu_to_be32((u32) stat-&gt;blksize);
	if (S_ISCHR(type) || S_ISBLK(type))
		*p++ = cpu_to_be32(new_encode_dev(stat-&gt;rdev));
	else
		*p++ = cpu_to_be32(0xffffffff);
	*p++ = cpu_to_be32((u32)stat-&gt;blocks);

	switch (fsid_source(fhp)) {
	case FSIDSOURCE_FSID:
		fsid = (u32)fhp-&gt;fh_export-&gt;ex_fsid;
		break;
	case FSIDSOURCE_UUID:
		fsid = ((u32 *)fhp-&gt;fh_export-&gt;ex_uuid)[0];
		fsid ^= ((u32 *)fhp-&gt;fh_export-&gt;ex_uuid)[1];
		fsid ^= ((u32 *)fhp-&gt;fh_export-&gt;ex_uuid)[2];
		fsid ^= ((u32 *)fhp-&gt;fh_export-&gt;ex_uuid)[3];
		break;
	default:
		fsid = new_encode_dev(stat-&gt;dev);
		break;
	}
	*p++ = cpu_to_be32(fsid);

	*p++ = cpu_to_be32((u32)stat-&gt;ino);
	p = encode_timeval(p, &amp;stat-&gt;atime);
	time = stat-&gt;mtime;
	lease_get_mtime(d_inode(dentry), &amp;time);
	p = encode_timeval(p, &amp;time);
	encode_timeval(p, &amp;stat-&gt;ctime);

	return true;
}

/*
 * XDR decode functions
 */

bool
nfssvc_decode_fhandleargs(struct svc_rqst *rqstp, struct xdr_stream *xdr)
{
	struct nfsd_fhandle *args = rqstp-&gt;rq_argp;

	return svcxdr_decode_fhandle(xdr, &amp;args-&gt;fh);
}

bool
nfssvc_decode_sattrargs(struct svc_rqst *rqstp, struct xdr_stream *xdr)
{
	struct nfsd_sattrargs *args = rqstp-&gt;rq_argp;

	return svcxdr_decode_fhandle(xdr, &amp;args-&gt;fh) &amp;&amp;
		svcxdr_decode_sattr(rqstp, xdr, &amp;args-&gt;attrs);
}

bool
nfssvc_decode_diropargs(struct svc_rqst *rqstp, struct xdr_stream *xdr)
{
	struct nfsd_diropargs *args = rqstp-&gt;rq_argp;

	return svcxdr_decode_diropargs(xdr, &amp;args-&gt;fh, &amp;args-&gt;name, &amp;args-&gt;len);
}

bool
nfssvc_decode_readargs(struct svc_rqst *rqstp, struct xdr_stream *xdr)
{
	struct nfsd_readargs *args = rqstp-&gt;rq_argp;
	u32 totalcount;

	if (!svcxdr_decode_fhandle(xdr, &amp;args-&gt;fh))
		return false;
	if (xdr_stream_decode_u32(xdr, &amp;args-&gt;offset) &lt; 0)
		return false;
	if (xdr_stream_decode_u32(xdr, &amp;args-&gt;count) &lt; 0)
		return false;
	/* totalcount is ignored */
	if (xdr_stream_decode_u32(xdr, &amp;totalcount) &lt; 0)
		return false;

	return true;
}

bool
nfssvc_decode_writeargs(struct svc_rqst *rqstp, struct xdr_stream *xdr)
{
	struct nfsd_writeargs *args = rqstp-&gt;rq_argp;
	u32 beginoffset, totalcount;

	if (!svcxdr_decode_fhandle(xdr, &amp;args-&gt;fh))
		return false;
	/* beginoffset is ignored */
	if (xdr_stream_decode_u32(xdr, &amp;beginoffset) &lt; 0)
		return false;
	if (xdr_stream_decode_u32(xdr, &amp;args-&gt;offset) &lt; 0)
		return false;
	/* totalcount is ignored */
	if (xdr_stream_decode_u32(xdr, &amp;totalcount) &lt; 0)
		return false;

	/* opaque data */
	if (xdr_stream_decode_u32(xdr, &amp;args-&gt;len) &lt; 0)
		return false;
	if (args-&gt;len &gt; NFSSVC_MAXBLKSIZE_V2)
		return false;

	return xdr_stream_subsegment(xdr, &amp;args-&gt;payload, args-&gt;len);
}

bool
nfssvc_decode_createargs(struct svc_rqst *rqstp, struct xdr_stream *xdr)
{
	struct nfsd_createargs *args = rqstp-&gt;rq_argp;

	return svcxdr_decode_diropargs(xdr, &amp;args-&gt;fh,
				       &amp;args-&gt;name, &amp;args-&gt;len) &amp;&amp;
		svcxdr_decode_sattr(rqstp, xdr, &amp;args-&gt;attrs);
}

bool
nfssvc_decode_renameargs(struct svc_rqst *rqstp, struct xdr_stream *xdr)
{
	struct nfsd_renameargs *args = rqstp-&gt;rq_argp;

	return svcxdr_decode_diropargs(xdr, &amp;args-&gt;ffh,
				       &amp;args-&gt;fname, &amp;args-&gt;flen) &amp;&amp;
		svcxdr_decode_diropargs(xdr, &amp;args-&gt;tfh,
					&amp;args-&gt;tname, &amp;args-&gt;tlen);
}

bool
nfssvc_decode_linkargs(struct svc_rqst *rqstp, struct xdr_stream *xdr)
{
	struct nfsd_linkargs *args = rqstp-&gt;rq_argp;

	return svcxdr_decode_fhandle(xdr, &amp;args-&gt;ffh) &amp;&amp;
		svcxdr_decode_diropargs(xdr, &amp;args-&gt;tfh,
					&amp;args-&gt;tname, &amp;args-&gt;tlen);
}

bool
nfssvc_decode_symlinkargs(struct svc_rqst *rqstp, struct xdr_stream *xdr)
{
	struct nfsd_symlinkargs *args = rqstp-&gt;rq_argp;
	struct kvec *head = rqstp-&gt;rq_arg.head;

	if (!svcxdr_decode_diropargs(xdr, &amp;args-&gt;ffh, &amp;args-&gt;fname, &amp;args-&gt;flen))
		return false;
	if (xdr_stream_decode_u32(xdr, &amp;args-&gt;tlen) &lt; 0)
		return false;
	if (args-&gt;tlen == 0)
		return false;

	args-&gt;first.iov_len = head-&gt;iov_len - xdr_stream_pos(xdr);
	args-&gt;first.iov_base = xdr_inline_decode(xdr, args-&gt;tlen);
	if (!args-&gt;first.iov_base)
		return false;
	return svcxdr_decode_sattr(rqstp, xdr, &amp;args-&gt;attrs);
}

bool
nfssvc_decode_readdirargs(struct svc_rqst *rqstp, struct xdr_stream *xdr)
{
	struct nfsd_readdirargs *args = rqstp-&gt;rq_argp;

	if (!svcxdr_decode_fhandle(xdr, &amp;args-&gt;fh))
		return false;
	if (xdr_stream_decode_u32(xdr, &amp;args-&gt;cookie) &lt; 0)
		return false;
	if (xdr_stream_decode_u32(xdr, &amp;args-&gt;count) &lt; 0)
		return false;

	return true;
}

/*
 * XDR encode functions
 */

bool
nfssvc_encode_statres(struct svc_rqst *rqstp, struct xdr_stream *xdr)
{
	struct nfsd_stat *resp = rqstp-&gt;rq_resp;

	return svcxdr_encode_stat(xdr, resp-&gt;status);
}

bool
nfssvc_encode_attrstatres(struct svc_rqst *rqstp, struct xdr_stream *xdr)
{
	struct nfsd_attrstat *resp = rqstp-&gt;rq_resp;

	if (!svcxdr_encode_stat(xdr, resp-&gt;status))
		return false;
	switch (resp-&gt;status) {
	case nfs_ok:
		if (!svcxdr_encode_fattr(rqstp, xdr, &amp;resp-&gt;fh, &amp;resp-&gt;stat))
			return false;
		break;
	}

	return true;
}

bool
nfssvc_encode_diropres(struct svc_rqst *rqstp, struct xdr_stream *xdr)
{
	struct nfsd_diropres *resp = rqstp-&gt;rq_resp;

	if (!svcxdr_encode_stat(xdr, resp-&gt;status))
		return false;
	switch (resp-&gt;status) {
	case nfs_ok:
		if (!svcxdr_encode_fhandle(xdr, &amp;resp-&gt;fh))
			return false;
		if (!svcxdr_encode_fattr(rqstp, xdr, &amp;resp-&gt;fh, &amp;resp-&gt;stat))
			return false;
		break;
	}

	return true;
}

bool
nfssvc_encode_readlinkres(struct svc_rqst *rqstp, struct xdr_stream *xdr)
{
	struct nfsd_readlinkres *resp = rqstp-&gt;rq_resp;
	struct kvec *head = rqstp-&gt;rq_res.head;

	if (!svcxdr_encode_stat(xdr, resp-&gt;status))
		return false;
	switch (resp-&gt;status) {
	case nfs_ok:
		if (xdr_stream_encode_u32(xdr, resp-&gt;len) &lt; 0)
			return false;
		xdr_write_pages(xdr, &amp;resp-&gt;page, 0, resp-&gt;len);
		if (svc_encode_result_payload(rqstp, head-&gt;iov_len, resp-&gt;len) &lt; 0)
			return false;
		break;
	}

	return true;
}

bool
nfssvc_encode_readres(struct svc_rqst *rqstp, struct xdr_stream *xdr)
{
	struct nfsd_readres *resp = rqstp-&gt;rq_resp;
	struct kvec *head = rqstp-&gt;rq_res.head;

	if (!svcxdr_encode_stat(xdr, resp-&gt;status))
		return false;
	switch (resp-&gt;status) {
	case nfs_ok:
		if (!svcxdr_encode_fattr(rqstp, xdr, &amp;resp-&gt;fh, &amp;resp-&gt;stat))
			return false;
		if (xdr_stream_encode_u32(xdr, resp-&gt;count) &lt; 0)
			return false;
		xdr_write_pages(xdr, resp-&gt;pages, rqstp-&gt;rq_res.page_base,
				resp-&gt;count);
		if (svc_encode_result_payload(rqstp, head-&gt;iov_len, resp-&gt;count) &lt; 0)
			return false;
		break;
	}

	return true;
}

bool
nfssvc_encode_readdirres(struct svc_rqst *rqstp, struct xdr_stream *xdr)
{
	struct nfsd_readdirres *resp = rqstp-&gt;rq_resp;
	struct xdr_buf *dirlist = &amp;resp-&gt;dirlist;

	if (!svcxdr_encode_stat(xdr, resp-&gt;status))
		return false;
	switch (resp-&gt;status) {
	case nfs_ok:
		xdr_write_pages(xdr, dirlist-&gt;pages, 0, dirlist-&gt;len);
		/* no more entries */
		if (xdr_stream_encode_item_absent(xdr) &lt; 0)
			return false;
		if (xdr_stream_encode_bool(xdr, resp-&gt;common.err == nfserr_eof) &lt; 0)
			return false;
		break;
	}

	return true;
}

bool
nfssvc_encode_statfsres(struct svc_rqst *rqstp, struct xdr_stream *xdr)
{
	struct nfsd_statfsres *resp = rqstp-&gt;rq_resp;
	struct kstatfs	*stat = &amp;resp-&gt;stats;
	__be32 *p;

	if (!svcxdr_encode_stat(xdr, resp-&gt;status))
		return false;
	switch (resp-&gt;status) {
	case nfs_ok:
		p = xdr_reserve_space(xdr, XDR_UNIT * 5);
		if (!p)
			return false;
		*p++ = cpu_to_be32(NFSSVC_MAXBLKSIZE_V2);
		*p++ = cpu_to_be32(stat-&gt;f_bsize);
		*p++ = cpu_to_be32(stat-&gt;f_blocks);
		*p++ = cpu_to_be32(stat-&gt;f_bfree);
		*p = cpu_to_be32(stat-&gt;f_bavail);
		break;
	}

	return true;
}

/**
 * nfssvc_encode_nfscookie - Encode a directory offset cookie
 * @resp: readdir result context
 * @offset: offset cookie to encode
 *
 * The buffer space for the offset cookie has already been reserved
 * by svcxdr_encode_entry_common().
 */
void nfssvc_encode_nfscookie(struct nfsd_readdirres *resp, u32 offset)
{
	__be32 cookie = cpu_to_be32(offset);

	if (!resp-&gt;cookie_offset)
		return;

	write_bytes_to_xdr_buf(&amp;resp-&gt;dirlist, resp-&gt;cookie_offset, &amp;cookie,
			       sizeof(cookie));
	resp-&gt;cookie_offset = 0;
}

static bool
svcxdr_encode_entry_common(struct nfsd_readdirres *resp, const char *name,
			   int namlen, loff_t offset, u64 ino)
{
	struct xdr_buf *dirlist = &amp;resp-&gt;dirlist;
	struct xdr_stream *xdr = &amp;resp-&gt;xdr;

	if (xdr_stream_encode_item_present(xdr) &lt; 0)
		return false;
	/* fileid */
	if (xdr_stream_encode_u32(xdr, (u32)ino) &lt; 0)
		return false;
	/* name */
	if (xdr_stream_encode_opaque(xdr, name, min(namlen, NFS2_MAXNAMLEN)) &lt; 0)
		return false;
	/* cookie */
	resp-&gt;cookie_offset = dirlist-&gt;len;
	if (xdr_stream_encode_u32(xdr, ~0U) &lt; 0)
		return false;

	return true;
}

/**
 * nfssvc_encode_entry - encode one NFSv2 READDIR entry
 * @data: directory context
 * @name: name of the object to be encoded
 * @namlen: length of that name, in bytes
 * @offset: the offset of the previous entry
 * @ino: the fileid of this entry
 * @d_type: unused
 *
 * Return values:
 *   %0: Entry was successfully encoded.
 *   %-EINVAL: An encoding problem occured, secondary status code in resp-&gt;common.err
 *
 * On exit, the following fields are updated:
 *   - resp-&gt;xdr
 *   - resp-&gt;common.err
 *   - resp-&gt;cookie_offset
 */
int nfssvc_encode_entry(void *data, const char *name, int namlen,
			loff_t offset, u64 ino, unsigned int d_type)
{
	struct readdir_cd *ccd = data;
	struct nfsd_readdirres *resp = container_of(ccd,
						    struct nfsd_readdirres,
						    common);
	unsigned int starting_length = resp-&gt;dirlist.len;

	/* The offset cookie for the previous entry */
	nfssvc_encode_nfscookie(resp, offset);

	if (!svcxdr_encode_entry_common(resp, name, namlen, offset, ino))
		goto out_toosmall;

	xdr_commit_encode(&amp;resp-&gt;xdr);
	resp-&gt;common.err = nfs_ok;
	return 0;

out_toosmall:
	resp-&gt;cookie_offset = 0;
	resp-&gt;common.err = nfserr_toosmall;
	resp-&gt;dirlist.len = starting_length;
	return -EINVAL;
}

/*
 * XDR release functions
 */
void nfssvc_release_attrstat(struct svc_rqst *rqstp)
{
	struct nfsd_attrstat *resp = rqstp-&gt;rq_resp;

	fh_put(&amp;resp-&gt;fh);
}

void nfssvc_release_diropres(struct svc_rqst *rqstp)
{
	struct nfsd_diropres *resp = rqstp-&gt;rq_resp;

	fh_put(&amp;resp-&gt;fh);
}

void nfssvc_release_readres(struct svc_rqst *rqstp)
{
	struct nfsd_readres *resp = rqstp-&gt;rq_resp;

	fh_put(&amp;resp-&gt;fh);
}
</okir@monad.swb.de>