// SPDX-License-Identifier: GPL-2.0-or-later
/* Provide a way to create a superblock configuration context within the kernel
 * that allows a superblock to be set up prior to mounting.
 *
 * Copyright (C) 2017 Red Hat, Inc. All Rights Reserved.
 * Written by David Howells (dhowells@redhat.com)
 */

#define pr_fmt(fmt) KBUILD_MODNAME ": " fmt
#include <linux module.h="">
#include <linux fs_context.h="">
#include <linux fs_parser.h="">
#include <linux fs.h="">
#include <linux mount.h="">
#include <linux nsproxy.h="">
#include <linux slab.h="">
#include <linux magic.h="">
#include <linux security.h="">
#include <linux mnt_namespace.h="">
#include <linux pid_namespace.h="">
#include <linux user_namespace.h="">
#include <net net_namespace.h="">
#include <asm sections.h="">
#include "mount.h"
#include "internal.h"

enum legacy_fs_param {
	LEGACY_FS_UNSET_PARAMS,
	LEGACY_FS_MONOLITHIC_PARAMS,
	LEGACY_FS_INDIVIDUAL_PARAMS,
};

struct legacy_fs_context {
	char			*legacy_data;	/* Data page for legacy filesystems */
	size_t			data_size;
	enum legacy_fs_param	param_type;
};

static int legacy_init_fs_context(struct fs_context *fc);

static const struct constant_table common_set_sb_flag[] = {
	{ "dirsync",	SB_DIRSYNC },
	{ "lazytime",	SB_LAZYTIME },
	{ "mand",	SB_MANDLOCK },
	{ "ro",		SB_RDONLY },
	{ "sync",	SB_SYNCHRONOUS },
	{ },
};

static const struct constant_table common_clear_sb_flag[] = {
	{ "async",	SB_SYNCHRONOUS },
	{ "nolazytime",	SB_LAZYTIME },
	{ "nomand",	SB_MANDLOCK },
	{ "rw",		SB_RDONLY },
	{ },
};

/*
 * Check for a common mount option that manipulates s_flags.
 */
static int vfs_parse_sb_flag(struct fs_context *fc, const char *key)
{
	unsigned int token;

	token = lookup_constant(common_set_sb_flag, key, 0);
	if (token) {
		fc-&gt;sb_flags |= token;
		fc-&gt;sb_flags_mask |= token;
		return 0;
	}

	token = lookup_constant(common_clear_sb_flag, key, 0);
	if (token) {
		fc-&gt;sb_flags &amp;= ~token;
		fc-&gt;sb_flags_mask |= token;
		return 0;
	}

	return -ENOPARAM;
}

/**
 * vfs_parse_fs_param_source - Handle setting "source" via parameter
 * @fc: The filesystem context to modify
 * @param: The parameter
 *
 * This is a simple helper for filesystems to verify that the "source" they
 * accept is sane.
 *
 * Returns 0 on success, -ENOPARAM if this is not  "source" parameter, and
 * -EINVAL otherwise. In the event of failure, supplementary error information
 *  is logged.
 */
int vfs_parse_fs_param_source(struct fs_context *fc, struct fs_parameter *param)
{
	if (strcmp(param-&gt;key, "source") != 0)
		return -ENOPARAM;

	if (param-&gt;type != fs_value_is_string)
		return invalf(fc, "Non-string source");

	if (fc-&gt;source)
		return invalf(fc, "Multiple sources");

	fc-&gt;source = param-&gt;string;
	param-&gt;string = NULL;
	return 0;
}
EXPORT_SYMBOL(vfs_parse_fs_param_source);

/**
 * vfs_parse_fs_param - Add a single parameter to a superblock config
 * @fc: The filesystem context to modify
 * @param: The parameter
 *
 * A single mount option in string form is applied to the filesystem context
 * being set up.  Certain standard options (for example "ro") are translated
 * into flag bits without going to the filesystem.  The active security module
 * is allowed to observe and poach options.  Any other options are passed over
 * to the filesystem to parse.
 *
 * This may be called multiple times for a context.
 *
 * Returns 0 on success and a negative error code on failure.  In the event of
 * failure, supplementary error information may have been set.
 */
int vfs_parse_fs_param(struct fs_context *fc, struct fs_parameter *param)
{
	int ret;

	if (!param-&gt;key)
		return invalf(fc, "Unnamed parameter\n");

	ret = vfs_parse_sb_flag(fc, param-&gt;key);
	if (ret != -ENOPARAM)
		return ret;

	ret = security_fs_context_parse_param(fc, param);
	if (ret != -ENOPARAM)
		/* Param belongs to the LSM or is disallowed by the LSM; so
		 * don't pass to the FS.
		 */
		return ret;

	if (fc-&gt;ops-&gt;parse_param) {
		ret = fc-&gt;ops-&gt;parse_param(fc, param);
		if (ret != -ENOPARAM)
			return ret;
	}

	/* If the filesystem doesn't take any arguments, give it the
	 * default handling of source.
	 */
	ret = vfs_parse_fs_param_source(fc, param);
	if (ret != -ENOPARAM)
		return ret;

	return invalf(fc, "%s: Unknown parameter '%s'",
		      fc-&gt;fs_type-&gt;name, param-&gt;key);
}
EXPORT_SYMBOL(vfs_parse_fs_param);

/**
 * vfs_parse_fs_string - Convenience function to just parse a string.
 */
int vfs_parse_fs_string(struct fs_context *fc, const char *key,
			const char *value, size_t v_size)
{
	int ret;

	struct fs_parameter param = {
		.key	= key,
		.type	= fs_value_is_flag,
		.size	= v_size,
	};

	if (value) {
		param.string = kmemdup_nul(value, v_size, GFP_KERNEL);
		if (!param.string)
			return -ENOMEM;
		param.type = fs_value_is_string;
	}

	ret = vfs_parse_fs_param(fc, &amp;param);
	kfree(param.string);
	return ret;
}
EXPORT_SYMBOL(vfs_parse_fs_string);

/**
 * generic_parse_monolithic - Parse key[=val][,key[=val]]* mount data
 * @ctx: The superblock configuration to fill in.
 * @data: The data to parse
 *
 * Parse a blob of data that's in key[=val][,key[=val]]* form.  This can be
 * called from the -&gt;monolithic_mount_data() fs_context operation.
 *
 * Returns 0 on success or the error returned by the -&gt;parse_option() fs_context
 * operation on failure.
 */
int generic_parse_monolithic(struct fs_context *fc, void *data)
{
	char *options = data, *key;
	int ret = 0;

	if (!options)
		return 0;

	ret = security_sb_eat_lsm_opts(options, &amp;fc-&gt;security);
	if (ret)
		return ret;

	while ((key = strsep(&amp;options, ",")) != NULL) {
		if (*key) {
			size_t v_len = 0;
			char *value = strchr(key, '=');

			if (value) {
				if (value == key)
					continue;
				*value++ = 0;
				v_len = strlen(value);
			}
			ret = vfs_parse_fs_string(fc, key, value, v_len);
			if (ret &lt; 0)
				break;
		}
	}

	return ret;
}
EXPORT_SYMBOL(generic_parse_monolithic);

/**
 * alloc_fs_context - Create a filesystem context.
 * @fs_type: The filesystem type.
 * @reference: The dentry from which this one derives (or NULL)
 * @sb_flags: Filesystem/superblock flags (SB_*)
 * @sb_flags_mask: Applicable members of @sb_flags
 * @purpose: The purpose that this configuration shall be used for.
 *
 * Open a filesystem and create a mount context.  The mount context is
 * initialised with the supplied flags and, if a submount/automount from
 * another superblock (referred to by @reference) is supplied, may have
 * parameters such as namespaces copied across from that superblock.
 */
static struct fs_context *alloc_fs_context(struct file_system_type *fs_type,
				      struct dentry *reference,
				      unsigned int sb_flags,
				      unsigned int sb_flags_mask,
				      enum fs_context_purpose purpose)
{
	int (*init_fs_context)(struct fs_context *);
	struct fs_context *fc;
	int ret = -ENOMEM;

	fc = kzalloc(sizeof(struct fs_context), GFP_KERNEL_ACCOUNT);
	if (!fc)
		return ERR_PTR(-ENOMEM);

	fc-&gt;purpose	= purpose;
	fc-&gt;sb_flags	= sb_flags;
	fc-&gt;sb_flags_mask = sb_flags_mask;
	fc-&gt;fs_type	= get_filesystem(fs_type);
	fc-&gt;cred	= get_current_cred();
	fc-&gt;net_ns	= get_net(current-&gt;nsproxy-&gt;net_ns);
	fc-&gt;log.prefix	= fs_type-&gt;name;

	mutex_init(&amp;fc-&gt;uapi_mutex);

	switch (purpose) {
	case FS_CONTEXT_FOR_MOUNT:
		fc-&gt;user_ns = get_user_ns(fc-&gt;cred-&gt;user_ns);
		break;
	case FS_CONTEXT_FOR_SUBMOUNT:
		fc-&gt;user_ns = get_user_ns(reference-&gt;d_sb-&gt;s_user_ns);
		break;
	case FS_CONTEXT_FOR_RECONFIGURE:
		atomic_inc(&amp;reference-&gt;d_sb-&gt;s_active);
		fc-&gt;user_ns = get_user_ns(reference-&gt;d_sb-&gt;s_user_ns);
		fc-&gt;root = dget(reference);
		break;
	}

	/* TODO: Make all filesystems support this unconditionally */
	init_fs_context = fc-&gt;fs_type-&gt;init_fs_context;
	if (!init_fs_context)
		init_fs_context = legacy_init_fs_context;

	ret = init_fs_context(fc);
	if (ret &lt; 0)
		goto err_fc;
	fc-&gt;need_free = true;
	return fc;

err_fc:
	put_fs_context(fc);
	return ERR_PTR(ret);
}

struct fs_context *fs_context_for_mount(struct file_system_type *fs_type,
					unsigned int sb_flags)
{
	return alloc_fs_context(fs_type, NULL, sb_flags, 0,
					FS_CONTEXT_FOR_MOUNT);
}
EXPORT_SYMBOL(fs_context_for_mount);

struct fs_context *fs_context_for_reconfigure(struct dentry *dentry,
					unsigned int sb_flags,
					unsigned int sb_flags_mask)
{
	return alloc_fs_context(dentry-&gt;d_sb-&gt;s_type, dentry, sb_flags,
				sb_flags_mask, FS_CONTEXT_FOR_RECONFIGURE);
}
EXPORT_SYMBOL(fs_context_for_reconfigure);

struct fs_context *fs_context_for_submount(struct file_system_type *type,
					   struct dentry *reference)
{
	return alloc_fs_context(type, reference, 0, 0, FS_CONTEXT_FOR_SUBMOUNT);
}
EXPORT_SYMBOL(fs_context_for_submount);

void fc_drop_locked(struct fs_context *fc)
{
	struct super_block *sb = fc-&gt;root-&gt;d_sb;
	dput(fc-&gt;root);
	fc-&gt;root = NULL;
	deactivate_locked_super(sb);
}

static void legacy_fs_context_free(struct fs_context *fc);

/**
 * vfs_dup_fc_config: Duplicate a filesystem context.
 * @src_fc: The context to copy.
 */
struct fs_context *vfs_dup_fs_context(struct fs_context *src_fc)
{
	struct fs_context *fc;
	int ret;

	if (!src_fc-&gt;ops-&gt;dup)
		return ERR_PTR(-EOPNOTSUPP);

	fc = kmemdup(src_fc, sizeof(struct fs_context), GFP_KERNEL);
	if (!fc)
		return ERR_PTR(-ENOMEM);

	mutex_init(&amp;fc-&gt;uapi_mutex);

	fc-&gt;fs_private	= NULL;
	fc-&gt;s_fs_info	= NULL;
	fc-&gt;source	= NULL;
	fc-&gt;security	= NULL;
	get_filesystem(fc-&gt;fs_type);
	get_net(fc-&gt;net_ns);
	get_user_ns(fc-&gt;user_ns);
	get_cred(fc-&gt;cred);
	if (fc-&gt;log.log)
		refcount_inc(&amp;fc-&gt;log.log-&gt;usage);

	/* Can't call put until we've called -&gt;dup */
	ret = fc-&gt;ops-&gt;dup(fc, src_fc);
	if (ret &lt; 0)
		goto err_fc;

	ret = security_fs_context_dup(fc, src_fc);
	if (ret &lt; 0)
		goto err_fc;
	return fc;

err_fc:
	put_fs_context(fc);
	return ERR_PTR(ret);
}
EXPORT_SYMBOL(vfs_dup_fs_context);

/**
 * logfc - Log a message to a filesystem context
 * @fc: The filesystem context to log to.
 * @fmt: The format of the buffer.
 */
void logfc(struct fc_log *log, const char *prefix, char level, const char *fmt, ...)
{
	va_list va;
	struct va_format vaf = {.fmt = fmt, .va = &amp;va};

	va_start(va, fmt);
	if (!log) {
		switch (level) {
		case 'w':
			printk(KERN_WARNING "%s%s%pV\n", prefix ? prefix : "",
						prefix ? ": " : "", &amp;vaf);
			break;
		case 'e':
			printk(KERN_ERR "%s%s%pV\n", prefix ? prefix : "",
						prefix ? ": " : "", &amp;vaf);
			break;
		default:
			printk(KERN_NOTICE "%s%s%pV\n", prefix ? prefix : "",
						prefix ? ": " : "", &amp;vaf);
			break;
		}
	} else {
		unsigned int logsize = ARRAY_SIZE(log-&gt;buffer);
		u8 index;
		char *q = kasprintf(GFP_KERNEL, "%c %s%s%pV\n", level,
						prefix ? prefix : "",
						prefix ? ": " : "", &amp;vaf);

		index = log-&gt;head &amp; (logsize - 1);
		BUILD_BUG_ON(sizeof(log-&gt;head) != sizeof(u8) ||
			     sizeof(log-&gt;tail) != sizeof(u8));
		if ((u8)(log-&gt;head - log-&gt;tail) == logsize) {
			/* The buffer is full, discard the oldest message */
			if (log-&gt;need_free &amp; (1 &lt;&lt; index))
				kfree(log-&gt;buffer[index]);
			log-&gt;tail++;
		}

		log-&gt;buffer[index] = q ? q : "OOM: Can't store error string";
		if (q)
			log-&gt;need_free |= 1 &lt;&lt; index;
		else
			log-&gt;need_free &amp;= ~(1 &lt;&lt; index);
		log-&gt;head++;
	}
	va_end(va);
}
EXPORT_SYMBOL(logfc);

/*
 * Free a logging structure.
 */
static void put_fc_log(struct fs_context *fc)
{
	struct fc_log *log = fc-&gt;log.log;
	int i;

	if (log) {
		if (refcount_dec_and_test(&amp;log-&gt;usage)) {
			fc-&gt;log.log = NULL;
			for (i = 0; i &lt;= 7; i++)
				if (log-&gt;need_free &amp; (1 &lt;&lt; i))
					kfree(log-&gt;buffer[i]);
			kfree(log);
		}
	}
}

/**
 * put_fs_context - Dispose of a superblock configuration context.
 * @fc: The context to dispose of.
 */
void put_fs_context(struct fs_context *fc)
{
	struct super_block *sb;

	if (fc-&gt;root) {
		sb = fc-&gt;root-&gt;d_sb;
		dput(fc-&gt;root);
		fc-&gt;root = NULL;
		deactivate_super(sb);
	}

	if (fc-&gt;need_free &amp;&amp; fc-&gt;ops &amp;&amp; fc-&gt;ops-&gt;free)
		fc-&gt;ops-&gt;free(fc);

	security_free_mnt_opts(&amp;fc-&gt;security);
	put_net(fc-&gt;net_ns);
	put_user_ns(fc-&gt;user_ns);
	put_cred(fc-&gt;cred);
	put_fc_log(fc);
	put_filesystem(fc-&gt;fs_type);
	kfree(fc-&gt;source);
	kfree(fc);
}
EXPORT_SYMBOL(put_fs_context);

/*
 * Free the config for a filesystem that doesn't support fs_context.
 */
static void legacy_fs_context_free(struct fs_context *fc)
{
	struct legacy_fs_context *ctx = fc-&gt;fs_private;

	if (ctx) {
		if (ctx-&gt;param_type == LEGACY_FS_INDIVIDUAL_PARAMS)
			kfree(ctx-&gt;legacy_data);
		kfree(ctx);
	}
}

/*
 * Duplicate a legacy config.
 */
static int legacy_fs_context_dup(struct fs_context *fc, struct fs_context *src_fc)
{
	struct legacy_fs_context *ctx;
	struct legacy_fs_context *src_ctx = src_fc-&gt;fs_private;

	ctx = kmemdup(src_ctx, sizeof(*src_ctx), GFP_KERNEL);
	if (!ctx)
		return -ENOMEM;

	if (ctx-&gt;param_type == LEGACY_FS_INDIVIDUAL_PARAMS) {
		ctx-&gt;legacy_data = kmemdup(src_ctx-&gt;legacy_data,
					   src_ctx-&gt;data_size, GFP_KERNEL);
		if (!ctx-&gt;legacy_data) {
			kfree(ctx);
			return -ENOMEM;
		}
	}

	fc-&gt;fs_private = ctx;
	return 0;
}

/*
 * Add a parameter to a legacy config.  We build up a comma-separated list of
 * options.
 */
static int legacy_parse_param(struct fs_context *fc, struct fs_parameter *param)
{
	struct legacy_fs_context *ctx = fc-&gt;fs_private;
	unsigned int size = ctx-&gt;data_size;
	size_t len = 0;
	int ret;

	ret = vfs_parse_fs_param_source(fc, param);
	if (ret != -ENOPARAM)
		return ret;

	if (ctx-&gt;param_type == LEGACY_FS_MONOLITHIC_PARAMS)
		return invalf(fc, "VFS: Legacy: Can't mix monolithic and individual options");

	switch (param-&gt;type) {
	case fs_value_is_string:
		len = 1 + param-&gt;size;
		fallthrough;
	case fs_value_is_flag:
		len += strlen(param-&gt;key);
		break;
	default:
		return invalf(fc, "VFS: Legacy: Parameter type for '%s' not supported",
			      param-&gt;key);
	}

	if (size + len + 2 &gt; PAGE_SIZE)
		return invalf(fc, "VFS: Legacy: Cumulative options too large");
	if (strchr(param-&gt;key, ',') ||
	    (param-&gt;type == fs_value_is_string &amp;&amp;
	     memchr(param-&gt;string, ',', param-&gt;size)))
		return invalf(fc, "VFS: Legacy: Option '%s' contained comma",
			      param-&gt;key);
	if (!ctx-&gt;legacy_data) {
		ctx-&gt;legacy_data = kmalloc(PAGE_SIZE, GFP_KERNEL);
		if (!ctx-&gt;legacy_data)
			return -ENOMEM;
	}

	ctx-&gt;legacy_data[size++] = ',';
	len = strlen(param-&gt;key);
	memcpy(ctx-&gt;legacy_data + size, param-&gt;key, len);
	size += len;
	if (param-&gt;type == fs_value_is_string) {
		ctx-&gt;legacy_data[size++] = '=';
		memcpy(ctx-&gt;legacy_data + size, param-&gt;string, param-&gt;size);
		size += param-&gt;size;
	}
	ctx-&gt;legacy_data[size] = '\0';
	ctx-&gt;data_size = size;
	ctx-&gt;param_type = LEGACY_FS_INDIVIDUAL_PARAMS;
	return 0;
}

/*
 * Add monolithic mount data.
 */
static int legacy_parse_monolithic(struct fs_context *fc, void *data)
{
	struct legacy_fs_context *ctx = fc-&gt;fs_private;

	if (ctx-&gt;param_type != LEGACY_FS_UNSET_PARAMS) {
		pr_warn("VFS: Can't mix monolithic and individual options\n");
		return -EINVAL;
	}

	ctx-&gt;legacy_data = data;
	ctx-&gt;param_type = LEGACY_FS_MONOLITHIC_PARAMS;
	if (!ctx-&gt;legacy_data)
		return 0;

	if (fc-&gt;fs_type-&gt;fs_flags &amp; FS_BINARY_MOUNTDATA)
		return 0;
	return security_sb_eat_lsm_opts(ctx-&gt;legacy_data, &amp;fc-&gt;security);
}

/*
 * Get a mountable root with the legacy mount command.
 */
static int legacy_get_tree(struct fs_context *fc)
{
	struct legacy_fs_context *ctx = fc-&gt;fs_private;
	struct super_block *sb;
	struct dentry *root;

	root = fc-&gt;fs_type-&gt;mount(fc-&gt;fs_type, fc-&gt;sb_flags,
				      fc-&gt;source, ctx-&gt;legacy_data);
	if (IS_ERR(root))
		return PTR_ERR(root);

	sb = root-&gt;d_sb;
	BUG_ON(!sb);

	fc-&gt;root = root;
	return 0;
}

/*
 * Handle remount.
 */
static int legacy_reconfigure(struct fs_context *fc)
{
	struct legacy_fs_context *ctx = fc-&gt;fs_private;
	struct super_block *sb = fc-&gt;root-&gt;d_sb;

	if (!sb-&gt;s_op-&gt;remount_fs)
		return 0;

	return sb-&gt;s_op-&gt;remount_fs(sb, &amp;fc-&gt;sb_flags,
				    ctx ? ctx-&gt;legacy_data : NULL);
}

const struct fs_context_operations legacy_fs_context_ops = {
	.free			= legacy_fs_context_free,
	.dup			= legacy_fs_context_dup,
	.parse_param		= legacy_parse_param,
	.parse_monolithic	= legacy_parse_monolithic,
	.get_tree		= legacy_get_tree,
	.reconfigure		= legacy_reconfigure,
};

/*
 * Initialise a legacy context for a filesystem that doesn't support
 * fs_context.
 */
static int legacy_init_fs_context(struct fs_context *fc)
{
	fc-&gt;fs_private = kzalloc(sizeof(struct legacy_fs_context), GFP_KERNEL_ACCOUNT);
	if (!fc-&gt;fs_private)
		return -ENOMEM;
	fc-&gt;ops = &amp;legacy_fs_context_ops;
	return 0;
}

int parse_monolithic_mount_data(struct fs_context *fc, void *data)
{
	int (*monolithic_mount_data)(struct fs_context *, void *);

	monolithic_mount_data = fc-&gt;ops-&gt;parse_monolithic;
	if (!monolithic_mount_data)
		monolithic_mount_data = generic_parse_monolithic;

	return monolithic_mount_data(fc, data);
}

/*
 * Clean up a context after performing an action on it and put it into a state
 * from where it can be used to reconfigure a superblock.
 *
 * Note that here we do only the parts that can't fail; the rest is in
 * finish_clean_context() below and in between those fs_context is marked
 * FS_CONTEXT_AWAITING_RECONF.  The reason for splitup is that after
 * successful mount or remount we need to report success to userland.
 * Trying to do full reinit (for the sake of possible subsequent remount)
 * and failing to allocate memory would've put us into a nasty situation.
 * So here we only discard the old state and reinitialization is left
 * until we actually try to reconfigure.
 */
void vfs_clean_context(struct fs_context *fc)
{
	if (fc-&gt;need_free &amp;&amp; fc-&gt;ops &amp;&amp; fc-&gt;ops-&gt;free)
		fc-&gt;ops-&gt;free(fc);
	fc-&gt;need_free = false;
	fc-&gt;fs_private = NULL;
	fc-&gt;s_fs_info = NULL;
	fc-&gt;sb_flags = 0;
	security_free_mnt_opts(&amp;fc-&gt;security);
	kfree(fc-&gt;source);
	fc-&gt;source = NULL;

	fc-&gt;purpose = FS_CONTEXT_FOR_RECONFIGURE;
	fc-&gt;phase = FS_CONTEXT_AWAITING_RECONF;
}

int finish_clean_context(struct fs_context *fc)
{
	int error;

	if (fc-&gt;phase != FS_CONTEXT_AWAITING_RECONF)
		return 0;

	if (fc-&gt;fs_type-&gt;init_fs_context)
		error = fc-&gt;fs_type-&gt;init_fs_context(fc);
	else
		error = legacy_init_fs_context(fc);
	if (unlikely(error)) {
		fc-&gt;phase = FS_CONTEXT_FAILED;
		return error;
	}
	fc-&gt;need_free = true;
	fc-&gt;phase = FS_CONTEXT_RECONF_PARAMS;
	return 0;
}
</asm></net></linux></linux></linux></linux></linux></linux></linux></linux></linux></linux></linux></linux>