"""
Modern ShapeGeneratorAgent using the unified BaseAgent system
Generates Three.js code from object descriptions
"""

import os
import re
import json
import time
import logging
from typing import Dict, Any, Optional
from pydantic import BaseModel
from agents.base_agent import BaseAgent
from utils.output_parser import OutputFormatError
from utils.simple_colors import success, progress, error, warning, stats


class ThreeJSCode(BaseModel):
    """Data model for Three.js export code output"""
    js_code_export: str = None
    
    @classmethod
    def extract_from_response(cls, text: str) -> "ThreeJSCode":
        """
        Extract js_code_export from LLM response
        
        Args:
            text: Raw LLM response
            
        Returns:
            ThreeJSCode instance with extracted export code
            
        Raises:
            OutputFormatError: If js_code_export block is missing
        """
        # Primary pattern for js_code_export
        export_match = re.search(r'<js_code_export>(.*?)</js_code_export>', text, re.DOTALL)
        
        if not export_match:
            # Log the response for debugging
            import logging
            logging.error(f"Failed to find <js_code_export> tags in response. Response preview: {text[:500]}...")
            raise OutputFormatError("Missing required code block: js_code_export")
        
        js_code_export = export_match.group(1).strip()
        
        if not cls._validate_code_block(js_code_export, "javascript"):
            raise OutputFormatError("Export JavaScript code block is invalid or empty")
            
        return cls(js_code_export=js_code_export)
    
    @staticmethod
    def _validate_code_block(code: str, code_type: str) -> bool:
        """Validate that JavaScript code block contains meaningful content"""
        if not code or not code.strip():
            return False
        
        # Basic validation for JavaScript
        if code_type == "javascript":
            return len(code.strip()) > 10  # Basic length check
        
        return True


class ShapeGeneratorAgent(BaseAgent):
    """
    Agent for generating Three.js export code from object descriptions.
    
    Uses the unified BaseAgent system for consistent LLM interaction.
    Only generates export.js code for mesh generation pipeline.
    """
    
    def __init__(self, config_manager, verbose_samples: bool = True):
        """
        Initialize the ShapeGeneratorAgent (export-only mode)
        
        Args:
            config_manager: Configuration manager instance
            verbose_samples: If True, include verbose samples in prompt
        """
        # Set attributes before calling super() since _load_system_prompt uses them
        self.verbose_samples = verbose_samples
        self.logger = logging.getLogger(self.__class__.__name__)
        
        # Now call parent initializer
        super().__init__(config_manager, 'shape_generator')
        
    def _load_system_prompt(self) -> str:
        """Load system prompt for Three.js export code generation"""
        try:
            # Always use export-only prompt
            from prompt.parter import system_prompt as export_only_prompt
            base_prompt = export_only_prompt
            
            # Add samples if requested
            if self.verbose_samples:
                from prompt.examples.parter_samples import samples as parter_samples
                return base_prompt + '\n' + parter_samples
            else:
                return base_prompt
                
        except ImportError as e:
            self.logger.error(f"Failed to load system prompt: {e}")
            return "Generate Three.js export code for the given object description."
    
    def _format_user_prompt(self, input_data: Dict[str, Any]) -> str:
        """
        Format user prompt for code generation
        
        Args:
            input_data: Dictionary containing description and optional links_json
            
        Returns:
            Formatted user prompt
        """
        description = input_data.get('articulated_object', '')
        links_json = input_data.get('links_json')
        retry_context = input_data.get('_retry_context', '')
        
        # Base prompt
        if links_json is not None:
            # Include links information in the prompt
            links_json_str = json.dumps(links_json, ensure_ascii=False, indent=2)
            # Escape braces for string formatting
            links_json_str = links_json_str.replace('{', '{{').replace('}', '}}')
            
            base_prompt = (
                f"Now please generate the codes of an articulated object.\n\n"
                f"The description of the object is: {description}\n\n"
                f"The links_json is:\n{links_json_str}\n\n"
                f"Please generate the Three.js code that creates this articulated object "
                f"based on the description and the provided link information."
            )
        else:
            base_prompt = (
                f"Now please generate the codes of an object.\n"
                f"The description of the object is: {description}"
            )
        
        # Add retry context if this is a retry attempt
        if retry_context:
            base_prompt += (
                f"\n\nRETRY ATTEMPT - IMPORTANT FEEDBACK:\n"
                f"{retry_context}\n\n"
                f"Please carefully address the above issue and generate corrected code."
            )
        
        return base_prompt
    
    def parse_response(self, response: str) -> ThreeJSCode:
        """
        Parse LLM response into structured Three.js code
        
        Args:
            response: Raw LLM response
            
        Returns:
            ThreeJSCode instance with parsed export code
            
        Raises:
            OutputFormatError: If response format is invalid
        """
        try:
            return ThreeJSCode.extract_from_response(response)
        except OutputFormatError as e:
            self.logger.error(f"Failed to parse shape response: {e}")
            raise
    
    def _prepare_input_data(self, articulated_object: str = None,
                           links_json: Any = None, **kwargs) -> Dict[str, Any]:
        """
        Prepare input data from method arguments
        
        Args:
            articulated_object: Object description
            links_json: Optional links information
            **kwargs: Additional arguments
            
        Returns:
            Dictionary of input data for the agent
        """
        obj_description = articulated_object or ""
        
        input_data = {'articulated_object': obj_description}
        if links_json is not None:
            input_data['links_json'] = links_json
            
        return input_data
    
    def _save_output_with_input(self, result: ThreeJSCode, output_folder: str, 
                               input_data: Dict[str, Any], metrics: Dict[str, Any] = None):
        """
        Save output with access to original input data
        
        Args:
            result: Generated Three.js export code
            output_folder: Directory to save output
            input_data: Original input data containing articulated_object description
            metrics: Generation metrics (optional)
        """
        # Extract description from input data
        description = input_data.get('articulated_object', '')
        
        # Call the original save_output method with description
        self.save_output(result, output_folder, description, metrics)
    
    def save_output(self, result: ThreeJSCode, output_folder: str, 
                   description: str = "", metrics: Dict[str, Any] = None):
        """
        Save generated Three.js export code to _export_temp folder
        
        Args:
            result: Generated Three.js export code
            output_folder: Directory to save output
            description: Input description for workflow.json
            metrics: Generation metrics (optional)
        """
        # Create _export_temp folder for export.js
        export_temp_folder = os.path.join(output_folder, '_export_temp')
        os.makedirs(export_temp_folder, exist_ok=True)
        
        # Create configs folder for workflow.json
        configs_folder = os.path.join(output_folder, 'configs')
        os.makedirs(configs_folder, exist_ok=True)
        
        # Save export.js file to _export_temp
        if result.js_code_export:
            export_path = os.path.join(export_temp_folder, 'export.js')
            with open(export_path, 'w', encoding='utf-8') as f:
                f.write(result.js_code_export)
            self.logger.info(f"Saved export code to {export_path}")
        
        # Save workflow.json to configs folder
        workflow_json_path = os.path.join(configs_folder, 'workflow.json')
        with open(workflow_json_path, 'w', encoding='utf-8') as f:
            json.dump({"raw_input_text": description}, f, ensure_ascii=False, indent=2)
        self.logger.info(f"Saved workflow.json to {workflow_json_path}")
        
        # Note: Metadata is now saved in pipeline_logs by BaseAgent
    
    def generate_with_export_validation(self, max_retries: int = 3, **kwargs) -> tuple:
        """
        Generate Three.js code with export validation and retry mechanism.
        
        This method generates code, tests if it can be exported to mesh format,
        and retries up to max_retries times if export fails.
        
        Args:
            max_retries: Maximum number of retry attempts (default: 3)
            **kwargs: Arguments passed to generate method
            
        Returns:
            Tuple of (result, success, metrics, raw_response, retry_count)
        """
        from utils.mesh_exporter import MeshExporter
        import shutil
        
        # Get output folder from kwargs
        output_folder = kwargs.get('output_folder')
        if not output_folder:
            raise ValueError("output_folder must be provided in kwargs for export validation")
            
        for attempt in range(max_retries + 1):  # +1 for initial attempt
            attempt_num = attempt + 1
            self.logger.info(f"Shape generation attempt {attempt_num}/{max_retries + 1}")
            
            # Generate code
            result, success, metrics, raw_response = self.generate(**kwargs)
            
            if not success or not result or not result.js_code_export:
                self.logger.warning(f"Generation attempt {attempt_num} failed - no code produced")
                if attempt < max_retries:
                    continue
                else:
                    return result, False, metrics, raw_response, attempt_num
            
            # Test export validation
            temp_dir = None
            try:
                # Create temporary directory for testing within output folder
                temp_dir_name = f'shape_export_test_{attempt_num}_{time.time():.0f}'
                temp_dir = os.path.join(output_folder, temp_dir_name)
                temp_export_dir = os.path.join(temp_dir, '_export_temp')
                os.makedirs(temp_export_dir, exist_ok=True)
                
                # Save generated code to temp file in correct structure
                export_path = os.path.join(temp_export_dir, 'export.js')
                with open(export_path, 'w', encoding='utf-8') as f:
                    f.write(result.js_code_export)
                
                # Test export silently (suppress verbose output during validation)
                import sys
                from io import StringIO
                
                # Temporarily capture stdout to suppress verbose mesh loading output
                old_stdout = sys.stdout
                sys.stdout = StringIO()
                
                try:
                    exporter = MeshExporter(temp_dir, temp_dir)
                    final_mesh, export_success, error_msg = exporter.export_to_obj()
                finally:
                    # Always restore stdout
                    sys.stdout = old_stdout
                
                if export_success:
                    self.logger.info(f"Shape generation attempt {attempt_num} succeeded with valid export")
                    return result, True, metrics, raw_response, attempt_num
                else:
                    self.logger.warning(warning(f"Generation attempt {attempt_num} produced code but export failed: {error_msg}"))
                    # Save failed export.js with error info
                    self._save_failed_export(kwargs.get('output_folder'), result.js_code_export, 
                                           attempt_num, error_msg or "Export validation failed", metrics)
                    
                    # Try to fix with ShapeCodeFixAgent before full regeneration
                    fix_result = self._try_fix_with_agent(
                        result.js_code_export, 
                        error_msg or "Export validation failed",
                        kwargs.get('output_folder'),
                        attempt_num
                    )
                    
                    if fix_result['success']:
                        # Fixed code worked! Use it as the result
                        result.js_code_export = fix_result['fixed_code']
                        self.logger.info(f"Fixed code exports successfully after fix agent intervention")
                        return result, True, metrics, raw_response, attempt_num
                    
                    if attempt < max_retries:
                        # Add error context to retry prompt
                        kwargs['_retry_context'] = f"Previous attempt failed export validation. Please ensure all variables are declared before use and avoid syntax errors."
                        continue
                    else:
                        self.logger.error(f"All {max_retries + 1} attempts failed export validation")
                        return result, False, metrics, raw_response, attempt_num
                        
            except Exception as e:
                self.logger.warning(warning(f"Export test attempt {attempt_num} failed with error: {e}"))
                # Save failed export.js with error info
                if result and result.js_code_export:
                    self._save_failed_export(kwargs.get('output_folder'), result.js_code_export, 
                                           attempt_num, str(e), metrics)
                    
                    # Try to fix with ShapeCodeFixAgent before full regeneration
                    fix_result = self._try_fix_with_agent(
                        result.js_code_export, 
                        str(e),
                        kwargs.get('output_folder'),
                        attempt_num
                    )
                    
                    if fix_result['success']:
                        # Fixed code worked! Use it as the result
                        result.js_code_export = fix_result['fixed_code']
                        self.logger.info(f"Fixed code exports successfully after fix agent intervention")
                        return result, True, metrics, raw_response, attempt_num
                
                if attempt < max_retries:
                    # Add specific error context for retry
                    kwargs['_retry_context'] = f"Previous attempt failed with error: {str(e)}. Please fix the issue and regenerate."
                    continue
                else:
                    return result, False, metrics, raw_response, attempt_num
                    
            finally:
                # Clean up temporary directory
                if temp_dir:
                    shutil.rmtree(temp_dir, ignore_errors=True)
        
        # Should never reach here
        return result, False, metrics, raw_response, max_retries + 1
    
    def _save_failed_export(self, output_folder: str, js_code: str, attempt_num: int, 
                           error_msg: str, metrics: Dict[str, Any] = None):
        """Save failed export.js with error info."""
        if not output_folder or not js_code:
            return
            
        try:
            export_temp = os.path.join(output_folder, '_export_temp')
            os.makedirs(export_temp, exist_ok=True)
            
            # Save failed JS with error header
            failed_path = os.path.join(export_temp, f'export_failed_attempt_{attempt_num}.js')
            with open(failed_path, 'w', encoding='utf-8') as f:
                f.write(f"// FAILED EXPORT - Attempt {attempt_num}\n")
                f.write(f"// Error: {error_msg}\n")
                f.write(f"// Time: {time.strftime('%Y-%m-%d %H:%M:%S')}\n")
                f.write("// " + "=" * 50 + "\n\n")
                f.write(js_code)
            
            self.logger.info(f"Saved failed export: attempt_{attempt_num}")
        except Exception as e:
            self.logger.warning(f"Could not save failed export: {e}")
    
    def _try_fix_with_agent(self, js_code: str, error_msg: str, 
                           output_folder: Optional[str], attempt_num: int) -> Dict[str, Any]:
        """
        Try to fix JavaScript code using ShapeCodeFixAgent.
        
        Args:
            js_code: JavaScript code with errors
            error_msg: Error message from export attempt
            output_folder: Output folder for saving fixes
            attempt_num: Current attempt number
            
        Returns:
            Dictionary with 'success' boolean and 'fixed_code' if successful
        """
        try:
            self.logger.info(progress(f"Attempting to fix code with ShapeCodeFixAgent..."))
            
            # Import and initialize fix agent
            from agents.shape_code_fix_agent import ShapeCodeFixAgent
            fix_agent = ShapeCodeFixAgent(self.config)
            
            # Try to fix the code
            fixed_code, fix_success, fix_metrics = fix_agent.fix_javascript_code(
                js_code=js_code,
                error_message=error_msg,
                max_attempts=2,  # Quick fix attempts
                output_folder=output_folder,
                attempt_num=attempt_num
            )
            
            if not fix_success:
                self.logger.warning("Fix agent could not resolve syntax errors")
                return {'success': False, 'fixed_code': None}
            
            # Test if fixed code can be exported
            from utils.mesh_exporter import MeshExporter
            import shutil
            
            temp_dir = None
            try:
                # Create temp directory for testing within output folder
                if output_folder:
                    temp_dir_name = f'fix_test_{attempt_num}_{time.time():.0f}'
                    temp_dir = os.path.join(output_folder, temp_dir_name)
                else:
                    # Fallback if no output folder provided
                    temp_dir = f'/tmp/fix_test_{attempt_num}_{time.time():.0f}'
                    
                export_temp = os.path.join(temp_dir, '_export_temp')
                os.makedirs(export_temp, exist_ok=True)
                
                # Save fixed code
                export_path = os.path.join(export_temp, 'export.js')
                with open(export_path, 'w', encoding='utf-8') as f:
                    f.write(fixed_code)
                
                # Test export
                exporter = MeshExporter(temp_dir)
                test_result = exporter.test_export(fixed_code)
                
                if test_result['success']:
                    self.logger.info(f"Fix agent successfully resolved the issue!")
                    
                    # Save successful fix to output folder
                    if output_folder:
                        export_temp_out = os.path.join(output_folder, '_export_temp')
                        os.makedirs(export_temp_out, exist_ok=True)
                        
                        # Overwrite export.js with fixed version
                        fixed_export_path = os.path.join(export_temp_out, 'export.js')
                        with open(fixed_export_path, 'w', encoding='utf-8') as f:
                            f.write(fixed_code)
                        
                        # Also save as a separate fixed file for reference
                        fixed_ref_path = os.path.join(export_temp_out, f'export_fixed_by_agent_{attempt_num}.js')
                        with open(fixed_ref_path, 'w', encoding='utf-8') as f:
                            f.write(f"// Successfully fixed by ShapeCodeFixAgent\n")
                            f.write(f"// Original error: {error_msg}\n")
                            f.write(f"// Fixed at: {time.strftime('%Y-%m-%d %H:%M:%S')}\n")
                            f.write("// " + "=" * 50 + "\n\n")
                            f.write(fixed_code)
                    
                    return {'success': True, 'fixed_code': fixed_code}
                else:
                    self.logger.warning(f"Fixed code still has issues: {test_result.get('error', 'Unknown')}")
                    return {'success': False, 'fixed_code': None}
                    
            finally:
                # Clean up temp directory
                if temp_dir and os.path.exists(temp_dir):
                    shutil.rmtree(temp_dir, ignore_errors=True)
                    
        except Exception as e:
            self.logger.error(f"Fix agent attempt failed: {e}")
            return {'success': False, 'fixed_code': None}