# In-context samples for parter system prompt
# This file stores general in-context samples for grouping logic (how to divide groups/links), not limited to keyboards.

"""
CRITICAL FORMAT WARNING FOR THE LLM:
- You MUST output your Three.js code ONLY inside a code block labeled <js_code_export> ... </js_code_export>.
- DO NOT use markdown code blocks (```) or any other tags (such as <js_code>).
- DO NOT include any explanation, text, or whitespace outside the required tags.
- If you do not use <js_code_export>...</js_code_export> as your ONLY code block, your response will be rejected and you will not receive credit.

Negative Example (do NOT do this):
```
```js
// code here
```

Or:
```
<js_code>
// code here
</js_code>
```

Positive Example (CORRECT):
<js_code_export>
// code here
</js_code_export>

Only use <js_code_export> ... </js_code_export> for your output.
"""

samples = r'''
# Example1:

### input links.json:
[
   {{
     "extended_description": "A picturesque, modern Ferris wheel with a tall, elegant structure. The design features a sleek A-frame steel support, an illuminated circular rim with evenly spaced spokes, and multiple glass-walled gondolas that freely swing from articulated hanger arms. The wheel rotates smoothly on a motorized hub and axle, anchored by a stable base platform. Each gondola has a hinged door and interior seating, and subtle lighting runs along the rim and spokes for night-time spectacle."
   }},
   {{
     "link_name": "base_platform",
     "description_shape": "A broad, sturdy platform composed of several rectangular and block-like elements, with anchor points and a textured surface for stability.",
     "description_position": "Root of the assembly at the world origin. All further links are positioned relative to the center of the platform.",
     "reasoning": "Provides a level, load-bearing foundation and attachment points for the A-frame legs. Ensures stability against wind forces and wheel torque."
   }},
   {{
     "link_name": "support_leg_left",
     "description_shape": "An A-frame made from two long, hollow steel segments meeting at an angle at the top, joined by a horizontal cross-beam and reinforced with thin plates at the joints.",
     "description_position": "The feet of both leg segments are attached to one side of the base platform. The apex aligns with the hub center, forming a mirror counterpart to the right support leg.",
     "reasoning": "Transfers vertical and lateral loads from the rotating wheel to the ground. The A-shape resists bending and halves the span needed for cross-bracing."
   }},
   {{
     "link_name": "support_leg_right",
     "description_shape": "Identical in shape and structure to the left support leg, mirrored across the central plane. Includes matching reinforcements and cross-beam socket.",
     "description_position": "Feet attach to the opposite side of the base platform. Apex meets the hub center, clamping the axle bearings from the opposite side.",
     "reasoning": "Forms the other half of the A-frame, creating a wide stance that keeps the wheel stable. Symmetry balances the structural load."
   }},
   {{
     "link_name": "cross_brace_top",
     "description_shape": "A long, rectangular beam with a hollow interior and reinforced end plates, spanning between the two A-frame apices.",
     "description_position": "Connects the tops of the two A-frames, directly under the axle bearings, bolted into the reinforcements of both support legs.",
     "reasoning": "Prevents the apices from spreading under the radial forces produced by the rotating rim and wind. Maintains precise bearing alignment."
   }},
   {{
     "link_name": "axle_shaft",
     "description_shape": "A thick, cylindrical shaft with chamfered ends and a central bore for weight reduction. Flange rings are welded near each end for bearing seats.",
     "description_position": "Centered at the top of the A-frame, horizontal and aligned with the main axis of the wheel. Ends sit inside the left and right bearings, which are nested inside the A-frame apices.",
     "reasoning": "Acts as the primary rotational axis for the Ferris wheel. Its diameter and flanges ensure minimal deflection and secure connection to the hub."
   }},
   {{
     "link_name": "left_bearing",
     "description_shape": "A composite of two concentric cylinders, populated by evenly spaced ball bearings and housed in a rectangular bearing block.",
     "description_position": "Encases the left axle flange, bolted to a reinforcement plate welded to the left support leg. Axis colinear with the axle shaft.",
     "reasoning": "Allows low-friction rotation of the axle while transferring radial loads into the support structure."
   }},
   {{
     "link_name": "right_bearing",
     "description_shape": "Identical in shape and structure to the left bearing.",
     "description_position": "Mirrored placement on the right support leg, capturing the axle's right flange.",
     "reasoning": "Provides symmetrical support and precise concentricity for smooth wheel rotation."
   }},
   {{
     "link_name": "wheel_hub",
     "description_shape": "A thick, disc-like structure with a hollow center to slide over the axle. Multiple rectangular sockets are evenly arrayed around the rim of the disc, with decorative radial ridges between sockets.",
     "description_position": "Rigidly welded and bolted at the midpoint of the axle shaft. Rotates with the axle.",
     "reasoning": "Distributes torque from the axle to all spokes while keeping the structure centralized. The socket system simplifies spoke installation and maintenance."
   }},
   {{
     "link_name": "wheel_spoke",
     "description_shape": "Tapered, beam-like elements radiating from the wheel hub to the rim, each with a hollow I-beam profile and decorative lighting along the faces.",
     "description_position": "Multiple instances radiate from the hub sockets to the inner surface of the wheel rim, evenly spaced around the hub and lying in the wheel plane. Rim ends bolt into corresponding rim lugs.",
     "reasoning": "Carry tensile and compressive loads, keeping the rim true and round. The tapered, hollow I-beam profile maximizes strength-to-weight ratio. Lighting provides visual spectacle."
   }},
   {{
     "link_name": "wheel_rim",
     "description_shape": "A large, circular ring with a thick, tubular cross-section. The outer surface is segmented with rectangular lugs for hanger attachment, and a groove runs along the inner circumference for drainage.",
     "description_position": "Concentric with the axle shaft, attached to spoke tips via high-strength bolts. Serves as parent for all gondola arms.",
     "reasoning": "Forms the circular track that defines the Ferris wheel's diameter. Its rigidity and balanced mass distribution are vital for smooth rotation."
   }},
   {{
     "link_name": "gondola_arm",
     "description_shape": "L-shaped weldment with a vertical hanger joined to a horizontal stub axle. The top of the hanger has a clevis joint with a pin hole.",
     "description_position": "Multiple instances are pin-hinged to the underside of each rim lug, allowing free swing about the tangent direction. The horizontal stub axle mates with the gondola body yoke, axis aligned with the main wheel axis.",
     "reasoning": "Transfers passenger load from the gondola to the rim while allowing the gondola to self-level under gravity as the wheel rotates."
   }},
   {{
     "link_name": "gondola_body",
     "description_shape": "An octagonal prism with glass panels inset into an aluminum frame. The floor is a circular plate, and the roof is a shallow cone. Interior seating wraps around the wall.",
     "description_position": "Suspended from the gondola arm stub axle via a U-shaped yoke welded to the roof. Center of mass directly below the stub axle to maintain upright orientation.",
     "reasoning": "Houses passengers, providing panoramic views through glass walls. Geometry optimized for strength, visibility, and aerodynamics."
   }},
   {{
     "link_name": "gondola_door",
     "description_shape": "A framed glass panel with a handle, sized to allow easy entry and exit. The frame surrounds a large glass area.",
     "description_position": "Hinged along one vertical edge to a door jamb on the gondola body. Hinge axis vertical and co-radial with the gondola body's local axis. Closed position flush with the outer wall.",
     "reasoning": "Provides safe entry and exit for passengers. Articulated opening required for operational flow and emergency evacuation."
   }},
   {{
     "link_name": "drive_motor_assembly",
     "description_shape": "A large, cylindrical motor housing attached to a gear reducer box. A pinion gear meshes with a large ring gear welded to the wheel rim.",
     "description_position": "Mounted on one side of the base platform, with the gear meshing with the ring gear on the wheel rim. Axis parallel to the axle shaft. Flexible coupling joins the gear reducer output to the motor shaft.",
     "reasoning": "Generates rotational torque to spin the wheel at controlled speeds. Positioning at ground level eases maintenance and balances structural loads through the ring gear interface."
   }}
]

### Output three.js code:
<js_code_export>
import * as THREE from 'three';

// --- Helper function to create a cylindrical beam (e.g., a steel support or spoke) between two points in 3D space ---
// This function computes the direction and length between two points, creates a cylinder of that length,
// and orients it so that its axis aligns with the vector from point1 to point2. This is essential for modeling
// structural elements like beams, spokes, or legs that must connect two arbitrary points in space.
function createCylinderBeam(point1, point2, radius, material) {{
    const direction = new THREE.Vector3().subVectors(point2, point1); // Vector from point1 to point2
    const length = direction.length(); // Physical length of the beam
    const geometry = new THREE.CylinderGeometry(radius, radius, length, 16); // Uniform radius cylinder
    const mesh = new THREE.Mesh(geometry, material);

    // Position the cylinder so its center is halfway between the two points
    mesh.position.copy(point1).add(direction.multiplyScalar(0.5));

    // By default, CylinderGeometry is aligned along the Y-axis. We rotate it to align with the direction vector.
    const Y_AXIS = new THREE.Vector3(0, 1, 0);
    mesh.quaternion.setFromUnitVectors(Y_AXIS, direction.clone().normalize()); // Align Y to direction

    return mesh;
}}

// --- Helper function to get a point along a line segment at a specific Y (vertical) value ---
// This is used to find where a horizontal cross-beam or brace would intersect a slanted leg at a given height.
// It uses linear interpolation to find the point at yTarget between start and end.
function getPointOnLineAtY(start, end, yTarget) {{
    if (Math.abs(end.y - start.y) < 0.001) return start.clone(); // If the line is nearly horizontal, return start
    const t = (yTarget - start.y) / (end.y - start.y); // Fraction along the line to reach yTarget
    // Extrapolation is allowed if t < 0 or t > 1
    return new THREE.Vector3().lerpVectors(start, end, t);
}}


export function createScene() {{
    const scene = new THREE.Scene(); // Root object for the 3D scene graph

    // --- MATERIALS ---
    // These are basic materials for visualization. In a real-world scenario, the exporter may ignore them,
    // but they are necessary for creating visible meshes in Three.js.
    const steelMaterial = new THREE.MeshStandardMaterial({{ color: 0x888888, roughness: 0.5, metalness: 0.8 }}); // Simulates steel
    const concreteMaterial = new THREE.MeshStandardMaterial({{ color: 0xa0a0a0, roughness: 0.8 }}); // Simulates concrete
    const glassMaterial = new THREE.MeshStandardMaterial({{ color: 0xaaddff, transparent: true, opacity: 0.5, roughness: 0.2 }}); // Simulates glass
    const ledMaterial = new THREE.MeshStandardMaterial({{ color: 0xffaa00, emissive: 0xffaa00, emissiveIntensity: 1 }}); // Simulates LED lighting
    const seatMaterial = new THREE.MeshStandardMaterial({{ color: 0x553311 }}); // Placeholder for gondola seats
    const hubMaterial = new THREE.MeshStandardMaterial({{ color: 0x666666, roughness: 0.6, metalness: 0.7 }}); // Simulates steel hub
    const rimMaterial = new THREE.MeshStandardMaterial({{ color: 0x777777, roughness: 0.5, metalness: 0.8 }}); // Simulates steel rim
    const gondolaBodyMaterial = new THREE.MeshStandardMaterial({{ color: 0xffccaa, roughness: 0.7 }}); // Simulates painted gondola body
    const motorMaterial = new THREE.MeshStandardMaterial({{ color: 0x444455 }}); // Simulates motor/gearbox

    // --- FERRIS WHEEL STRUCTURE ---

    // Base Platform Group: Contains the main deck and anchor blocks that support the entire Ferris wheel.
    // This is the static foundation, typically made of reinforced concrete.
    const basePlatformGroup = new THREE.Group();
    basePlatformGroup.name = "BasePlatform";

    // 1. Main Deck: The primary platform on which the wheel and its supports rest.
    // Physically, this would be a thick, reinforced concrete slab.
    const mainDeckGeo = new THREE.BoxGeometry(6, 0.5, 20); // Width, height, depth
    const mainDeck = new THREE.Mesh(mainDeckGeo, concreteMaterial);
    mainDeck.name = "MainDeck";
    mainDeck.position.set(0, -0.25, -2-1); // Centered horizontally, slightly below world origin
    basePlatformGroup.add(mainDeck);

    // 2. Anchor Blocks: Large concrete blocks at each end of the deck to anchor the support legs.
    const anchorBlockGeo = new THREE.BoxGeometry(6, 1, 4);
    const leftAnchorBlock = new THREE.Mesh(anchorBlockGeo, concreteMaterial);
    leftAnchorBlock.name = "LeftAnchorBlock";
    leftAnchorBlock.position.set(0, -0.75, -5-2-2-2); // Positioned at one end of the deck
    basePlatformGroup.add(leftAnchorBlock);

    const rightAnchorBlock = new THREE.Mesh(anchorBlockGeo, concreteMaterial);
    rightAnchorBlock.name = "RightAnchorBlock";
    rightAnchorBlock.position.set(0, -0.75, 5); // Positioned at the opposite end
    basePlatformGroup.add(rightAnchorBlock);

    // 3. Axle Position: The global position of the wheel's axle (center of rotation)
    // This is the highest point of the support structure and the center of the wheel.
    const axleGlobalPosition = new THREE.Vector3(0, 50, -3);

    // 4. Support Legs: Steel beams forming two A-frames (left and right) to support the axle.
    // Each A-frame consists of two slanted legs and a cross-beam for lateral stability.
    const supportLegMaterial = steelMaterial;
    const supportLegRadius = 0.35; // Realistic radius for a large steel support

    // --- LEFT SUPPORT LEGS (A-frame) ---
    const supportLegLeftGroup = new THREE.Group();
    supportLegLeftGroup.name = "SupportLegsLeft";
    basePlatformGroup.add(supportLegLeftGroup);
    // Apex points: where the legs meet at the top (axle height)
    const apexL = new THREE.Vector3(-2, axleGlobalPosition.y, axleGlobalPosition.z); // Left apex
    const apexR = new THREE.Vector3(2.0, axleGlobalPosition.y, axleGlobalPosition.z); // Right apex (used for right legs)
    // Foot points: where the legs are anchored to the base
    const footL1 = new THREE.Vector3(-2, -0.25, -5-2-2-2); // Left foot (matches left anchor block)
    const footL2 = new THREE.Vector3(2, -0.25, -5-2-2-2); // Right foot (front left)

    // Leg L1: Left leg of the A-frame (from left foot to left apex)
    const legL1 = createCylinderBeam(footL1, apexL, supportLegRadius, supportLegMaterial);
    legL1.name = "Leg_L1";
    supportLegLeftGroup.add(legL1);
    // Leg L2: Right leg of the left A-frame (from right foot to right apex)
    const legL2 = createCylinderBeam(footL2, apexR, supportLegRadius, supportLegMaterial); // Connects to apexR
    legL2.name = "Leg_L2";
    supportLegLeftGroup.add(legL2);

    // Cross-beam: Provides lateral stability and resists racking forces.
    // Placed at a specific height (crossBeamY) above the base.
    const crossBeamY = 6; // Height above base for the cross-beam
    const crossBeamRadius = 0.3; // Slightly thinner than main legs
    // Find points on each leg at the cross-beam height
    const pL1_Y6_on_legL1 = getPointOnLineAtY(footL1, apexL, crossBeamY); // On legL1
    const pL2_Y6_on_legL2 = getPointOnLineAtY(footL2, apexR, crossBeamY); // On legL2

    // The original logic also computes a diagonal cross-brace for additional rigidity
    // (X-bracing is common in real-world structures to resist lateral loads)
    const pL1_Y6 = getPointOnLineAtY(footL1, apexL, crossBeamY); // On legL1
    const pL2_Y6 = getPointOnLineAtY(footL2, apexL, crossBeamY); // On a diagonal from footL2 to apexL

    if (pL1_Y6 && pL2_Y6) {{
        // This cross-beam acts as a diagonal brace, increasing the frame's resistance to side loads.
        const crossBeamL = createCylinderBeam(pL1_Y6, pL2_Y6, crossBeamRadius, supportLegMaterial);
        crossBeamL.name = "CrossBeam_Left";
        supportLegLeftGroup.add(crossBeamL);
    }}

    // --- RIGHT SUPPORT LEGS (A-frame) ---
    const supportLegRightGroup = new THREE.Group();
    supportLegRightGroup.name = "SupportLegsRight";
    basePlatformGroup.add(supportLegRightGroup);
    // Foot points for right A-frame
    const footR1 = new THREE.Vector3(-2, -0.25, 5); // Left foot (rear right)
    const footR2 = new THREE.Vector3(2, -0.25, 5); // Right foot (rear right)

    // Leg R1: Left leg of the right A-frame (from left foot to left apex)
    const legR1 = createCylinderBeam(footR1, apexL, supportLegRadius, supportLegMaterial); // footR1 to apexL
    legR1.name = "Leg_R1";
    supportLegRightGroup.add(legR1);
    // Leg R2: Right leg of the right A-frame (from right foot to right apex)
    const legR2 = createCylinderBeam(footR2, apexR, supportLegRadius, supportLegMaterial); // footR2 to apexR
    legR2.name = "Leg_R2";
    supportLegRightGroup.add(legR2);

    // Cross-beam for right A-frame: connects the two legs at a given height for stability
    const pR1_Y6 = getPointOnLineAtY(footR1, apexR, crossBeamY); // On legR1
    const pR2_Y6 = getPointOnLineAtY(footR2, apexR, crossBeamY); // On legR2
    if (pR1_Y6 && pR2_Y6) {{
        const crossBeamR = createCylinderBeam(pR1_Y6, pR2_Y6, crossBeamRadius, supportLegMaterial);
        crossBeamR.name = "CrossBeam_Right";
        supportLegRightGroup.add(crossBeamR);
    }}

    // Top cross-brace: Connects the tops of the A-frames, providing torsional rigidity and supporting the axle.
    // This is a critical structural element to prevent the frames from spreading apart under load.
    const crossBraceTopGeo = new THREE.BoxGeometry(4, 0.6, 0.6); // Wide, thick steel beam
    const crossBraceTop = new THREE.Mesh(crossBraceTopGeo, steelMaterial);
    crossBraceTop.name = "CrossBraceTop";
    crossBraceTop.position.set(0, axleGlobalPosition.y, axleGlobalPosition.z); // At axle height
    basePlatformGroup.add(crossBraceTop); // Part of the static base

    // --- STATIC BASE GROUP ---
    // This group contains all non-rotating structural elements (foundation, supports, axle)
    const staticBase = new THREE.Group();
    staticBase.name = "StaticBase";
    scene.add(staticBase);
    staticBase.add(basePlatformGroup);

    // --- AXLE SHAFT ---
    // The main axle is a large steel cylinder that supports the rotating wheel assembly.
    // It must be strong enough to bear the weight of the wheel and all gondolas, and to resist bending and torsion.
    const axleShaftGeo = new THREE.CylinderGeometry(0.9, 0.9, 7, 32); // Realistic axle size
    const axleShaft = new THREE.Mesh(axleShaftGeo, steelMaterial);
    axleShaft.name = "AxleShaft";
    axleShaft.rotation.z = Math.PI / 2; // Align cylinder axis with X (wheel rotation axis)
    axleShaft.position.copy(axleGlobalPosition);
    staticBase.add(axleShaft);

    // --- WHEEL ASSEMBLY (rotates as a unit) ---
    const wheelAssembly = new THREE.Group();
    wheelAssembly.name = "WheelAssembly";
    wheelAssembly.position.copy(axleGlobalPosition); // Centered at axle
    scene.add(wheelAssembly);

    // --- HUB (central part of the wheel) ---
    // The hub is a thick steel cylinder that connects the axle to the spokes.
    const hubGeo = new THREE.CylinderGeometry(3, 3, 1, 32); // Large, thick hub
    const hub = new THREE.Mesh(hubGeo, hubMaterial);
    hub.name = "Hub";
    hub.rotation.z = Math.PI / 2; // Align cylinder axis with wheel's X-axis
    wheelAssembly.add(hub);

    // --- SPOKES ---
    // The spokes are steel rods radiating from the hub to the rim, transmitting forces and supporting the rim.
    // In a real Ferris wheel, the spokes are under tension (like bicycle spokes) and keep the rim circular.
    const numSpokes = 16; // Number of spokes (evenly spaced)
    const spokeLength = 27; // Length from hub to rim
    const spokeHubSection = 0.4; // Thicker at the hub
    const spokeRimSection = 0.2; // Thinner at the rim
    const spokeGeo = new THREE.CylinderGeometry(spokeRimSection, spokeHubSection, spokeLength, 8); // Tapered cylinder

    for (let i = 0; i < numSpokes; i++) {{
        const spoke = new THREE.Mesh(spokeGeo, steelMaterial);
        spoke.name = `Spoke_${{i}}`;
        const angle = (i / numSpokes) * Math.PI * 2; // Even angular spacing
        const spokeAttachRadius = 3 + spokeLength / 2; // Hub radius + half spoke length (centered)

        // Position each spoke in the YZ plane, radiating from the hub
        spoke.position.set(0, Math.cos(angle) * spokeAttachRadius, Math.sin(angle) * spokeAttachRadius);
        spoke.lookAt(0,0,0); // Orient the spoke to point toward the hub center
        spoke.rotateX(Math.PI/2); // Align cylinder's axis with the direction

        // LED strip: Decorative lighting attached to each spoke (not structural)
        const ledStripGeo = new THREE.PlaneGeometry(0.1, spokeLength);
        const ledStrip = new THREE.Mesh(ledStripGeo, ledMaterial);
        ledStrip.name = `LEDStrip_Spoke_${{i}}`;
        // Place the LED strip just above the surface of the spoke
        ledStrip.position.set(0,0, Math.max(spokeHubSection, spokeRimSection) + 0.01);
        spoke.add(ledStrip);

        wheelAssembly.add(spoke);
    }}

    // --- RIM ---
    // The rim is a large steel torus (ring) that forms the outer edge of the wheel.
    // It must be strong enough to support the gondolas and transmit forces to the spokes.
    const rimMajorRadius = 30; // Physical radius of the wheel
    const rimTubeRadius = 1.2; // Thickness of the rim
    const rimGeo = new THREE.TorusGeometry(rimMajorRadius, rimTubeRadius, 16, 64); // High segment count for smoothness
    const rim = new THREE.Mesh(rimGeo, rimMaterial);
    rim.name = "Rim";
    // Align the torus so its central hole is along the X-axis (rotation axis)
    rim.rotation.y = Math.PI / 2;
    rim.rotation.z = Math.PI / 2;
    wheelAssembly.add(rim);

    // --- GONDOLAS ---
    // Each gondola is a passenger cabin suspended from the rim. It must remain upright as the wheel rotates.
    // The gondola is attached to the rim by a short arm, and its center of mass is below the pivot for stability.
    const numGondolas = 16; // One per spoke
    const gondolaArmLength = 4; // Length of the suspension arm
    const gondolaArmRadius = 0.25; // Thickness of the arm

    for (let i = 0; i < numGondolas; i++) {{
        const gondolaGroup = new THREE.Group();
        gondolaGroup.name = `GondolaGroup_${{i}}`;
        const angle = (i / numGondolas) * Math.PI * 2; // Even spacing around the rim

        // Position the gondola's pivot point on the rim (in the YZ plane)
        const gondolaPivotX = 0;
        const gondolaPivotY = Math.cos(angle) * rimMajorRadius;
        const gondolaPivotZ = Math.sin(angle) * rimMajorRadius;
        gondolaGroup.position.set(gondolaPivotX, gondolaPivotY, gondolaPivotZ);
        wheelAssembly.add(gondolaGroup);

        // Gondola arm: vertical steel rod from rim to gondola body
        const gondolaArm = new THREE.Group(); // Allows for counter-rotation (to keep gondola upright)
        gondolaArm.name = `GondolaArmAssembly_${{i}}`;

        const armVerticalGeo = new THREE.CylinderGeometry(gondolaArmRadius, gondolaArmRadius, gondolaArmLength, 12);
        const armVertical = new THREE.Mesh(armVerticalGeo, steelMaterial);
        armVertical.name = `GondolaArmVertical_${{i}}`;
        armVertical.position.y = -gondolaArmLength / 2; // Pivot is at the top
        gondolaArm.add(armVertical);
        const gondolaBodyAttachPoint = new THREE.Vector3(0, -gondolaArmLength, 0); // Bottom of the arm

        // Gondola body: cylindrical cabin for passengers
        const bodyRadius = 2.8 / 2; // Realistic size for 4-6 passengers
        const bodyHeight = 2.4; // Headroom
        const gondolaBodyGeo = new THREE.CylinderGeometry(bodyRadius, bodyRadius, bodyHeight, 8);
        const gondolaBodyMesh = new THREE.Mesh(gondolaBodyGeo, gondolaBodyMaterial);
        gondolaBodyMesh.name = `GondolaBody_${{i}}`;
        gondolaBodyMesh.position.copy(gondolaBodyAttachPoint);
        gondolaBodyMesh.position.y -= bodyHeight / 2; // Body hangs below the arm
        gondolaArm.add(gondolaBodyMesh);

        // Gondola floor: thin disk at the bottom for passengers to stand on
        const floorGeo = new THREE.CylinderGeometry(bodyRadius * 0.95, bodyRadius * 0.95, 0.1, 16);
        const floor = new THREE.Mesh(floorGeo, concreteMaterial);
        floor.name = `GondolaFloor_${{i}}`;
        floor.position.y = -bodyHeight/2 + 0.05; // At the bottom inside the body
        gondolaBodyMesh.add(floor);

        // Gondola roof: conical roof for weather protection
        const roofGeo = new THREE.ConeGeometry(bodyRadius, 0.4, 8);
        const roof = new THREE.Mesh(roofGeo, gondolaBodyMaterial);
        roof.name = `GondolaRoof_${{i}}`;
        roof.position.y = bodyHeight/2 + 0.4/2; // On top of the body
        gondolaBodyMesh.add(roof);

        // Gondola door: glass panel on the side for entry/exit
        const doorGeo = new THREE.PlaneGeometry(1.2, 2.0);
        const door = new THREE.Mesh(doorGeo, glassMaterial);
        door.name = `GondolaDoor_${{i}}`;
        door.position.set(bodyRadius, 0, 0); // On the +X side
        door.lookAt(new THREE.Vector3(bodyRadius*2, 0, 0)); // Face outward
        gondolaBodyMesh.add(door);

        gondolaGroup.add(gondolaArm);
        // The gondola arm can be counter-rotated in animation to keep the gondola upright as the wheel turns.
        gondolaArm.rotation.x = 0; // Default: upright
    }}

    // --- DRIVE MOTOR ASSEMBLY ---
    // The drive motor and gearbox provide the torque to rotate the wheel. They are mounted on the base platform.
    // In a real Ferris wheel, the motor would drive the axle via gears or friction rollers.
    const driveMotorGroup = new THREE.Group();
    driveMotorGroup.name = "DriveMotorAssembly";
    // Motor housing: cylindrical steel shell
    const motorHousingGeo = new THREE.CylinderGeometry(1.5, 1.5, 2, 16);
    const motorHousing = new THREE.Mesh(motorHousingGeo, motorMaterial);
    motorHousing.name = "MotorHousing";
    motorHousing.rotation.x = Math.PI / 2; // Align with axle
    motorHousing.position.set(0, 1.5/2, 0); // Centered on base

    // Gearbox: box-shaped housing for gears
    const gearBoxGeo = new THREE.BoxGeometry(2.5, 1.5, 1.5);
    const gearBox = new THREE.Mesh(gearBoxGeo, motorMaterial);
    gearBox.name = "GearBox";
    gearBox.position.set(0, 1.5/2, 2 + 1.5/2 - 1); // Positioned at end of motor housing
    gearBox.position.z = 1 + 1.5/2; // Adjacent to motor

    driveMotorGroup.add(motorHousing);
    driveMotorGroup.add(gearBox);
    // Position the motor assembly at one end of the main deck
    // The motor must be securely anchored to resist the reaction torque from the wheel.
    driveMotorGroup.position.set(0, 0.25, 14/2 + 1); // At the end of the deck
    basePlatformGroup.add(driveMotorGroup); // Add to the static base

    // The exporter will call rootObject.updateMatrixWorld(true) to finalize transforms.
    return scene;
}}

// If you want to export the scene as a variable:
// export const sceneObject = createScene();
</js_code_export>

# Example 2: a camera
# Input extended description:
[
{{
"extended_description": "A professional mirrorless digital camera featuring an interchangeable zoom lens. The body includes a textured hand-grip, pop-up flash, tilting rear LCD, dual command dials, hot-shoe, twin strap lugs, battery door, and a metal tripod thread. The lens assembly is composed of multiple concentric rings—mount, zoom, and focus—allowing independent rotation. The flash deploys on a hinged arm, while the LCD pivots on a side hinge for high- and low-angle shooting. All mechanical joints are modeled so that a three.js developer can animate zoom/focus rotation, flash deployment, LCD tilt, shutter-button travel, and battery-door swing."
}},
{{
"link_name": "camera_body",
"description_shape": "The main shell of the camera is an L-shaped profile, with a prominent hand-grip protruding on the right side. The body features a sturdy, ergonomic design with a slightly textured surface for grip. The top deck is smoothly contoured with gentle curves and subtle chamfers, and the rear top has a shallow recess for the viewfinder. The front center includes a circular opening for the lens mount.",
"description_position": "This is the root component, positioned at the world origin with the lens mount facing forward. All other parts are attached relative to this body.",
"reasoning": "Provides structural backbone, houses electronics, and defines reference coordinate system."
}},
{{
"link_name": "lens_mount_ring",
"description_shape": "A relatively thin, annular ring that fits snugly around the lens mount opening on the camera body. The outer rim is dotted with several small, evenly spaced locating pins that protrude slightly forward.",
"description_position": "Concentric with the lens mount opening on the camera body, flush with the front surface. It is fixed in place and does not articulate.",
"reasoning": "Mechanical interface that locks interchangeable lens to body; pins serve alignment and bayonet engagement."
}},
{{
"link_name": "lens_barrel_base",
"description_shape": "A hollow cylindrical structure forming the main housing of the lens, with a capped rear and a matte black finish. The interior includes a stepped ledge to accept glass elements.",
"description_position": "The rear flange mates with the lens mount ring, extending forward from the camera body. It is fixed in place once mounted.",
"reasoning": "Forms structural housing of lens; foundation for zoom and focus rings."
}},
{{
"link_name": "lens_zoom_ring",
"description_shape": "A textured cylindrical ring that sleeves over the lens barrel base. The surface is covered with raised rubber grip ridges, and focal length markings are visible around the circumference.",
"description_position": "Positioned just forward of the lens mount, encircling the lens barrel. It rotates around the lens axis to adjust zoom.",
"reasoning": "Allows user to adjust focal length; rotation animates zoom. Slightly wider diameter for tactile distinction."
}},
{{
"link_name": "lens_focus_ring",
"description_shape": "A cylindrical ring with a finely knurled surface for grip, located in front of the zoom ring. It is slightly narrower and separated by a small gap.",
"description_position": "Mounted in front of the zoom ring, sharing the same axis. It rotates independently to adjust focus.",
"reasoning": "Permits manual focus; separated positionally to avoid accidental operation."
}},
{{
"link_name": "lens_front_element",
"description_shape": "A convex glass disc forming the front optical element of the lens, with a subtle outward bulge. It is nested inside a thin protective bezel that extends just beyond the glass surface.",
"description_position": "Seated at the very front end of the lens barrel, flush with the front edge of the focus ring.",
"reasoning": "Optical element that completes light path. Non-moving but visible, giving realism."
}},
{{
"link_name": "lens_cap",
"description_shape": "A shallow, dish-shaped cover with two inward spring clips on opposite sides and an embossed logo on the front. The cap is designed to snap securely over the lens bezel.",
"description_position": "When attached, the cap fits snugly over the front bezel of the lens, slightly offset forward. It can be animated as a removable accessory.",
"reasoning": "Protects glass when not shooting. Demonstrates removable accessory."
}},
{{
"link_name": "shutter_button",
"description_shape": "A small, tapered button with a slightly concave top surface, positioned for easy access on the hand-grip. The button is wider at the base and narrows toward the top.",
"description_position": "Located on the sloped front of the hand-grip, set back from the front edge and just below the top deck. It can be pressed downward for animation.",
"reasoning": "Primary user control; pressing triggers exposure. Travel animation adds realism."
}},
{{
"link_name": "mode_dial",
"description_shape": "A knurled, circular dial with engraved icons on the top face, providing tactile feedback for mode selection.",
"description_position": "Sits on the top deck to the left of the viewfinder, slightly raised above the camera body. It rotates about its vertical axis.",
"reasoning": "Selects shooting mode; rotational articulation and textured edge for grip."
}},
{{
"link_name": "hot_shoe_mount",
"description_shape": "An extruded U-shaped rail with a central contact strip inset along its length, designed to accept external accessories.",
"description_position": "Centered on the top deck, just behind the viewfinder hump. It is fixed in place.",
"reasoning": "Accepts external flashes or accessories; static but visually important connector."
}},
{{
"link_name": "flash_hinge_arm",
"description_shape": "A slim, rectangular arm with a cylindrical hinge at its base, designed to pivot upward. The arm is proportioned to support the flash unit above the camera body when deployed.",
"description_position": "The hinge is inserted into a recess behind the hot-shoe mount, allowing the arm to rotate upward from the camera body.",
"reasoning": "Mechanically links flash_unit_body to camera body, enabling pop-up motion."
}},
{{
"link_name": "flash_unit_body",
"description_shape": "A thin, rectangular box forming the main body of the flash, with a translucent diffuser panel on the front face.",
"description_position": "Attached to the end of the flash hinge arm; when stowed, it lies flush with the top deck, and when deployed, it rotates up with the arm.",
"reasoning": "Houses flash tube; articulation provides clearance for light emission."
}},
{{
"link_name": "viewfinder_eyepiece",
"description_shape": "A rubberized, torus-shaped eyepiece that surrounds a small glass plate, providing a comfortable surface for the user to look through.",
"description_position": "Mounted on the rear upper center of the camera body, surrounding the small LCD inside the viewfinder hump.",
"reasoning": "Allows photographer to compose image; fixed part but distinct geometry."
}},
{{
"link_name": "lcd_hinge_arm",
"description_shape": "A flat, bracket-like arm with a cylindrical pivot pin at one end, designed to support the LCD screen and allow it to swing outward.",
"description_position": "Secured to the rear left edge of the LCD recess, with the pivot axis oriented vertically. It allows the LCD to swing outward from the camera body.",
"reasoning": "Provides first axis of articulation for tilting LCD; supports subsequent pitch."
}},
{{
"link_name": "lcd_screen",
"description_shape": "A slim, rectangular screen with a prominent display area on the front and a slightly recessed back for finger grip. The screen is surrounded by a narrow bezel.",
"description_position": "Attached to the end of the LCD hinge arm via a secondary hinge, allowing it to tilt up and down for flexible viewing angles.",
"reasoning": "Displays images and menus; two-axis articulation enables flexible viewing angles."
}},
{{
"link_name": "battery_door",
"description_shape": "A curved, rectangular panel with chamfered edges and an internal latch tab, forming the access door for the battery compartment.",
"description_position": "Located on the underside of the hand-grip, hinged at the rear edge to swing outward from the camera body.",
"reasoning": "Gives access to battery compartment; hinge modeled for authenticity."
}},
{{
"link_name": "tripod_thread_insert",
"description_shape": "A short, metal cylinder with internal threading, embedded flush in the underside of the camera body.",
"description_position": "Positioned along the centerline of the underside, offset to one side of the battery door.",
"reasoning": "Standard mount for tripods; static but critical detail."
}},
{{
"link_name": "strap_lug_left",
"description_shape": "A horseshoe-shaped stainless steel loop, welded to a small cube-shaped anchor, serving as an attachment point for a camera strap.",
"description_position": "Affixed to the upper left shoulder of the camera body, just below the top deck and near the side face.",
"reasoning": "Attachment point for camera strap; symmetrical counterpart included next."
}},
{{
"link_name": "strap_lug_right",
"description_shape": "Identical in form to the left strap lug, providing a second attachment point for balance.",
"description_position": "Mirrored on the right shoulder above the hand-grip, attached to the camera body.",
"reasoning": "Second strap attachment; balance camera when carried."
}}
]

# Output threejs code:
<js_code_export>
import * as THREE from 'three';

// --- Configuration ---
const SCALE_FACTOR = 0.1; // 1 unit in Three.js = 10mm

// The SCALE_FACTOR is chosen so that all dimensions in the code (in mm) match the real-world camera specs described in the JSON.
// This ensures the 3D model is to scale and matches the physical proportions described for each part.

function s(val) {{ // Scale helper
    return val * SCALE_FACTOR;
}}

// Materials to be used by the meshes
const materials = {{
    body: new THREE.MeshStandardMaterial({{ color: 0x333333, roughness: 0.6 }}),
    grip: new THREE.MeshStandardMaterial({{ color: 0x222222, roughness: 0.8 }}),
    lensBody: new THREE.MeshStandardMaterial({{ color: 0x1a1a1a, roughness: 0.5 }}),
    lensRing: new THREE.MeshStandardMaterial({{ color: 0x101010, roughness: 0.7 }}),
    glass: new THREE.MeshStandardMaterial({{ color: 0xaaccff, roughness: 0.1, transparent: true, opacity: 0.3, side: THREE.DoubleSide }}),
    metal: new THREE.MeshStandardMaterial({{ color: 0x888888, roughness: 0.3, metalness: 0.8 }}),
    plastic: new THREE.MeshStandardMaterial({{ color: 0x444444, roughness: 0.7 }}),
    rubber: new THREE.MeshStandardMaterial({{ color: 0x050505, roughness: 0.9 }}),
    screen: new THREE.MeshStandardMaterial({{ color: 0x000000, emissive: 0x112233, roughness: 0.9 }}),
    diffuser: new THREE.MeshStandardMaterial({{ color: 0xffffff, transparent: true, opacity: 0.7, roughness: 0.5 }}),
    blackHole: new THREE.MeshBasicMaterial({{color: 0x000000, side: THREE.DoubleSide}}), // For openings
}};

export function createScene() {{
    // ---
    // The following code builds a digital camera model, mapping each part and link to the JSON description.
    // The structure, hierarchy, and articulation are designed to match the reasoning and animation requirements in the JSON.
    // ---
    const originalParts = {{}}; // To store references to ALL original Object3Ds
    const originalMeshes = {{}}; // To store references to original MESHES specifically for easier iteration

    const originalCameraModelGroup = new THREE.Group();
    originalCameraModelGroup.name = "OriginalDigitalCameraModel";

    // Helper to create parts in the original hierarchy
    function createOriginalPart(name, geometry, material, parent, isMesh = true) {{
        // Each part corresponds to a link or mesh described in the JSON.
        // The hierarchy (parent/child) reflects the articulated structure (e.g., lens rings, flash arm, LCD hinge).
        const obj = isMesh ? new THREE.Mesh(geometry, material) : new THREE.Object3D();
        obj.name = name;
        if (parent) {{
            parent.add(obj);
        }} else {{
            originalCameraModelGroup.add(obj);
        }}
        originalParts[name] = obj;
        if (isMesh && geometry) {{ // only add actual meshes to originalMeshes
             originalMeshes[name] = obj;
        }}
        return obj;
    }}

    // --- Replicate Original Camera Model Creation Logic (Simplified) ---
    // This section builds the original hierarchy in memory to get correct world matrices.

    // --- Camera Body ---
    // JSON: camera_body
    // Shape: L-shaped profile, 140x95x70mm, with hand-grip protrusion. Root link at world origin.
    // Reasoning: Provides structural backbone, houses electronics, defines reference coordinate system.
    const bodyWidth = s(140), bodyHeight = s(95), bodyDepth = s(70);
    const bodyGeo = new THREE.BoxGeometry(bodyWidth, bodyHeight, bodyDepth);
    const cameraBody = createOriginalPart("camera_body_main_orig", bodyGeo, materials.body, originalCameraModelGroup);
    cameraBody.position.set(0, bodyHeight / 2, 0); // Place so base is at y=0
    originalParts['camera_body_ref_orig'] = cameraBody; // Reference for CameraBody link positioning

    // Hand grip: 30mm thick, protrudes 15mm, matches JSON description for ergonomic grip
    const gripWidth = s(40), gripThickness = s(30), gripProtrusion = s(15);
    const gripGeo = new THREE.BoxGeometry(gripWidth, bodyHeight, gripThickness);
    const handGrip = createOriginalPart("hand_grip_orig", gripGeo, materials.grip, cameraBody);
    handGrip.position.set((bodyWidth / 2) - (gripWidth / 2) + s(5), 0, (bodyDepth / 2) - (gripThickness / 2) + gripProtrusion);

    // Lens mount opening: 44mm diameter, matches JSON cut-out for lens
    const lensMountOpeningGeo = new THREE.CircleGeometry(s(44/2), 32);
    const lensMountOpening = createOriginalPart("lens_mount_opening_visual_orig", lensMountOpeningGeo, materials.blackHole, cameraBody);
    lensMountOpening.position.set(0, 0, bodyDepth/2 + s(0.1));
    lensMountOpening.rotation.y = Math.PI; // 

    // --- Lens Mount Ring ---
    // JSON: lens_mount_ring
    // Shape: 4mm thick annular ring, 56mm OD, 44mm ID, 8 pins
    // Reasoning: Mechanical interface for lens attachment
    const lensMountRingOuterR = s(56/2), lensMountRingInnerR = s(44/2), lensMountRingThickness = s(4);
    const lensMountRingShape = new THREE.Shape();
    lensMountRingShape.absarc(0, 0, lensMountRingOuterR, 0, Math.PI * 2, false);
    const holePath = new THREE.Path();
    holePath.absarc(0, 0, lensMountRingInnerR, 0, Math.PI * 2, true);
    lensMountRingShape.holes.push(holePath);
    const extrudeSettings = {{ depth: lensMountRingThickness, bevelEnabled: false }};
    const lensMountRingGeo = new THREE.ExtrudeGeometry(lensMountRingShape, extrudeSettings);
    const lensMountRing = createOriginalPart("lens_mount_ring_orig", lensMountRingGeo, materials.metal, cameraBody);
    lensMountRing.position.set(0, 0, bodyDepth/2 - lensMountRingThickness/2);

    // 8 locating pins for bayonet engagement, as per JSON
    const pinGeo = new THREE.CylinderGeometry(s(2/2), s(2/2), s(1), 8);
    for (let i = 0; i < 8; i++) {{
        const angle = (i / 8) * Math.PI * 2;
        const pin = createOriginalPart(`lens_mount_pin_${{i}}_orig`, pinGeo, materials.metal, lensMountRing);
        const radiusForPins = lensMountRingOuterR - s(1.5);
        pin.position.set(Math.cos(angle) * radiusForPins, Math.sin(angle) * radiusForPins, lensMountRingThickness / 2 + s(1/2));
        pin.rotation.y = Math.PI / 2;
    }}

    // --- Lens Barrel Base ---
    // JSON: lens_barrel_base
    // Shape: 25mm long, 60mm OD, wall 2mm, capped at rear
    // Reasoning: Foundation for zoom/focus rings
    const lbbLength = s(25), lbbOD = s(60/2); // lbbWall = s(2); lbbID = lbbOD - lbbWall; (Simplified as solid)
    const lbbGeo = new THREE.CylinderGeometry(lbbOD, lbbOD, lbbLength, 32); // Corrected: lbbLength for height
    const lensBarrelBase = createOriginalPart("lens_barrel_base_orig", lbbGeo, materials.lensBody, lensMountRing);
    lensBarrelBase.position.set(0, 0, lensMountRingThickness/2 + lbbLength/2);
    lensBarrelBase.rotation.x = Math.PI / 2; // Cylinders are Y-up by default, lens points along Z

    // --- Lens Zoom Ring ---
    // JSON: lens_zoom_ring
    // Shape: 20mm long, 68mm OD, rubber grip ridges
    // Reasoning: User rotates to zoom; wider for tactile feedback
    const lensZoomRingPivot = createOriginalPart("lensZoomRingPivot_orig", null, null, lensBarrelBase, false);
    lensZoomRingPivot.position.z = -lbbLength/2 + s(2); // Relative to lensBarrelBase local Z (which is world Y after rotation)
    lensZoomRingPivot.position.y = -lbbLength/2 + s(2); // Barrel length is along its Y axis after rotation.
    lensZoomRingPivot.position.z = 0;                   // Reset Z to avoid double offset.

    const lzrLength = s(20), lzrOD = s(68/2);
    const lzrGeo = new THREE.CylinderGeometry(lzrOD, lzrOD, lzrLength, 32);
    const lensZoomRing = createOriginalPart("lens_zoom_ring_orig", lzrGeo, materials.lensRing, lensZoomRingPivot);
    lensZoomRing.position.y = lzrLength/2; // Centered on its pivot, along its local Y (which will be Z after pivot rotation)
    lensZoomRing.rotation.y = - Math.PI / 2; // Align with Z axis of camera

    // 12 rubber grip ridges, as described in JSON
    for(let i=0; i < 12; i++) {{
        const angle = (i/12) * Math.PI * 2;
        const ridgeGeo = new THREE.BoxGeometry(s(2), lzrLength * 1.05, s(2));
        const ridge = createOriginalPart(`zoom_ridge_${{i}}_orig`, ridgeGeo, materials.rubber, lensZoomRing);
        ridge.position.set(Math.cos(angle) * (lzrOD - s(1)), 0, Math.sin(angle) * (lzrOD - s(1))); // Y is height, XZ for circle
        ridge.rotation.y = angle + Math.PI/2; // Align ridge length with ring's local Y
    }}

    // --- Lens Focus Ring ---
    // JSON: lens_focus_ring
    // Shape: 15mm long, 66mm OD, knurled
    // Reasoning: Manual focus, separated from zoom ring
    const lensFocusRingPivot = createOriginalPart("lensFocusRingPivot_orig", null, null, lensBarrelBase, false);
    // Zoom ring ends at: lensZoomRingPivot.position.y + lzrLength (in barrel base's frame)
    lensFocusRingPivot.position.y = lensZoomRingPivot.position.y + lzrLength + s(1);
    lensFocusRingPivot.position.z = 0;

    const lfrLength = s(15), lfrOD = s(66/2);
    const lfrGeo = new THREE.CylinderGeometry(lfrOD, lfrOD, lfrLength, 32);
    const lensFocusRing = createOriginalPart("lens_focus_ring_orig", lfrGeo, materials.lensRing, lensFocusRingPivot);
    lensFocusRing.position.y = lfrLength/2;
    lensFocusRing.rotation.y = Math.PI / 2;

    // --- Lens Front Element ---
    // JSON: lens_front_element
    // Shape: Convex glass disc, 44mm aperture, 2mm rim, 4mm bulge
    // Reasoning: Optical realism, non-moving
    const lfeAperture = s(44/2), lfeRim = s(2);
    const lfeGeo = new THREE.CylinderGeometry(lfeAperture, lfeAperture, lfeRim, 32);
    const lensFrontElement = createOriginalPart("lens_front_element_orig", lfeGeo, materials.glass, lensBarrelBase);
    // Position: Seated at front end of lens_barrel_base. Front end of barrel is at +lbbLength/2 in its local Y.
    lensFrontElement.position.y = lbbLength/2 + lfeRim/2; // Relative to lens_barrel_base (after its rotation)
    lensFrontElement.rotation.y = Math.PI/2; // Align with barrel's orientation

    // Protective bezel, as described in JSON
    const bezelGeo = new THREE.CylinderGeometry(lfeAperture + s(0.5), lfeAperture + s(0.5), s(1) + lfeRim, 32, 1, true);
    const bezel = createOriginalPart("lens_front_bezel_orig", bezelGeo, materials.lensBody, lensFrontElement);
    bezel.position.z = 0; // Centered with glass, assuming glass is Z-thin after its rotation
    bezel.rotation.y = Math.PI/2;
    // --- Lens Cap ---
    // JSON: lens_cap
    // Shape: 6mm deep, 70mm OD, spring clips
    // Reasoning: Removable accessory, animatable
    const lensCapPivot = createOriginalPart("lensCapPivot_orig", null, null, lensFrontElement, false);
    // Cap pivot at the front of the bezel. Bezel protrudes (s(1)+lfeRim)/2 from lfe center.
    // lfeGeo is lfeRim thick along its local Y (which becomes world Z after rotation of lfe).
    // Bezel Geo is s(1)+lfeRim thick.
    // Cap pivot should be at the front face of bezel.
    lensCapPivot.position.z = (s(1)+lfeRim)/2; // Bezel extends this much from lfe center along world Z

    const lcDepth = s(6), lcOD = s(70/2);
    const lcGeo = new THREE.CylinderGeometry(lcOD, lcOD, lcDepth, 32);
    const lensCap = createOriginalPart("lens_cap_orig", lcGeo, materials.plastic, lensCapPivot);
    lensCap.position.z = lcDepth/2 - s(5); // Cap's center, s(2) behind its front face which is at pivot
    lensCap.rotation.y = Math.PI/2; // Align with camera Z

    // --- Shutter Button ---
    // JSON: shutter_button
    // Shape: Tapered cone, 8mm tall, 10mm/12mm diameters
    // Reasoning: Primary user control, animatable travel
    const shutterButtonPivot = createOriginalPart("shutterButtonPivot_orig", null, null, cameraBody, false);
    const sbHeight = s(8), sbTopR = s(10/2), sbBaseR = s(12/2);
    const gripFrontZ = handGrip.position.z + gripThickness/2;
    const topDeckY = bodyHeight/2;
    shutterButtonPivot.position.set(handGrip.position.x, topDeckY - s(10) - sbHeight/2, gripFrontZ - s(12));
    shutterButtonPivot.rotation.x = -Math.PI / 12;

    const sbGeo = new THREE.CylinderGeometry(sbTopR, sbBaseR, sbHeight, 16);
    const shutterButton = createOriginalPart("shutter_button_orig", sbGeo, materials.metal, shutterButtonPivot);
    // shutterButton.position.y is 0 in original, animated via pivot.userData.travel

    // --- Mode Dial ---
    // JSON: mode_dial
    // Shape: 24mm diameter, 7mm tall, knurled, engraved icons
    // Reasoning: Selects shooting mode, rotational articulation
    const modeDialPivot = createOriginalPart("modeDialPivot_orig", null, null, cameraBody, false);
    const mdD = s(24/2), mdHeight = s(7);
    modeDialPivot.position.set(-s(20), bodyHeight/2 + mdHeight/2, 0);

    const mdGeo = new THREE.CylinderGeometry(mdD, mdD, mdHeight, 16);
    const modeDial = createOriginalPart("mode_dial_orig", mdGeo, materials.plastic, modeDialPivot);

    // --- Hot Shoe Mount ---
    // JSON: hot_shoe_mount
    // Shape: U-rail, 18x3x20mm, 5-pin contact
    // Reasoning: Accepts external flashes, static
    const hsWidth = s(18), hsHeight = s(3), hsLength = s(20);
    const hsGeo = new THREE.BoxGeometry(hsWidth, hsHeight, hsLength);
    const hotShoeMount = createOriginalPart("hot_shoe_mount_orig", hsGeo, materials.metal, cameraBody);
    hotShoeMount.position.set(0, bodyHeight/2 + hsHeight/2, -bodyDepth/4 - s(8) - hsLength/2);

    // --- Flash Hinge Arm & Unit ---
    // JSON: flash_hinge_arm, flash_unit_body
    // Shape: Arm 4x10x22mm, hinge boss; unit 40x12x18mm, diffuser
    // Reasoning: Enables pop-up flash articulation
    const flashHingeArmPivot = createOriginalPart("flashHingeArmPivot_orig", null, null, cameraBody, false);
    flashHingeArmPivot.position.set(hotShoeMount.position.x, hotShoeMount.position.y, hotShoeMount.position.z - hsLength/2 - s(6/2));

    const fhaThick = s(4), fhaWidth = s(10), fhaLength = s(22);
    const fhaGeo = new THREE.BoxGeometry(fhaWidth, fhaThick, fhaLength); // X,Y,Z for BoxGeometry
    const flashHingeArm = createOriginalPart("flash_hinge_arm_orig", fhaGeo, materials.plastic, flashHingeArmPivot);
    flashHingeArm.position.z = fhaLength/2; // Arm extends forward from pivot

    const fubWidth = s(40), fubHeight = s(12), fubDepth = s(18);
    const fubGeo = new THREE.BoxGeometry(fubWidth, fubHeight, fubDepth);
    const flashUnitBody = createOriginalPart("flash_unit_body_orig", fubGeo, materials.plastic, flashHingeArm);
    flashUnitBody.position.set(0, fhaThick/2 + fubHeight/2, fhaLength/2 - fubDepth/2);

    // Diffuser panel, as per JSON
    const diffuserGeo = new THREE.PlaneGeometry(fubWidth * 0.9, fubHeight*0.8);
    const diffuser = createOriginalPart("flash_diffuser_orig", diffuserGeo, materials.diffuser, flashUnitBody);
    diffuser.position.z = fubDepth/2 + s(0.1);

    // --- Viewfinder Eyepiece ---
    // JSON: viewfinder_eyepiece
    // Shape: Torus, 28mm outer, 18mm inner, 6mm thick, glass plate
    // Reasoning: Fixed, distinct geometry for composing image
    const veOuterR = s(28/2), veInnerR_tube = s((28-18)/2 / 2);
    const veGeo = new THREE.TorusGeometry(veOuterR - veInnerR_tube, veInnerR_tube, 8, 24);
    const viewfinderEyepiece = createOriginalPart("viewfinder_eyepiece_orig", veGeo, materials.rubber, cameraBody);
    viewfinderEyepiece.position.set(0, bodyHeight/2 - s(10), -bodyDepth/2 - s(6/2));
    viewfinderEyepiece.rotation.x = Math.PI/2;

    // --- LCD Hinge Arm ---
    // JSON: lcd_hinge_arm
    // Shape: 2x30x6mm bracket, pivot pin
    // Reasoning: First axis for LCD articulation
    const lcdHingeArmPivotY = createOriginalPart("lcdHingeArmPivotY_orig", null, null, cameraBody, false);
    const lhaThick = s(2), lhaHeight = s(30), lhaWidth = s(6);
    lcdHingeArmPivotY.position.set(-bodyWidth/2 + lhaThick/2, 0, -bodyDepth/2);

    const lhaGeo = new THREE.BoxGeometry(lhaThick, lhaHeight, lhaWidth);
    const lcdHingeArm = createOriginalPart("lcd_hinge_arm_orig", lhaGeo, materials.plastic, lcdHingeArmPivotY);
    lcdHingeArm.position.x = lhaThick/2; // Arm extends from pivot

    // --- LCD Screen ---
    // JSON: lcd_screen
    // Shape: 70x50x4mm, 64x44mm panel, 3mm bezel
    // Reasoning: Two-axis articulation for flexible viewing
    const lcdScreenPivotX = createOriginalPart("lcdScreenPivotX_orig", null, null, lcdHingeArm, false);
    lcdScreenPivotX.position.set(0, 0, lhaWidth/2 ); // At the end of the arm (local Z of arm)

    const lsWidth = s(70), lsHeight = s(50), lsThick = s(4);
    const lsGeo = new THREE.BoxGeometry(lsWidth, lsHeight, lsThick);
    const lcdScreenBody = createOriginalPart("lcd_screen_body_orig", lsGeo, materials.plastic, lcdScreenPivotX);
    lcdScreenBody.position.x = lsWidth/2;

    // Emissive panel, as per JSON
    const panelGeo = new THREE.PlaneGeometry(s(64), s(44));
    const panel = createOriginalPart("lcd_emissive_panel_orig", panelGeo, materials.screen, lcdScreenBody);
    panel.position.z = lsThick/2 + s(0.1);

    // --- Battery Door ---
    // JSON: battery_door
    // Shape: 28x40x2mm, latch tab, chamfered
    // Reasoning: Hinge for battery access, animatable
    const batteryDoorPivot = createOriginalPart("batteryDoorPivot_orig", null, null, cameraBody, false);
    const bdWidth = s(28), bdLength = s(22), bdThick = s(2);
    batteryDoorPivot.position.set(handGrip.position.x, -bodyHeight/2, handGrip.position.z - gripThickness/2 + gripProtrusion/2 ); // Simplified Z

    const bdGeo = new THREE.BoxGeometry(bdWidth, bdThick, bdLength);
    const batteryDoor = createOriginalPart("battery_door_orig", bdGeo, materials.plastic, batteryDoorPivot);
    batteryDoor.position.y = -bdThick/2;
    batteryDoor.position.z = bdLength/2;

    // --- Tripod Thread Insert ---
    // JSON: tripod_thread_insert
    // Shape: 10mm diameter, 5mm deep, metal, threaded
    // Reasoning: Standard tripod mount, static
    const ttiD = s(10/2), ttiDepth = s(5);
    const ttiGeo = new THREE.CylinderGeometry(ttiD, ttiD, ttiDepth, 12);
    const tripodThreadInsert = createOriginalPart("tripod_thread_insert_orig", ttiGeo, materials.metal, cameraBody);
    tripodThreadInsert.position.set(0, -bodyHeight/2 + ttiDepth/2, batteryDoorPivot.position.z - s(30)); // Z relative to battery door area
    tripodThreadInsert.rotation.x = Math.PI/2;

    // --- Strap Lugs ---
    // JSON: strap_lug_left, strap_lug_right
    // Shape: Horseshoe loop, 14mm outer, 2.5mm rod, 6mm anchor
    // Reasoning: Attachment for camera strap, symmetrical
    const lugAnchorSize = s(6), lugLoopOuterD = s(14), lugLoopRodD = s(2.5);
    function createOriginalStrapLug(namePrefix, sideMultiplier, parent) {{
        const lugGroup = createOriginalPart(`${{namePrefix}}_group_orig`, null, null, parent, false);
        lugGroup.position.set(sideMultiplier * (bodyWidth/2 - lugAnchorSize/2 - s(5)), bodyHeight/2 - s(6) - lugAnchorSize/2, 0);
        if (namePrefix === "strap_lug_right") {{ // Adjust right lug
             lugGroup.position.x = handGrip.position.x;
        }}

        const anchorGeo = new THREE.BoxGeometry(lugAnchorSize, lugAnchorSize, lugAnchorSize);
        const anchor = createOriginalPart(`${{namePrefix}}_anchor_orig`, anchorGeo, materials.metal, lugGroup);

        const loopGeo = new THREE.TorusGeometry(lugLoopOuterD/2 - lugLoopRodD/2, lugLoopRodD/2, 8, 16, Math.PI * 1.5);
        const loop = createOriginalPart(`${{namePrefix}}_loop_orig`, loopGeo, materials.metal, anchor);
        loop.position.x = sideMultiplier * (lugAnchorSize/2 + (lugLoopOuterD/2 - lugLoopRodD/2));
        loop.rotation.z = sideMultiplier * Math.PI/2;
        loop.rotation.y = Math.PI/2;
        return lugGroup;
    }}
    createOriginalStrapLug("strap_lug_left", -1, cameraBody);
    createOriginalStrapLug("strap_lug_right", 1, cameraBody);

    // Update world matrices for all original parts
    originalCameraModelGroup.updateMatrixWorld(true);

    // --- Create New Export Scene Structure ---
    // The following section creates the articulated link structure for animation, matching the JSON's articulation and parent/child relationships.
    const sceneObject = new THREE.Group();
    sceneObject.name = "DigitalCameraExport";
    const newLinks = {{}};

    // Define and Position Link Groups based on original pivots/main components
    // Each link corresponds to a major articulated or animatable part in the JSON (e.g., zoom ring, LCD, flash, battery door)
    const linkDefinitions = {{
        CameraBody: originalParts.camera_body_ref_orig, // The main body box is the reference
        LensZoomRing: originalParts.lensZoomRingPivot_orig,
        LensFocusRing: originalParts.lensFocusRingPivot_orig,
        LensCap: originalParts.lensCapPivot_orig,
        FlashAssembly: originalParts.flashHingeArmPivot_orig,
        LCDHingeArm: originalParts.lcdHingeArmPivotY_orig,
        LCDScreen: originalParts.lcdScreenPivotX_orig,
        BatteryDoor: originalParts.batteryDoorPivot_orig,
    }};

    for (const linkName in linkDefinitions) {{
        const originalReferenceObject = linkDefinitions[linkName];
        const newLinkGroup = new THREE.Group();
        newLinkGroup.name = `link_${{linkName}}`;

        // Set new link group's transform to match the world transform of its original reference object
        // This ensures that the exported link structure matches the articulated hierarchy described in the JSON.
        newLinkGroup.matrix.copy(originalReferenceObject.matrixWorld);
        newLinkGroup.matrix.decompose(newLinkGroup.position, newLinkGroup.quaternion, newLinkGroup.scale);

        sceneObject.add(newLinkGroup);
        newLinks[linkName] = newLinkGroup;
    }}
    
    // Define which original meshes belong to which new link group
    // This mapping is based on the JSON's parent/child and articulation structure.
    const meshToLinkMap = {{
        // CameraBody
        camera_body_main_orig: "CameraBody",
        hand_grip_orig: "CameraBody",
        lens_mount_opening_visual_orig: "CameraBody",
        lens_mount_ring_orig: "CameraBody",
        lens_barrel_base_orig: "CameraBody",
        lens_front_element_orig: "CameraBody",
        lens_front_bezel_orig: "CameraBody",
        shutter_button_orig: "CameraBody",
        mode_dial_orig: "CameraBody",
        hot_shoe_mount_orig: "CameraBody",
        viewfinder_eyepiece_orig: "CameraBody",
        tripod_thread_insert_orig: "CameraBody",
        strap_lug_left_anchor_orig: "CameraBody",
        strap_lug_left_loop_orig: "CameraBody",
        strap_lug_right_anchor_orig: "CameraBody",
        strap_lug_right_loop_orig: "CameraBody",
        // LensMount Pins
        ...Object.fromEntries(Array.from({{length: 8}}, (_, i) => [`lens_mount_pin_${{i}}_orig`, "CameraBody"])),
        // LensZoomRing
        lens_zoom_ring_orig: "LensZoomRing",
        ...Object.fromEntries(Array.from({{length: 12}}, (_, i) => [`zoom_ridge_${{i}}_orig`, "LensZoomRing"])),
        // LensFocusRing
        lens_focus_ring_orig: "LensFocusRing",
        // LensCap
        lens_cap_orig: "LensCap",
        // FlashAssembly
        flash_hinge_arm_orig: "FlashAssembly",
        flash_unit_body_orig: "FlashAssembly",
        flash_diffuser_orig: "FlashAssembly",
        // LCDHingeArm
        lcd_hinge_arm_orig: "LCDHingeArm",
        // LCDScreen
        lcd_screen_body_orig: "LCDScreen",
        lcd_emissive_panel_orig: "LCDScreen",
        // BatteryDoor
        battery_door_orig: "BatteryDoor",
    }};

    // Populate Link Groups with Meshes
    // This loop ensures that each mesh is parented to the correct link group for animation, as described in the JSON articulation reasoning.
    for (const originalMeshName in originalMeshes) {{
        const originalMesh = originalMeshes[originalMeshName];
        const targetLinkName = meshToLinkMap[originalMeshName];

        if (targetLinkName && newLinks[targetLinkName]) {{
            const newMesh = new THREE.Mesh(originalMesh.geometry, originalMesh.material);
            newMesh.name = originalMesh.name.replace("_orig", ""); // Clean up name

            const parentLinkGroup = newLinks[targetLinkName];
            
            // We need parentLinkGroup's matrixWorld to be up-to-date.
            // Since we set it from a world matrix and added it to sceneObject,
            // and sceneObject is at origin, its matrixWorld is what we set.
            // If sceneObject itself was transformed, we'd need sceneObject.updateMatrixWorld(true).
            // For safety, let's ensure world matrices are current.
            sceneObject.updateMatrixWorld(true); // Ensures all link groups' matrixWorld are correct.

            const originalMeshWorldMatrix = originalMesh.matrixWorld.clone();
            const invParentLinkMatrix = parentLinkGroup.matrixWorld.clone().invert();

            newMesh.matrix.multiplyMatrices(invParentLinkMatrix, originalMeshWorldMatrix);
            newMesh.matrix.decompose(newMesh.position, newMesh.quaternion, newMesh.scale);
            
            parentLinkGroup.add(newMesh);
        }} else if (originalMesh.geometry) {{ // Catch meshes not mapped (e.g. if map is incomplete)
            console.warn(`Original mesh "${{originalMeshName}}" was not mapped to a new link group or target link group not found.`);
        }}
    }}
    return sceneObject;
}}

// To make this file runnable with the export script, if it expects 'sceneObject' variable:
// export const sceneObject = createScene();
</js_code_export>

# Example 3: a beautiful laptop
# Input extended description:
[
  {{
    "extended_description": "A beautiful laptop with many keys and a trackpad, featuring a slim profile, a wide screen, and a well-organized keyboard layout. The design balances elegance and usability, with each component articulated for smooth interaction."
  }},
    {{"link_name": "laptop_base",
      "description_shape": "A flat, rectangular base with gently rounded corners, providing a sturdy foundation for the keyboard and trackpad.",
      "description_position": "Forms the bottom part of the laptop, resting on a surface. Serves as the main reference for positioning other components.",
      "reasoning": "The base supports all other components and houses the internal hardware."
    }},
    {{"link_name": "laptop_screen",
      "description_shape": "A thin, wide rectangular panel, proportioned to provide a large display area. The screen is framed by a narrow bezel.",
      "description_position": "Hinged along one long edge of the base, allowing it to open and close like a clamshell.",
      "reasoning": "The screen displays information and is articulated for adjustable viewing angles."
    }},
    {{"link_name": "laptop_trackpad",
      "description_shape": "A smooth, rectangular surface, slightly recessed into the base, positioned below the keyboard area.",
      "description_position": "Centered horizontally on the base, below the keyboard.",
      "reasoning": "The trackpad enables pointer control and gesture input."
    }},
    {{"link_name": "laptop_keyboard",
      "description_shape": "A grid of small, slightly concave rectangular keys arranged in rows, covering most of the upper surface of the base.",
      "description_position": "Occupies the upper central area of the base, just below the hinge.",
      "reasoning": "The keyboard allows text input and interaction with the laptop."
    }},
    {{"link_name": "laptop_key_1",
      "description_shape": "A small, square key with a slightly concave top surface, matching the other keys in the keyboard.",
      "description_position": "Arranged in a grid pattern as part of the keyboard, aligned with neighboring keys.",
      "reasoning": "Each key enables a specific input function."
    }},
    {{"link_name": "laptop_key_2",
      "description_shape": "A small, square key with a slightly concave top surface, matching the other keys in the keyboard.",
      "description_position": "Arranged in a grid pattern as part of the keyboard, aligned with neighboring keys.",
      "reasoning": "Each key enables a specific input function."
    }},
    {{"link_name": "laptop_key_3",
      "description_shape": "A small, square key with a slightly concave top surface, matching the other keys in the keyboard.",
      "description_position": "Arranged in a grid pattern as part of the keyboard, aligned with neighboring keys.",
      "reasoning": "Each key enables a specific input function."
    }},
    {{"link_name": "laptop_key_4",
      "description_shape": "A small, square key with a slightly concave top surface, matching the other keys in the keyboard.",
      "description_position": "Arranged in a grid pattern as part of the keyboard, aligned with neighboring keys.",
      "reasoning": "Each key enables a specific input function."
    }},
    {{"link_name": "laptop_key_5",
      "description_shape": "A small, square key with a slightly concave top surface, matching the other keys in the keyboard.",
      "description_position": "Arranged in a grid pattern as part of the keyboard, aligned with neighboring keys.",
      "reasoning": "Each key enables a specific input function."
    }},
    ......
]

# Output threejs code:
<js_code_export>
import * as THREE from 'three';

export function createScene() {{
    const laptop = new THREE.Group();
    laptop.name = 'laptop'; // Root group name

    // --- Base Part ---
    const base = new THREE.Group();
    base.name = 'base';

    // Base body
    const baseBodyWidth = 0.5;
    const baseBodyHeight = 0.02;
    const baseBodyDepth = 0.3;
    const baseBodyGeometry = new THREE.BoxGeometry(baseBodyWidth, baseBodyHeight, baseBodyDepth);
    const baseBodyMesh = new THREE.Mesh(baseBodyGeometry);
    baseBodyMesh.name = 'baseBody';
    base.add(baseBodyMesh);

    // Keyboard area (slightly inset or different material area)
    const keyboardAreaWidth = baseBodyWidth * 0.92;
    const keyboardAreaHeight = baseBodyHeight * 0.25; // Thinner than base
    const keyboardAreaDepth = baseBodyDepth * 0.65;
    const keyboardAreaGeometry = new THREE.BoxGeometry(keyboardAreaWidth, keyboardAreaHeight, keyboardAreaDepth);
    const keyboardAreaMesh = new THREE.Mesh(keyboardAreaGeometry);
    keyboardAreaMesh.name = 'keyboardArea';
    // Position slightly above the base body, towards the back
    keyboardAreaMesh.position.set(0, baseBodyHeight / 2 + keyboardAreaHeight / 2, -baseBodyDepth * 0.1);
    base.add(keyboardAreaMesh);

    // Trackpad area (recessed)
    const trackpadAreaWidth = baseBodyWidth * 0.25;
    const trackpadAreaHeight = baseBodyHeight * 0.25; // Thinner than base
    const trackpadAreaDepth = baseBodyDepth * 0.25;
    const trackpadAreaGeometry = new THREE.BoxGeometry(trackpadAreaWidth, trackpadAreaHeight, trackpadAreaDepth);
    const trackpadAreaMesh = new THREE.Mesh(trackpadAreaGeometry);
    trackpadAreaMesh.name = 'trackpadArea';
     // Position slightly above the base body, towards the front
    trackpadAreaMesh.position.set(0, baseBodyHeight / 2 + trackpadAreaHeight / 2, baseBodyDepth * 0.3);
    base.add(trackpadAreaMesh);

    // Add some keys (simple small boxes) - added to the base group
    const keyWidth = 0.02;
    const keyHeight = 0.008;
    const keyDepth = 0.02;
    const keyGeometry = new THREE.BoxGeometry(keyWidth, keyHeight, keyDepth);
    const keySpacingX = 0.025;
    const keySpacingZ = 0.025;
    const numRows = 5;
    const numCols = 15;
    // Calculate starting position based on keyboardAreaMesh
    const startX = keyboardAreaMesh.position.x - (numCols - 1) * keySpacingX / 2;
    const startZ = keyboardAreaMesh.position.z - (numRows - 1) * keySpacingZ / 2;
    const keyBaseHeight = baseBodyHeight / 2 + keyHeight / 2; // Position keys on top of the base

    for (let row = 0; row < numRows; row++) {{
        for (let col = 0; col < numCols; col++) {{
            const keyMesh = new THREE.Mesh(keyGeometry);
            keyMesh.name = `key_{{row}}_{{col}}`;
            keyMesh.position.set(
                startX + col * keySpacingX,
                keyBaseHeight,
                startZ + row * keySpacingZ
            );
            base.add(keyMesh); // Add keys directly to the base group
        }}
    }}


    // --- Screen Part ---
    const screen = new THREE.Group();
    screen.name = 'screen';

    // Screen panel (the main display + bezel)
    const screenPanelWidth = baseBodyWidth; // Match base width
    const screenPanelHeight = baseBodyDepth * 1.1; // Taller than base depth
    const screenPanelDepth = 0.008; // Very thin
    const screenPanelGeometry = new THREE.BoxGeometry(screenPanelWidth, screenPanelHeight, screenPanelDepth);
    const screenPanelMesh = new THREE.Mesh(screenPanelGeometry);
    screenPanelMesh.name = 'screenPanel';

    // Position the screen panel mesh relative to the screen group's pivot (hinge).
    // The screen group's pivot is at its origin (0,0,0).
    // We want the bottom edge of the screen panel to align with the hinge when rotated.
    // The panel's center is at screenPanelHeight / 2 from its bottom edge.
    // Its center is at screenPanelDepth / 2 from its back edge.
    // Position the center of the panel mesh such that its bottom edge is at Y=0 of the group, and its back edge is at Z=0 of the group.
    const screenPanelMeshPosY = screenPanelHeight / 2;
    const screenPanelMeshPosZ = screenPanelDepth / 2; // Position slightly in front of the Z=0 hinge axis
    screenPanelMesh.position.set(0, screenPanelMeshPosY, screenPanelMeshPosZ);
    screen.add(screenPanelMesh);

    // Screen surface (recessed inside the panel)
    const screenSurfaceWidth = screenPanelWidth * 0.95;
    const screenSurfaceHeight = screenPanelHeight * 0.9;
    const screenSurfaceDepth = screenPanelDepth * 0.5;
    const screenSurfaceGeometry = new THREE.BoxGeometry(screenSurfaceWidth, screenSurfaceHeight, screenSurfaceDepth);
    const screenSurfaceMesh = new THREE.Mesh(screenSurfaceGeometry);
    screenSurfaceMesh.name = 'screenSurface';
    // Position relative to the screen panel mesh, slightly in front
    screenSurfaceMesh.position.set(
        screenPanelMesh.position.x,
        screenPanelMesh.position.y,
        screenPanelMesh.position.z + screenPanelDepth * 0.5 / 2 // Slightly in front of the panel center
    );
    screen.add(screenSurfaceMesh);

    // Camera bump (small cylinder at top center of screen)
    const cameraRadius = 0.003;
    const cameraHeight = 0.005;
    const cameraSegments = 8;
    const cameraGeometry = new THREE.CylinderGeometry(cameraRadius, cameraRadius, cameraHeight, cameraSegments);
    const cameraMesh = new THREE.Mesh(cameraGeometry);
    cameraMesh.name = 'screenCamera';
    // Position relative to screen panel mesh, near top edge and slightly in front
    cameraMesh.position.set(
        screenPanelMesh.position.x,
        screenPanelMesh.position.y + screenPanelHeight / 2 - cameraHeight / 2 - 0.01, // Near top edge
        screenPanelMesh.position.z + screenPanelDepth / 2 + cameraRadius // Slightly in front of screen surface
    );
    screen.add(cameraMesh);

    // --- Hinge Geometry (optional, just for visual detail) ---
    // This represents the cylindrical part of the hinge, rotates with the screen
    const hingeCylinderRadius = 0.005;
    const hingeCylinderLength = screenPanelWidth * 0.8;
    const hingeCylinderSegments = 16;
    const hingeCylinderGeometry = new THREE.CylinderGeometry(hingeCylinderRadius, hingeCylinderRadius, hingeCylinderLength, hingeCylinderSegments);
    const hingeCylinderMesh = new THREE.Mesh(hingeCylinderGeometry);
    hingeCylinderMesh.name = 'hingeCylinder';
    hingeCylinderMesh.rotation.z = Math.PI / 2; // Rotate to be horizontal
    // Position it at the pivot point (screen group origin)
    hingeCylinderMesh.position.set(0, 0, 0); // Position relative to the screen group's pivot
    screen.add(hingeCylinderMesh);


    // --- Position the screen group relative to the base ---
    // The hinge is at the back edge of the base (z = -baseBodyDepth / 2)
    // and at the top surface of the base (y = baseBodyHeight / 2).
    const hingeZ = -baseBodyDepth / 2;
    const hingeY = baseBodyHeight / 2;
    screen.position.set(0, hingeY, hingeZ); // Position the screen's pivot at the base's hinge location

    // --- Set initial screen rotation (partially open) ---
    // Rotate the screen group around its Y axis (the hinge axis)
    screen.rotation.x = -Math.PI / 4; // Rotate 45 degrees back

    // --- Add parts to the main laptop group ---
    laptop.add(base);
    laptop.add(screen);

    return laptop;
}}
</js_code_export>
'''