# combine_meshes.py
import trimesh
import os
import argparse
import glob

def combine_meshes(input_dir, output_file, pattern="*.obj;*.stl"):
    """
    Loads mesh files from a directory, combines them, and saves as a single file.
    Searches recursively in subdirectories to find mesh files.
    """
    mesh_list = []
    all_files = []

    # Find all mesh files recursively
    for p in pattern.split(';'):
        p = p.strip()
        # Search recursively using ** pattern
        search_path = os.path.join(input_dir, "**", p)
        all_files.extend(glob.glob(search_path, recursive=True))

    # Load each mesh file
    for file_path in all_files:
        mesh = trimesh.load_mesh(file_path, process=False)
        
        # Handle Scene objects
        if isinstance(mesh, trimesh.Scene):
            mesh = trimesh.util.concatenate(mesh.dump())
        
        mesh_list.append(mesh)

    # Combine all meshes
    combined_mesh = trimesh.util.concatenate(mesh_list)
    
    # Create output directory if needed
    output_dir = os.path.dirname(output_file)
    if output_dir:
        os.makedirs(output_dir, exist_ok=True)

    # Export mesh (silently)
    if hasattr(combined_mesh, 'visual'):
        combined_mesh.visual = trimesh.visual.ColorVisuals(combined_mesh)
    
    combined_mesh.export(output_file, include_texture=False, include_color=False)


# --- Main Execution ---
if __name__ == "__main__":
    parser = argparse.ArgumentParser(description="Combine multiple mesh files from a directory into a single mesh file.")
    parser.add_argument("input_dir", help="Directory containing the individual mesh files.")
    parser.add_argument("output_file", help="Path for the combined output mesh file.")
    parser.add_argument("-p", "--pattern", default="*.obj;*.stl", help="Glob patterns (default: '*.obj;*.stl')")

    args = parser.parse_args()
    combine_meshes(args.input_dir, args.output_file, args.pattern)