"""
Modern ArticulationGeneratorAgent using the unified BaseAgent system
Generates joint/articulation specifications from workflow and code blocks
"""

import os
import re
import json
import logging
from typing import Dict, Any, List
from agents.base_agent import BaseAgent
from utils.output_parser import OutputFormatError


class ArticulationOutput:
    """Utility class for parsing articulation output from LLM responses"""
    
    @staticmethod
    def extract_from_response(text: str) -> List[Dict[str, Any]]:
        """
        Extract articulation specification from LLM response
        
        Args:
            text: Raw LLM response text
            
        Returns:
            Parsed articulation specification as list of joint dictionaries
            
        Raises:
            OutputFormatError: If articulation spec cannot be extracted
        """
        # Clean up markdown formatting
        text = text.strip()
        if text.startswith('```json'):
            text = text[7:]
        if text.startswith('```'):
            text = text[3:]
        if text.endswith('```'):
            text = text[:-3]
        
        # Try to find JSON array in the response
        array_matches = re.findall(r'\[.*?\]', text, re.DOTALL)
        
        for match in array_matches:
            # Clean up comments and trailing commas
            cleaned = re.sub(r'//.*', '', match)  # Remove comments
            cleaned = re.sub(r',\s*([}\]])', r'\1', cleaned)  # Remove trailing commas
            
            try:
                parsed = json.loads(cleaned)
                # Only accept a list of joint dicts with required keys
                if (
                    isinstance(parsed, list) and len(parsed) > 0 and
                    all(isinstance(item, dict) for item in parsed) and
                    all('joint_name' in item and 'parent' in item and 'child' in item and 'type' in item for item in parsed)
                ):
                    return parsed
            except json.JSONDecodeError:
                continue
        
        # Fallback: try to find individual JSON objects and collect them
        json_matches = re.findall(r'\{[^{}]*(?:\{[^{}]*\}[^{}]*)*\}', text, re.DOTALL)
        
        if json_matches:
            joints = []
            for match in json_matches:
                cleaned = re.sub(r'//.*', '', match)
                cleaned = re.sub(r',\s*([}\]])', r'\1', cleaned)
                try:
                    joint = json.loads(cleaned)
                    if isinstance(joint, dict) and 'joint_name' in joint:
                        joints.append(joint)
                except json.JSONDecodeError:
                    continue
            
            if joints:
                return joints
        
        raise OutputFormatError(
            f"No valid JSON array or joint objects found in articulation response.\n"
            f"Content: {text[:500]}..."
        )


class ArticulationGeneratorAgent(BaseAgent):
    """
    Agent for generating articulation (joint) specifications.
    
    Takes workflow JSON and code blocks to generate joint definitions
    that specify how links are connected and can move.
    """
    
    def __init__(self, config_manager):
        """
        Initialize the ArticulationGeneratorAgent
        
        Args:
            config_manager: Configuration manager instance
        """
        super().__init__(config_manager, 'articulation_generator')
        self.logger = logging.getLogger(self.__class__.__name__)
        
    def _load_system_prompt(self) -> str:
        """Load system prompt for articulation generation"""
        try:
            from prompt.articulationer import system_prompt as articulation_system_prompt
            from prompt.examples.articulationer_samples import samples as articulation_samples
            
            # Combine system prompt with examples
            return articulation_system_prompt + '\n' + articulation_samples
            
        except ImportError as e:
            self.logger.error(f"Failed to load articulation prompts: {e}")
            return (
                "Generate articulation (joint) specifications for the given "
                "workflow and code blocks. Specify how links connect and move."
            )
    
    def _format_user_prompt(self, input_data: Dict[str, Any]) -> str:
        """
        Format user prompt for articulation generation
        
        Args:
            input_data: Dictionary containing workflow_json and group_code_blocks
            
        Returns:
            Formatted user prompt
        """
        workflow_json = input_data.get('workflow_json', {})
        group_code_blocks = input_data.get('group_code_blocks', '')
        
        # Format workflow JSON as string
        workflow_str = json.dumps(workflow_json, indent=2, ensure_ascii=False)
        
        return (
            f"Given the following workflow.json and code blocks for each group, "
            f"generate the articulation (joint) specification as instructed.\n\n"
            f"workflow.json:\n{workflow_str}\n\n"
            f"[GROUP CODE BLOCKS]\n{group_code_blocks}"
        )
    
    def parse_response(self, response: str) -> List[Dict[str, Any]]:
        """
        Parse LLM response into structured articulation data
        
        Args:
            response: Raw LLM response
            
        Returns:
            Parsed articulation specification as list of joint dictionaries
            
        Raises:
            OutputFormatError: If response format is invalid
        """
        try:
            return ArticulationOutput.extract_from_response(response)
        except OutputFormatError as e:
            self.logger.error(f"Failed to parse articulation response: {e}")
            raise
    
    def _prepare_input_data(self, workflow_json: str, 
                           export_js_path: str = None,
                           group_code_blocks: str = None, **kwargs) -> Dict[str, Any]:
        """
        Prepare input data from method arguments
        
        Args:
            workflow_json: Path to workflow.json file
            export_js_path: Path to export.js file 
            group_code_blocks: Pre-extracted code blocks (optional)
            **kwargs: Additional arguments
            
        Returns:
            Dictionary of input data for the agent
        """
        # Read workflow.json from path (standard format only)
        with open(workflow_json, 'r', encoding='utf-8') as f:
            workflow_data = json.load(f)
        
        input_data = {'workflow_json': workflow_data}
        
        # Extract code blocks from export.js
        if export_js_path and os.path.exists(export_js_path):
            # Extract group names from workflow's threejs_info
            group_names = self._extract_group_names(workflow_data)
            code_blocks = self._extract_group_code_blocks(export_js_path, group_names)
            input_data['group_code_blocks'] = code_blocks
        else:
            input_data['group_code_blocks'] = ""
        
        return input_data
    
    def _extract_group_names(self, workflow_json: Dict[str, Any]) -> List[str]:
        """
        Extract group names from workflow JSON threejs_info
        
        Args:
            workflow_json: Workflow specification with threejs_info
            
        Returns:
            List of group names
        """
        # Only handle standard format: threejs_info contains group names
        group_names = []
        for item in workflow_json.get('threejs_info', []):
            if isinstance(item, dict) and item.get('name'):
                group_names.append(item['name'])
        
        return group_names
    
    def _extract_group_code_blocks(self, export_js_path: str, group_names: List[str]) -> str:
        """
        Extract code blocks for each group from export.js file
        
        Args:
            export_js_path: Path to the export.js file
            group_names: List of group names to extract
            
        Returns:
            Formatted string containing all group code blocks
        """
        try:
            with open(export_js_path, 'r', encoding='utf-8') as f:
                code = f.read()
        except (FileNotFoundError, IOError) as e:
            self.logger.warning(f"Could not read export.js file: {e}")
            return "Code blocks not available - file read error"
        
        blocks = []
        
        for group in group_names:
            # Try to find the group definition
            pattern = rf'(const|let|var)\s+{re.escape(group)}\s*=\s*new\s+THREE\.Group\(\);.*?\n(.*?)(?=\n(const|let|var)\s+|$)'
            match = re.search(pattern, code, re.DOTALL)
            
            if match:
                blocks.append(f"// Group: {group}\n" + match.group(0))
            else:
                blocks.append(f"// Group: {group}\n// Code not found in export.js")
        
        return '\n\n'.join(blocks) if blocks else "No group code blocks found"
    
    def save_output(self, result: List[Dict[str, Any]], output_folder: str, 
                   metrics: Dict[str, Any] = None):
        """
        Save generated articulation specification to configs folder
        
        Args:
            result: Generated articulation data (list of joint dictionaries)
            output_folder: Directory to save output
            metrics: Generation metrics (optional)
        """
        # Create configs folder
        configs_folder = os.path.join(output_folder, 'configs')
        os.makedirs(configs_folder, exist_ok=True)
        
        # Save the main articulation output to configs
        output_path = os.path.join(configs_folder, 'articulation.json')
        with open(output_path, 'w', encoding='utf-8') as f:
            json.dump(result, f, indent=2, ensure_ascii=False)
        
        # Note: Metadata is now saved in pipeline_logs by BaseAgent
        
        self.logger.info(f"Saved articulation specification to {output_path}")
