"""
This file contains examples of Three.js code for visualization and exporting.
We can use these in-context examples to generate the code for the object.
"""


"""
## Example:

# !!!! First example for visualization !!!!
import * as THREE from 'three';
import { OrbitControls } from 'three/addons/controls/OrbitControls.js';
import TWEEN from '@tweenjs/tween.js';

let scene, camera, renderer, controls;
let cabinetUnit, drawerGroup, leftDoorGroup, rightDoorGroup;
let raycaster, mouse;
const interactiveObjects = []; // Array to hold objects that can be clicked

// --- State Variables ---
let isDrawerOpen = false;
let isLeftDoorOpen = false;
let isRightDoorOpen = false;
let isDemoAnimating = false; // Prevent multiple demo animations at once

// --- Constants for Dimensions ---
const cabinetWidth = 1.2;
const cabinetHeight = 0.8;
const cabinetDepth = 0.5;
const woodThickness = 0.02; // Main cabinet frame thickness
const drawerHeight = 0.2;
const handleDepth = 0.03;
const handleWidth = 0.1;
const handleHeight = 0.02;
const doorGap = 0.005; // Small gap between doors/drawer front
const drawerPanelThickness = 0.015; // Thickness for drawer box panels (can be thinner)
const drawerInset = 0.01; // Small inset from the main cabinet depth

// --- Animation Parameters ---
const animationDuration = 500; // milliseconds for open/close
const demoAnimationDuration = 750; // Slightly longer for demonstration
// Calculate drawer closed position once
const drawerBoxDepth = cabinetDepth - woodThickness - drawerInset;
const drawerClosedZ = cabinetDepth / 2 - drawerBoxDepth / 2;
const drawerOpenDistance = drawerBoxDepth * 0.75; // How far it opens interactively
const drawerDemoDistance = drawerBoxDepth * 0.5; // How far it moves for demo
const doorOpenAngle = Math.PI * 0.45; // Angle for interactive open
const doorDemoAngle = Math.PI * 0.3; // Angle for demo swing

init();
animate();

function init() {
    // --- Scene ---
    scene = new THREE.Scene();
    scene.background = new THREE.Color(0xf0f0f0);

    // --- Camera ---
    camera = new THREE.PerspectiveCamera(60, window.innerWidth / window.innerHeight, 0.1, 100);
    camera.position.set(1, 1, 1.5);
    camera.lookAt(0, 0, 0);

    // --- Renderer ---
    renderer = new THREE.WebGLRenderer({ antialias: true });
    renderer.setSize(window.innerWidth, window.innerHeight);
    renderer.shadowMap.enabled = true;
    renderer.shadowMap.type = THREE.PCFSoftShadowMap;
    document.body.appendChild(renderer.domElement);

    // --- Controls ---
    controls = new OrbitControls(camera, renderer.domElement);
    controls.enableDamping = true;
    controls.dampingFactor = 0.05;
    controls.screenSpacePanning = false;
    controls.minDistance = 0.5;
    controls.maxDistance = 5;
    controls.target.set(0, cabinetHeight / 3, 0);
    controls.update();

    // --- Lighting ---
    const ambientLight = new THREE.AmbientLight(0xffffff, 0.6);
    scene.add(ambientLight);

    const directionalLight = new THREE.DirectionalLight(0xffffff, 1.5);
    directionalLight.position.set(3, 5, 4);
    directionalLight.castShadow = true;
    directionalLight.shadow.mapSize.width = 1024;
    directionalLight.shadow.mapSize.height = 1024;
    directionalLight.shadow.camera.near = 0.5;
    directionalLight.shadow.camera.far = 15;
    directionalLight.shadow.camera.left = -2;
    directionalLight.shadow.camera.right = 2;
    directionalLight.shadow.camera.top = 2;
    directionalLight.shadow.camera.bottom = -2;
    scene.add(directionalLight);

    // --- Ground Plane (Optional) ---
    const groundGeo = new THREE.PlaneGeometry(10, 10);
    const groundMat = new THREE.MeshStandardMaterial({ color: 0xcccccc, side: THREE.DoubleSide });
    const ground = new THREE.Mesh(groundGeo, groundMat);
    ground.rotation.x = -Math.PI / 2;
    ground.position.y = 0; // Align ground with cabinet base if unit is shifted
    ground.receiveShadow = true;
    scene.add(ground);


    // --- Create Cabinet ---
    cabinetUnit = createCabinet();
    // Shift the entire unit up so its base is at y=0
    cabinetUnit.position.y = cabinetHeight / 2;
    scene.add(cabinetUnit);

    // --- Interaction Setup ---
    raycaster = new THREE.Raycaster();
    mouse = new THREE.Vector2();
    // window.addEventListener('click', onClick);
    // window.addEventListener('keydown', onKeyDown); // <-- Add key listener

    // --- Resize Listener ---
    // window.addEventListener('resize', onWindowResize);
}

function createCabinet() {
    const unitGroup = new THREE.Group();
    // *** NOTE: unitGroup.position.y is set AFTER creation in init() ***

    // --- Materials ---
    const woodMaterial = new THREE.MeshStandardMaterial({ color: 0x8B4513, roughness: 0.8, metalness: 0.1 });
    const drawerWoodMaterial = new THREE.MeshStandardMaterial({ color: 0x965A21, roughness: 0.85, metalness: 0.1 });
    const handleMaterial = new THREE.MeshStandardMaterial({ color: 0xA9A9A9, roughness: 0.4, metalness: 0.8 });

    // Helper function for creating panels
    const createPanel = (w, h, d, mat) => {
        const geometry = new THREE.BoxGeometry(w, h, d);
        const mesh = new THREE.Mesh(geometry, mat);
        mesh.castShadow = true;
        mesh.receiveShadow = true;
        return mesh;
    };

    // --- Static Cabinet Body ---
    const bodyGroup = new THREE.Group();
    unitGroup.add(bodyGroup);

    // Top Panel
    const topPanel = createPanel(cabinetWidth, woodThickness, cabinetDepth, woodMaterial);
    topPanel.position.y = cabinetHeight / 2 - woodThickness / 2;
    bodyGroup.add(topPanel);

    // Bottom Panel
    const bottomPanel = createPanel(cabinetWidth, woodThickness, cabinetDepth, woodMaterial);
    bottomPanel.position.y = -cabinetHeight / 2 + woodThickness / 2;
    bodyGroup.add(bottomPanel);

    // Left Side Panel
    const leftSide = createPanel(woodThickness, cabinetHeight, cabinetDepth, woodMaterial);
    leftSide.position.x = -cabinetWidth / 2 + woodThickness / 2;
    bodyGroup.add(leftSide);

    // Right Side Panel
    const rightSide = createPanel(woodThickness, cabinetHeight, cabinetDepth, woodMaterial);
    rightSide.position.x = cabinetWidth / 2 - woodThickness / 2;
    bodyGroup.add(rightSide);

    // Back Panel (slightly inset)
    const backPanel = createPanel(cabinetWidth - 2 * woodThickness, cabinetHeight - 2 * woodThickness, woodThickness, woodMaterial);
    backPanel.position.z = -cabinetDepth / 2 + woodThickness / 2;
    backPanel.position.y = 0;
    bodyGroup.add(backPanel);

    // Internal Horizontal Shelf
    const shelfY = cabinetHeight / 2 - woodThickness - drawerHeight - woodThickness / 2;
    const horizontalShelf = createPanel(cabinetWidth - 2 * woodThickness, woodThickness, cabinetDepth - woodThickness, woodMaterial);
    horizontalShelf.position.y = shelfY;
    horizontalShelf.position.z = 0;
    bodyGroup.add(horizontalShelf);

    // Internal Vertical Divider
    const dividerHeight = cabinetHeight - drawerHeight - 3 * woodThickness;
    const dividerY = -cabinetHeight / 2 + woodThickness + dividerHeight / 2;
    const verticalDivider = createPanel(woodThickness, dividerHeight, cabinetDepth - woodThickness, woodMaterial);
    verticalDivider.position.y = dividerY;
    verticalDivider.position.z = 0;
    bodyGroup.add(verticalDivider);

    // --- Drawer Box ---
    drawerGroup = new THREE.Group();
    const drawerInnerWidth = cabinetWidth - 2 * woodThickness - 2 * doorGap;
    const drawerInnerHeight = drawerHeight - 2 * doorGap;

    // Front Panel
    const drawerFront = createPanel(drawerInnerWidth, drawerInnerHeight, drawerPanelThickness, woodMaterial);
    drawerFront.position.z = drawerBoxDepth / 2 - drawerPanelThickness / 2;
    drawerFront.name = 'drawerFrontMesh'; // Name for click detection
    drawerGroup.add(drawerFront);

    // Bottom Panel
    const drawerBottom = createPanel(drawerInnerWidth, drawerPanelThickness, drawerBoxDepth - drawerPanelThickness, drawerWoodMaterial);
    drawerBottom.position.y = -drawerInnerHeight / 2 + drawerPanelThickness / 2;
    drawerGroup.add(drawerBottom);

    // Back Panel
    const drawerBack = createPanel(drawerInnerWidth, drawerInnerHeight - drawerPanelThickness, drawerPanelThickness, drawerWoodMaterial);
    drawerBack.position.z = -drawerBoxDepth / 2 + drawerPanelThickness / 2;
    drawerBack.position.y = drawerPanelThickness / 2;
    drawerGroup.add(drawerBack);

    // Left Panel
    const drawerLeft = createPanel(drawerPanelThickness, drawerInnerHeight - drawerPanelThickness, drawerBoxDepth - drawerPanelThickness, drawerWoodMaterial);
    drawerLeft.position.x = -drawerInnerWidth / 2 + drawerPanelThickness / 2;
    drawerLeft.position.y = drawerPanelThickness / 2;
    drawerGroup.add(drawerLeft);

    // Right Panel
    const drawerRight = createPanel(drawerPanelThickness, drawerInnerHeight - drawerPanelThickness, drawerBoxDepth - drawerPanelThickness, drawerWoodMaterial);
    drawerRight.position.x = drawerInnerWidth / 2 - drawerPanelThickness / 2;
    drawerRight.position.y = drawerPanelThickness / 2;
    drawerGroup.add(drawerRight);

    // Drawer Handle
    const drawerHandle = createPanel(handleWidth, handleHeight, handleDepth, handleMaterial);
    drawerHandle.position.z = drawerFront.position.z + drawerPanelThickness / 2 + handleDepth / 2;
    drawerHandle.position.y = 0;
    drawerGroup.add(drawerHandle);

    // Position the entire drawer group
    drawerGroup.position.y = cabinetHeight / 2 - woodThickness - drawerHeight / 2;
    drawerGroup.position.z = drawerClosedZ; // Use calculated closed Z
    unitGroup.add(drawerGroup);
    interactiveObjects.push(drawerFront); // Add only front for clicking

    // --- Cabinet Doors ---
    const doorHeight = cabinetHeight - drawerHeight - 3 * woodThickness - 2 * doorGap;
    const doorWidth = (cabinetWidth - 3 * woodThickness) / 2 - doorGap;

    // Left Door
    leftDoorGroup = new THREE.Group();
    const leftDoor = createPanel(doorWidth, doorHeight, woodThickness, woodMaterial);
    leftDoor.position.x = doorWidth / 2;
    leftDoor.position.z = woodThickness / 2;
    leftDoor.name = 'leftDoorMesh'; // Name for click detection
    const leftHandle = createPanel(handleHeight, handleWidth, handleDepth, handleMaterial);
    leftHandle.position.x = doorWidth - handleHeight / 2 - doorGap;
    leftHandle.position.z = woodThickness + handleDepth / 2;
    leftDoorGroup.add(leftDoor);
    leftDoorGroup.add(leftHandle);
    leftDoorGroup.position.x = -cabinetWidth / 2 + woodThickness;
    leftDoorGroup.position.y = dividerY;
    leftDoorGroup.position.z = cabinetDepth / 2 - woodThickness / 2;
    unitGroup.add(leftDoorGroup);
    interactiveObjects.push(leftDoor); // Add only door panel for clicking

    // Right Door
    rightDoorGroup = new THREE.Group();
    const rightDoor = createPanel(doorWidth, doorHeight, woodThickness, woodMaterial);
    rightDoor.position.x = -doorWidth / 2;
    rightDoor.position.z = woodThickness / 2;
    rightDoor.name = 'rightDoorMesh'; // Name for click detection
    const rightHandle = createPanel(handleHeight, handleWidth, handleDepth, handleMaterial);
    rightHandle.position.x = -doorWidth + handleHeight / 2 + doorGap;
    rightHandle.position.z = woodThickness + handleDepth / 2;
    rightDoorGroup.add(rightDoor);
    rightDoorGroup.add(rightHandle);
    rightDoorGroup.position.x = cabinetWidth / 2 - woodThickness;
    rightDoorGroup.position.y = dividerY;
    rightDoorGroup.position.z = cabinetDepth / 2 - woodThickness / 2;
    unitGroup.add(rightDoorGroup);
    interactiveObjects.push(rightDoor); // Add only door panel for clicking

    return unitGroup;
}

function onWindowResize() {
    camera.aspect = window.innerWidth / window.innerHeight;
    camera.updateProjectionMatrix();
    renderer.setSize(window.innerWidth, window.innerHeight);
}

function onClick(event) {
    // Don't allow clicks if a demo is running
    if (isDemoAnimating) return;

    mouse.x = (event.clientX / window.innerWidth) * 2 - 1;
    mouse.y = -(event.clientY / window.innerHeight) * 2 + 1;
    raycaster.setFromCamera(mouse, camera);
    const intersects = raycaster.intersectObjects(interactiveObjects, false);

    if (intersects.length > 0) {
        const clickedObjectName = intersects[0].object.name;

        if (clickedObjectName === 'drawerFrontMesh') {
             toggleDrawer();
        } else if (clickedObjectName === 'leftDoorMesh') {
             toggleLeftDoor();
        } else if (clickedObjectName === 'rightDoorMesh') {
             toggleRightDoor();
        }
    }
}

// --- NEW KeyDown Listener ---
function onKeyDown(event) {
    // Don't allow new demos if one is already running
    if (isDemoAnimating) return;

    switch (event.key.toLowerCase()) {
        case 'd':
            demonstrateDrawerMotion();
            break;
        case 'l':
            demonstrateLeftDoorMotion();
            break;
        case 'r':
            demonstrateRightDoorMotion();
            break;
    }
}


// --- Interaction Animation Functions ---

function toggleDrawer() {
    const targetZ = isDrawerOpen ? drawerClosedZ : drawerClosedZ + drawerOpenDistance;
    const startPos = { z: drawerGroup.position.z };
    const targetPos = { z: targetZ };

    new TWEEN.Tween(startPos)
        .to(targetPos, animationDuration)
        .easing(TWEEN.Easing.Quadratic.Out)
        .onUpdate(() => { drawerGroup.position.z = startPos.z; })
        .onComplete(() => { isDrawerOpen = !isDrawerOpen; }) // Toggle state on completion
        .start();
    // Note: State is toggled in onComplete now to reflect final state
}

function toggleLeftDoor() {
    const targetAngle = isLeftDoorOpen ? 0 : doorOpenAngle;
    const startRot = { y: leftDoorGroup.rotation.y };
    const targetRot = { y: targetAngle };

    new TWEEN.Tween(startRot)
        .to(targetRot, animationDuration)
        .easing(TWEEN.Easing.Quadratic.Out)
        .onUpdate(() => { leftDoorGroup.rotation.y = startRot.y; })
        .onComplete(() => { isLeftDoorOpen = !isLeftDoorOpen; }) // Toggle state on completion
        .start();
}

function toggleRightDoor() {
    const targetAngle = isRightDoorOpen ? 0 : -doorOpenAngle; // Negative angle
    const startRot = { y: rightDoorGroup.rotation.y };
    const targetRot = { y: targetAngle };

    new TWEEN.Tween(startRot)
        .to(targetRot, animationDuration)
        .easing(TWEEN.Easing.Quadratic.Out)
        .onUpdate(() => { rightDoorGroup.rotation.y = startRot.y; })
         .onComplete(() => { isRightDoorOpen = !isRightDoorOpen; }) // Toggle state on completion
        .start();
}

// --- NEW Demonstration Animation Functions ---

function demonstrateDrawerMotion() {
    // Only run demo if drawer is closed and no other demo is running
    if (isDrawerOpen || isDemoAnimating) return;

    isDemoAnimating = true; // Set flag
    const targetZ = drawerClosedZ + drawerDemoDistance;
    const startPos = { z: drawerGroup.position.z }; // Should be drawerClosedZ
    const targetPos = { z: targetZ };

    new TWEEN.Tween(startPos)
        .to(targetPos, demoAnimationDuration)
        .easing(TWEEN.Easing.Quadratic.InOut) // Use InOut for smoother back-and-forth
        .yoyo(true) // Automatically animate back to start
        .repeat(1) // Repeat once (out -> back)
        .onUpdate(() => { drawerGroup.position.z = startPos.z; })
        .onComplete(() => { isDemoAnimating = false; }) // Clear flag when done
        .start();
}

function demonstrateLeftDoorMotion() {
    // Only run demo if door is closed and no other demo is running
    if (isLeftDoorOpen || isDemoAnimating) return;

    isDemoAnimating = true; // Set flag
    const targetAngle = doorDemoAngle;
    const startRot = { y: leftDoorGroup.rotation.y }; // Should be 0
    const targetRot = { y: targetAngle };

    new TWEEN.Tween(startRot)
        .to(targetRot, demoAnimationDuration)
        .easing(TWEEN.Easing.Quadratic.InOut)
        .yoyo(true)
        .repeat(1)
        .onUpdate(() => { leftDoorGroup.rotation.y = startRot.y; })
        .onComplete(() => { isDemoAnimating = false; }) // Clear flag when done
        .start();
}

function demonstrateRightDoorMotion() {
     // Only run demo if door is closed and no other demo is running
    if (isRightDoorOpen || isDemoAnimating) return;

    isDemoAnimating = true; // Set flag
    const targetAngle = -doorDemoAngle; // Negative angle
    const startRot = { y: rightDoorGroup.rotation.y }; // Should be 0
    const targetRot = { y: targetAngle };

    new TWEEN.Tween(startRot)
        .to(targetRot, demoAnimationDuration)
        .easing(TWEEN.Easing.Quadratic.InOut)
        .yoyo(true)
        .repeat(1)
        .onUpdate(() => { rightDoorGroup.rotation.y = startRot.y; })
        .onComplete(() => { isDemoAnimating = false; }) // Clear flag when done
        .start();
}


function animate(time) {
    requestAnimationFrame(animate);

    if (time) {
       TWEEN.update(time);
    }

    controls.update();
    renderer.render(scene, camera);
}

# !!!! First example for exporting !!!!
import * as THREE from 'three';
import { OrbitControls } from 'three/addons/controls/OrbitControls.js';
import TWEEN from '@tweenjs/tween.js';

let scene, camera, renderer, controls;
let cabinetUnit, drawerGroup, leftDoorGroup, rightDoorGroup;
let raycaster, mouse;
const interactiveObjects = []; // Array to hold objects that can be clicked

// --- State Variables ---
let isDrawerOpen = false;
let isLeftDoorOpen = false;
let isRightDoorOpen = false;
let isDemoAnimating = false; // Prevent multiple demo animations at once

// --- Constants for Dimensions ---
const cabinetWidth = 1.2;
const cabinetHeight = 0.8;
const cabinetDepth = 0.5;
const woodThickness = 0.02; // Main cabinet frame thickness
const drawerHeight = 0.2;
const handleDepth = 0.03;
const handleWidth = 0.1;
const handleHeight = 0.02;
const doorGap = 0.005; // Small gap between doors/drawer front
const drawerPanelThickness = 0.015; // Thickness for drawer box panels (can be thinner)
const drawerInset = 0.01; // Small inset from the main cabinet depth

// --- Animation Parameters ---
const animationDuration = 500; // milliseconds for open/close
const demoAnimationDuration = 750; // Slightly longer for demonstration
// Calculate drawer closed position once
const drawerBoxDepth = cabinetDepth - woodThickness - drawerInset;
const drawerClosedZ = cabinetDepth / 2 - drawerBoxDepth / 2;
const drawerOpenDistance = drawerBoxDepth * 0.75; // How far it opens interactively
const drawerDemoDistance = drawerBoxDepth * 0.5; // How far it moves for demo
const doorOpenAngle = Math.PI * 0.45; // Angle for interactive open
const doorDemoAngle = Math.PI * 0.3; // Angle for demo swing

init();
animate();

function init() {
    // --- Scene ---
    scene = new THREE.Scene();
    scene.background = new THREE.Color(0xf0f0f0);

    // --- Camera ---
    camera = new THREE.PerspectiveCamera(60, window.innerWidth / window.innerHeight, 0.1, 100);
    camera.position.set(1, 1, 1.5);
    camera.lookAt(0, 0, 0);

    // --- Renderer ---
    renderer = new THREE.WebGLRenderer({ antialias: true });
    renderer.setSize(window.innerWidth, window.innerHeight);
    renderer.shadowMap.enabled = true;
    renderer.shadowMap.type = THREE.PCFSoftShadowMap;
    document.body.appendChild(renderer.domElement);

    // --- Controls ---
    controls = new OrbitControls(camera, renderer.domElement);
    controls.enableDamping = true;
    controls.dampingFactor = 0.05;
    controls.screenSpacePanning = false;
    controls.minDistance = 0.5;
    controls.maxDistance = 5;
    controls.target.set(0, cabinetHeight / 3, 0);
    controls.update();

    // --- Lighting ---
    const ambientLight = new THREE.AmbientLight(0xffffff, 0.6);
    scene.add(ambientLight);

    const directionalLight = new THREE.DirectionalLight(0xffffff, 1.5);
    directionalLight.position.set(3, 5, 4);
    directionalLight.castShadow = true;
    directionalLight.shadow.mapSize.width = 1024;
    directionalLight.shadow.mapSize.height = 1024;
    directionalLight.shadow.camera.near = 0.5;
    directionalLight.shadow.camera.far = 15;
    directionalLight.shadow.camera.left = -2;
    directionalLight.shadow.camera.right = 2;
    directionalLight.shadow.camera.top = 2;
    directionalLight.shadow.camera.bottom = -2;
    scene.add(directionalLight);

    // --- Ground Plane (Optional) ---
    const groundGeo = new THREE.PlaneGeometry(10, 10);
    const groundMat = new THREE.MeshStandardMaterial({ color: 0xcccccc, side: THREE.DoubleSide });
    const ground = new THREE.Mesh(groundGeo, groundMat);
    ground.rotation.x = -Math.PI / 2;
    ground.position.y = 0; // Align ground with cabinet base if unit is shifted
    ground.receiveShadow = true;
    scene.add(ground);


    // --- Create Cabinet ---
    cabinetUnit = createCabinet();
    // Shift the entire unit up so its base is at y=0
    cabinetUnit.position.y = cabinetHeight / 2;
    scene.add(cabinetUnit);

    // --- Interaction Setup ---
    raycaster = new THREE.Raycaster();
    mouse = new THREE.Vector2();
    // window.addEventListener('click', onClick);
    // window.addEventListener('keydown', onKeyDown); // <-- Add key listener

    // --- Resize Listener ---
    // window.addEventListener('resize', onWindowResize);
}

function createCabinet() {
    const unitGroup = new THREE.Group();
    // *** NOTE: unitGroup.position.y is set AFTER creation in init() ***

    // --- Materials ---
    const woodMaterial = new THREE.MeshStandardMaterial({ color: 0x8B4513, roughness: 0.8, metalness: 0.1 });
    const drawerWoodMaterial = new THREE.MeshStandardMaterial({ color: 0x965A21, roughness: 0.85, metalness: 0.1 });
    const handleMaterial = new THREE.MeshStandardMaterial({ color: 0xA9A9A9, roughness: 0.4, metalness: 0.8 });

    // Helper function for creating panels
    const createPanel = (w, h, d, mat) => {
        const geometry = new THREE.BoxGeometry(w, h, d);
        const mesh = new THREE.Mesh(geometry, mat);
        mesh.castShadow = true;
        mesh.receiveShadow = true;
        return mesh;
    };

    // --- Static Cabinet Body ---
    const bodyGroup = new THREE.Group();
    unitGroup.add(bodyGroup);

    // Top Panel
    const topPanel = createPanel(cabinetWidth, woodThickness, cabinetDepth, woodMaterial);
    topPanel.position.y = cabinetHeight / 2 - woodThickness / 2;
    bodyGroup.add(topPanel);

    // Bottom Panel
    const bottomPanel = createPanel(cabinetWidth, woodThickness, cabinetDepth, woodMaterial);
    bottomPanel.position.y = -cabinetHeight / 2 + woodThickness / 2;
    bodyGroup.add(bottomPanel);

    // Left Side Panel
    const leftSide = createPanel(woodThickness, cabinetHeight, cabinetDepth, woodMaterial);
    leftSide.position.x = -cabinetWidth / 2 + woodThickness / 2;
    bodyGroup.add(leftSide);

    // Right Side Panel
    const rightSide = createPanel(woodThickness, cabinetHeight, cabinetDepth, woodMaterial);
    rightSide.position.x = cabinetWidth / 2 - woodThickness / 2;
    bodyGroup.add(rightSide);

    // Back Panel (slightly inset)
    const backPanel = createPanel(cabinetWidth - 2 * woodThickness, cabinetHeight - 2 * woodThickness, woodThickness, woodMaterial);
    backPanel.position.z = -cabinetDepth / 2 + woodThickness / 2;
    backPanel.position.y = 0;
    bodyGroup.add(backPanel);

    // Internal Horizontal Shelf
    const shelfY = cabinetHeight / 2 - woodThickness - drawerHeight - woodThickness / 2;
    const horizontalShelf = createPanel(cabinetWidth - 2 * woodThickness, woodThickness, cabinetDepth - woodThickness, woodMaterial);
    horizontalShelf.position.y = shelfY;
    horizontalShelf.position.z = 0;
    bodyGroup.add(horizontalShelf);

    // Internal Vertical Divider
    const dividerHeight = cabinetHeight - drawerHeight - 3 * woodThickness;
    const dividerY = -cabinetHeight / 2 + woodThickness + dividerHeight / 2;
    const verticalDivider = createPanel(woodThickness, dividerHeight, cabinetDepth - woodThickness, woodMaterial);
    verticalDivider.position.y = dividerY;
    verticalDivider.position.z = 0;
    bodyGroup.add(verticalDivider);

    // --- Drawer Box ---
    drawerGroup = new THREE.Group();
    const drawerInnerWidth = cabinetWidth - 2 * woodThickness - 2 * doorGap;
    const drawerInnerHeight = drawerHeight - 2 * doorGap;

    // Front Panel
    const drawerFront = createPanel(drawerInnerWidth, drawerInnerHeight, drawerPanelThickness, woodMaterial);
    drawerFront.position.z = drawerBoxDepth / 2 - drawerPanelThickness / 2;
    drawerFront.name = 'drawerFrontMesh'; // Name for click detection
    drawerGroup.add(drawerFront);

    // Bottom Panel
    const drawerBottom = createPanel(drawerInnerWidth, drawerPanelThickness, drawerBoxDepth - drawerPanelThickness, drawerWoodMaterial);
    drawerBottom.position.y = -drawerInnerHeight / 2 + drawerPanelThickness / 2;
    drawerGroup.add(drawerBottom);

    // Back Panel
    const drawerBack = createPanel(drawerInnerWidth, drawerInnerHeight - drawerPanelThickness, drawerPanelThickness, drawerWoodMaterial);
    drawerBack.position.z = -drawerBoxDepth / 2 + drawerPanelThickness / 2;
    drawerBack.position.y = drawerPanelThickness / 2;
    drawerGroup.add(drawerBack);

    // Left Panel
    const drawerLeft = createPanel(drawerPanelThickness, drawerInnerHeight - drawerPanelThickness, drawerBoxDepth - drawerPanelThickness, drawerWoodMaterial);
    drawerLeft.position.x = -drawerInnerWidth / 2 + drawerPanelThickness / 2;
    drawerLeft.position.y = drawerPanelThickness / 2;
    drawerGroup.add(drawerLeft);

    // Right Panel
    const drawerRight = createPanel(drawerPanelThickness, drawerInnerHeight - drawerPanelThickness, drawerBoxDepth - drawerPanelThickness, drawerWoodMaterial);
    drawerRight.position.x = drawerInnerWidth / 2 - drawerPanelThickness / 2;
    drawerRight.position.y = drawerPanelThickness / 2;
    drawerGroup.add(drawerRight);

    // Drawer Handle
    const drawerHandle = createPanel(handleWidth, handleHeight, handleDepth, handleMaterial);
    drawerHandle.position.z = drawerFront.position.z + drawerPanelThickness / 2 + handleDepth / 2;
    drawerHandle.position.y = 0;
    drawerGroup.add(drawerHandle);

    // Position the entire drawer group
    drawerGroup.position.y = cabinetHeight / 2 - woodThickness - drawerHeight / 2;
    drawerGroup.position.z = drawerClosedZ; // Use calculated closed Z
    unitGroup.add(drawerGroup);
    interactiveObjects.push(drawerFront); // Add only front for clicking

    // --- Cabinet Doors ---
    const doorHeight = cabinetHeight - drawerHeight - 3 * woodThickness - 2 * doorGap;
    const doorWidth = (cabinetWidth - 3 * woodThickness) / 2 - doorGap;

    // Left Door
    leftDoorGroup = new THREE.Group();
    const leftDoor = createPanel(doorWidth, doorHeight, woodThickness, woodMaterial);
    leftDoor.position.x = doorWidth / 2;
    leftDoor.position.z = woodThickness / 2;
    leftDoor.name = 'leftDoorMesh'; // Name for click detection
    const leftHandle = createPanel(handleHeight, handleWidth, handleDepth, handleMaterial);
    leftHandle.position.x = doorWidth - handleHeight / 2 - doorGap;
    leftHandle.position.z = woodThickness + handleDepth / 2;
    leftDoorGroup.add(leftDoor);
    leftDoorGroup.add(leftHandle);
    leftDoorGroup.position.x = -cabinetWidth / 2 + woodThickness;
    leftDoorGroup.position.y = dividerY;
    leftDoorGroup.position.z = cabinetDepth / 2 - woodThickness / 2;
    unitGroup.add(leftDoorGroup);
    interactiveObjects.push(leftDoor); // Add only door panel for clicking

    // Right Door
    rightDoorGroup = new THREE.Group();
    const rightDoor = createPanel(doorWidth, doorHeight, woodThickness, woodMaterial);
    rightDoor.position.x = -doorWidth / 2;
    rightDoor.position.z = woodThickness / 2;
    rightDoor.name = 'rightDoorMesh'; // Name for click detection
    const rightHandle = createPanel(handleHeight, handleWidth, handleDepth, handleMaterial);
    rightHandle.position.x = -doorWidth + handleHeight / 2 + doorGap;
    rightHandle.position.z = woodThickness + handleDepth / 2;
    rightDoorGroup.add(rightDoor);
    rightDoorGroup.add(rightHandle);
    rightDoorGroup.position.x = cabinetWidth / 2 - woodThickness;
    rightDoorGroup.position.y = dividerY;
    rightDoorGroup.position.z = cabinetDepth / 2 - woodThickness / 2;
    unitGroup.add(rightDoorGroup);
    interactiveObjects.push(rightDoor); // Add only door panel for clicking

    return unitGroup;
}

function onWindowResize() {
    camera.aspect = window.innerWidth / window.innerHeight;
    camera.updateProjectionMatrix();
    renderer.setSize(window.innerWidth, window.innerHeight);
}

function onClick(event) {
    // Don't allow clicks if a demo is running
    if (isDemoAnimating) return;

    mouse.x = (event.clientX / window.innerWidth) * 2 - 1;
    mouse.y = -(event.clientY / window.innerHeight) * 2 + 1;
    raycaster.setFromCamera(mouse, camera);
    const intersects = raycaster.intersectObjects(interactiveObjects, false);

    if (intersects.length > 0) {
        const clickedObjectName = intersects[0].object.name;

        if (clickedObjectName === 'drawerFrontMesh') {
             toggleDrawer();
        } else if (clickedObjectName === 'leftDoorMesh') {
             toggleLeftDoor();
        } else if (clickedObjectName === 'rightDoorMesh') {
             toggleRightDoor();
        }
    }
}

// --- NEW KeyDown Listener ---
function onKeyDown(event) {
    // Don't allow new demos if one is already running
    if (isDemoAnimating) return;

    switch (event.key.toLowerCase()) {
        case 'd':
            demonstrateDrawerMotion();
            break;
        case 'l':
            demonstrateLeftDoorMotion();
            break;
        case 'r':
            demonstrateRightDoorMotion();
            break;
    }
}


// --- Interaction Animation Functions ---

function toggleDrawer() {
    const targetZ = isDrawerOpen ? drawerClosedZ : drawerClosedZ + drawerOpenDistance;
    const startPos = { z: drawerGroup.position.z };
    const targetPos = { z: targetZ };

    new TWEEN.Tween(startPos)
        .to(targetPos, animationDuration)
        .easing(TWEEN.Easing.Quadratic.Out)
        .onUpdate(() => { drawerGroup.position.z = startPos.z; })
        .onComplete(() => { isDrawerOpen = !isDrawerOpen; }) // Toggle state on completion
        .start();
    // Note: State is toggled in onComplete now to reflect final state
}

function toggleLeftDoor() {
    const targetAngle = isLeftDoorOpen ? 0 : doorOpenAngle;
    const startRot = { y: leftDoorGroup.rotation.y };
    const targetRot = { y: targetAngle };

    new TWEEN.Tween(startRot)
        .to(targetRot, animationDuration)
        .easing(TWEEN.Easing.Quadratic.Out)
        .onUpdate(() => { leftDoorGroup.rotation.y = startRot.y; })
        .onComplete(() => { isLeftDoorOpen = !isLeftDoorOpen; }) // Toggle state on completion
        .start();
}

function toggleRightDoor() {
    const targetAngle = isRightDoorOpen ? 0 : -doorOpenAngle; // Negative angle
    const startRot = { y: rightDoorGroup.rotation.y };
    const targetRot = { y: targetAngle };

    new TWEEN.Tween(startRot)
        .to(targetRot, animationDuration)
        .easing(TWEEN.Easing.Quadratic.Out)
        .onUpdate(() => { rightDoorGroup.rotation.y = startRot.y; })
         .onComplete(() => { isRightDoorOpen = !isRightDoorOpen; }) // Toggle state on completion
        .start();
}

// --- NEW Demonstration Animation Functions ---

function demonstrateDrawerMotion() {
    // Only run demo if drawer is closed and no other demo is running
    if (isDrawerOpen || isDemoAnimating) return;

    isDemoAnimating = true; // Set flag
    const targetZ = drawerClosedZ + drawerDemoDistance;
    const startPos = { z: drawerGroup.position.z }; // Should be drawerClosedZ
    const targetPos = { z: targetZ };

    new TWEEN.Tween(startPos)
        .to(targetPos, demoAnimationDuration)
        .easing(TWEEN.Easing.Quadratic.InOut) // Use InOut for smoother back-and-forth
        .yoyo(true) // Automatically animate back to start
        .repeat(1) // Repeat once (out -> back)
        .onUpdate(() => { drawerGroup.position.z = startPos.z; })
        .onComplete(() => { isDemoAnimating = false; }) // Clear flag when done
        .start();
}

function demonstrateLeftDoorMotion() {
    // Only run demo if door is closed and no other demo is running
    if (isLeftDoorOpen || isDemoAnimating) return;

    isDemoAnimating = true; // Set flag
    const targetAngle = doorDemoAngle;
    const startRot = { y: leftDoorGroup.rotation.y }; // Should be 0
    const targetRot = { y: targetAngle };

    new TWEEN.Tween(startRot)
        .to(targetRot, demoAnimationDuration)
        .easing(TWEEN.Easing.Quadratic.InOut)
        .yoyo(true)
        .repeat(1)
        .onUpdate(() => { leftDoorGroup.rotation.y = startRot.y; })
        .onComplete(() => { isDemoAnimating = false; }) // Clear flag when done
        .start();
}

function demonstrateRightDoorMotion() {
     // Only run demo if door is closed and no other demo is running
    if (isRightDoorOpen || isDemoAnimating) return;

    isDemoAnimating = true; // Set flag
    const targetAngle = -doorDemoAngle; // Negative angle
    const startRot = { y: rightDoorGroup.rotation.y }; // Should be 0
    const targetRot = { y: targetAngle };

    new TWEEN.Tween(startRot)
        .to(targetRot, demoAnimationDuration)
        .easing(TWEEN.Easing.Quadratic.InOut)
        .yoyo(true)
        .repeat(1)
        .onUpdate(() => { rightDoorGroup.rotation.y = startRot.y; })
        .onComplete(() => { isDemoAnimating = false; }) // Clear flag when done
        .start();
}


function animate(time) {
    requestAnimationFrame(animate);

    if (time) {
       TWEEN.update(time);
    }

    controls.update();
    renderer.render(scene, camera);
}

## Code to Read
Please make sure the output format can be read and output mesh by the following export.js script.

// export.js - General Three.js Mesh Exporter
import * as THREE from 'three';
import { OBJExporter } from 'three/examples/jsm/exporters/OBJExporter.js';
import fs from 'fs';
import path from 'path';
import { pathToFileURL } from 'url'; // Needed for reliable dynamic import

// --- Configuration ---
const EXPECTED_EXPORT_FUNCTION = 'createScene'; // Name of the function to call in the user module
const EXPECTED_EXPORT_VARIABLE = 'sceneObject'; // Name of the variable to use in the user module

// --- Main Async Function ---
async function runExport() {
    // --- Command Line Argument Handling ---
    const args = process.argv.slice(2);
    if (args.length !== 2) {
        console.error("Usage: node export.js <path_to_scene_module.js> <output_directory>");
        console.error("\nThe scene module must export a function 'createScene' or a variable 'sceneObject'.");
        process.exit(1);
    }

    const sceneModulePath = path.resolve(args[0]);
    const outputDirectory = path.resolve(args[1]);

    // Check if scene module file exists
    if (!fs.existsSync(sceneModulePath)) {
        console.error(`Error: Scene module file not found at '${sceneModulePath}'`);
        process.exit(1);
    }

    // Ensure output directory exists
    try {
        fs.mkdirSync(outputDirectory, { recursive: true });
        console.log(`Output directory: ${outputDirectory}`);
    } catch (err) {
        console.error(`Error creating output directory '${outputDirectory}':`, err);
        process.exit(1);
    }

    // --- Dynamically Import User's Scene Module ---
    let sceneModule;
    try {
        // Convert file path to file URL for reliable dynamic import
        const sceneModuleURL = pathToFileURL(sceneModulePath).href;
        console.log(`Importing scene module from: ${sceneModuleURL}`);
        sceneModule = await import(sceneModuleURL);
    } catch (error) {
        console.error(`Error importing scene module from '${sceneModulePath}':`);
        console.error(error);
        process.exit(1);
    }

    // --- Get Scene Object from Module ---
    let rootObject = null;
    if (typeof sceneModule[EXPECTED_EXPORT_FUNCTION] === 'function') {
        console.log(`Calling exported function '${EXPECTED_EXPORT_FUNCTION}'...`);
        try {
            rootObject = sceneModule[EXPECTED_EXPORT_FUNCTION]();
        } catch (error) {
             console.error(`Error executing exported function '${EXPECTED_EXPORT_FUNCTION}':`);
             console.error(error);
             process.exit(1);
        }
    } else if (sceneModule[EXPECTED_EXPORT_VARIABLE]) {
        console.log(`Using exported variable '${EXPECTED_EXPORT_VARIABLE}'...`);
        rootObject = sceneModule[EXPECTED_EXPORT_VARIABLE];
    } else {
        console.error(`Error: Scene module '${sceneModulePath}' must export either a function named '${EXPECTED_EXPORT_FUNCTION}' or a variable named '${EXPECTED_EXPORT_VARIABLE}'.`);
        process.exit(1);
    }

    // Validate the obtained object
    if (!rootObject || !(rootObject instanceof THREE.Object3D)) {
         console.error(`Error: The exported value from '${sceneModulePath}' is not a valid THREE.Object3D (Scene, Group, Mesh, etc.).`);
         process.exit(1);
    }
    console.log(`Successfully obtained root object: ${rootObject.name || '[unnamed]'}`);


    // --- Prepare for Export ---
    // Ensure world matrices are calculated for correct positioning
    rootObject.updateMatrixWorld(true);

    const exporter = new OBJExporter();
    let meshIndex = 0; // Counter for unnamed meshes
    let exportCount = 0;

    console.log('\nStarting mesh export...');

    // --- Traverse and Export Meshes ---
    rootObject.traverse((object) => {
        if (object instanceof THREE.Mesh) {
            exportCount++;
            // Generate a filename
            let filename = object.name || `mesh_${meshIndex++}`;
            // Sanitize filename
            filename = filename.replace(/[^a-z0-9_.-]/gi, '_') + '.obj';
            const filePath = path.join(outputDirectory, filename);

            try {
                // Exporter uses the object's world matrix automatically
                const objData = exporter.parse(object);
                fs.writeFileSync(filePath, objData);
                console.log(`  [${exportCount}] Exported: ${filename} (Name: ${object.name || 'N/A'})`);
            } catch (error) {
                console.error(`  [${exportCount}] Error exporting ${filename}:`, error);
            }
        }
    });

    console.log(`\nExport process finished. Found and attempted to export ${exportCount} mesh(es).`);
}

// --- Run the export process ---
runExport().catch(err => {
    console.error("\nAn unexpected error occurred during the export process:");
    console.error(err);
    process.exit(1);
});
"""