"""
Base provider class for all LLM API providers
"""

from abc import ABC, abstractmethod
from typing import Dict, Any, Tuple
import logging


class BaseProvider(ABC):
    """Abstract base class for all LLM providers"""
    
    def __init__(self, config: Dict[str, Any]):
        """
        Initialize provider with configuration
        
        Args:
            config: Configuration dictionary containing:
                - api_key: API key for the provider
                - model_name: Name of the model
                - temperature: Temperature for generation
                - max_tokens: Maximum tokens for generation
        """
        self.config = config
        self.api_key = config.get('api_key')
        self.model_name = config.get('model_name')
        self.temperature = config.get('temperature', 0.7)
        self.max_tokens = config.get('max_tokens', 4000)
        self.logger = logging.getLogger(self.__class__.__name__)
        
    @abstractmethod
    def invoke(self, prompt: str, system_prompt: str = None, **kwargs) -> Tuple[str, Dict]:
        """
        Invoke the LLM with given prompt
        
        Returns:
            Tuple of (response_text, metadata_dict)
        """
        pass
    
    @abstractmethod
    def count_tokens(self, text: str) -> int:
        """Count tokens in the given text"""
        pass