"""
Smart model detector for automatic provider and endpoint identification
"""

from typing import Dict


class ModelDetector:
    """Automatically detect provider and configuration based on model name"""
    
    # Provider detection patterns (in order of priority)
    PROVIDER_PATTERNS = {
        'openai': [
            'gpt-',      # gpt-4o, gpt-5, etc.
            'o1-',       # o1-mini, o1-pro, etc.
            'o3',        # o3, o3-pro, etc.
            'o4',        # o4-mini, etc.
        ],
        'google': [
            'gemini',    # gemini series
        ],
        'anthropic': [
            'claude',    # Reserved for future Claude support
        ]
    }
    
    # Special endpoint mappings (models that don't use standard endpoints)
    SPECIAL_ENDPOINTS = {
        # OpenAI special endpoints
        'o3-pro': 'responses',
        'o3': 'responses',
        'o1-pro': 'responses',
        'o1-pro-2025': 'responses',  # Matches any o1-pro-2025-XX-XX
        'o1-mini': 'responses',
        # GPT-5 models use responses endpoint (recommended by OpenAI)
        'gpt-5': 'responses',
        'gpt-5-mini': 'responses',
        'gpt-5-nano': 'responses',
    }
    
    # Default endpoints for each provider
    DEFAULT_ENDPOINTS = {
        'openai': 'chat/completions',
        'google': 'generateContent',
        'anthropic': 'messages',
    }
    
    # Model-specific configurations (overrides) - temperature is unified to 1.0 for all models
    MODEL_CONFIGS = {
        # GPT-5 models use reasoning_effort and verbosity parameters
        'gpt-5': {'reasoning_effort': 'medium', 'verbosity': 'medium'},
        'gpt-5-mini': {'reasoning_effort': 'medium', 'verbosity': 'medium'},
        'gpt-5-nano': {'reasoning_effort': 'low', 'verbosity': 'low'},
        # Other models use default temperature=1.0 (no need to specify)
    }
    
    @classmethod
    def detect_provider(cls, model_name: str) -> str:
        """
        Detect provider based on model name
        
        Args:
            model_name: Name of the model
            
        Returns:
            Provider name ('openai', 'google', 'anthropic')
            
        Raises:
            ValueError: If provider cannot be detected
        """
        model_lower = model_name.lower()
        
        # Check each provider's patterns
        for provider, patterns in cls.PROVIDER_PATTERNS.items():
            for pattern in patterns:
                if model_lower.startswith(pattern):
                    return provider
        
        # If no pattern matches, raise error
        raise ValueError(
            f"Cannot detect provider for model '{model_name}'. "
            f"Model name should start with one of the known patterns: "
            f"{', '.join([p for patterns in cls.PROVIDER_PATTERNS.values() for p in patterns])}"
        )
    
    @classmethod
    def detect_endpoint(cls, model_name: str, provider: str = None) -> str:
        """
        Detect API endpoint based on model name
        
        Args:
            model_name: Name of the model
            provider: Optional provider name (if already known)
            
        Returns:
            API endpoint string
        """
        model_lower = model_name.lower()
        
        # Check for special endpoints
        for model_prefix, endpoint in cls.SPECIAL_ENDPOINTS.items():
            if model_lower.startswith(model_prefix):
                return endpoint
        
        # Get provider if not provided
        if provider is None:
            provider = cls.detect_provider(model_name)
        
        # Return default endpoint for provider
        return cls.DEFAULT_ENDPOINTS.get(provider, 'default')
    
    @classmethod
    def get_model_config(cls, model_name: str) -> Dict:
        """
        Get model-specific configuration overrides
        
        Args:
            model_name: Name of the model
            
        Returns:
            Dictionary of configuration overrides
        """
        model_lower = model_name.lower()
        
        # Check for exact match first
        if model_lower in cls.MODEL_CONFIGS:
            return cls.MODEL_CONFIGS[model_lower].copy()
        
        # Check for prefix match
        for model_prefix, config in cls.MODEL_CONFIGS.items():
            if model_lower.startswith(model_prefix):
                return config.copy()
        
        # Return empty dict if no specific config
        return {}
    
    @classmethod
    def get_handler_class(cls, model_name: str, provider: str = None) -> str:
        """
        Get the handler class path for a model
        
        Args:
            model_name: Name of the model
            provider: Optional provider name (if already known)
            
        Returns:
            Handler class path string
        """
        # Get provider if not provided
        if provider is None:
            provider = cls.detect_provider(model_name)
        
        # Handler mapping based on provider
        handler_map = {
            'openai': 'providers.openai_handler.OpenAIHandler',
            'google': 'providers.google_handler.GoogleHandler',
            'anthropic': 'providers.anthropic_handler.AnthropicHandler',
        }
        
        return handler_map.get(provider, 'providers.base_provider.BaseProvider')
    
    @classmethod
    def analyze_model(cls, model_name: str) -> Dict:
        """
        Complete analysis of a model name
        
        Args:
            model_name: Name of the model
            
        Returns:
            Dictionary containing:
                - provider: Detected provider
                - endpoint: API endpoint to use
                - handler_class: Handler class path
                - config: Model-specific configurations
        """
        provider = cls.detect_provider(model_name)
        endpoint = cls.detect_endpoint(model_name, provider)
        handler_class = cls.get_handler_class(model_name, provider)
        config = cls.get_model_config(model_name)
        
        return {
            'provider': provider,
            'endpoint': endpoint,
            'handler_class': handler_class,
            'model_config': config,
            'model_name': model_name
        }