"""
Smart factory for automatic LLM provider creation and configuration.

This factory automatically detects the appropriate provider (OpenAI, Google) based on model names
and creates properly configured provider instances. It eliminates the need for manual provider 
selection and configuration, making the system more maintainable and user-friendly.

Key Features:
- Automatic provider detection from model names (e.g., "gpt-4o" → OpenAI, "gemini-2.5-pro" → Google)
- Intelligent configuration merging (defaults + model-specific + user overrides)
- Dynamic provider class loading with caching for performance
- Comprehensive error handling and validation
- Support for new models without code changes (via ModelDetector patterns)

Architecture Flow:
    1. Model name input (e.g., "gpt-4o", "gemini-2.5-pro")
    2. ModelDetector analyzes and determines provider
    3. Configuration builder merges all config sources
    4. Provider class is loaded/cached and instantiated
    5. Fully configured provider returned

Example Usage:
    # Simple usage
    provider = SmartProviderFactory.create_provider(
        "gpt-4o", 
        {"temperature": 0.7},
        {"openai": "sk-xxx", "google": "AIza-xxx"}
    )
    
    # Provider automatically handles OpenAI API calls, token counting, etc.
    response, metadata = provider.invoke("Hello world")

Supported Models:
- OpenAI: gpt-4o, gpt-4o-mini, o3, o1-pro, etc.
- Google: gemini-2.5-pro, gemini-2.5-flash, gemini-1.5-pro, etc.
"""

import importlib
import logging
from typing import Dict, Any, Optional
from .model_detector import ModelDetector
from .base_provider import BaseProvider


class SmartProviderFactory:
    """Factory class that automatically creates the right provider for any model"""
    
    # Cache for loaded provider classes
    _provider_cache = {}
    
    @classmethod
    def create_provider(
        cls, 
        model_name: str, 
        config: Dict[str, Any],
        api_keys: Dict[str, str] = None
    ) -> BaseProvider:
        """
        Create a provider instance based on model name
        
        Args:
            model_name: Name of the model (e.g., 'gpt-4o', 'o3', 'gemini-2.5-pro')
            config: Configuration dictionary (can contain overrides)
            api_keys: Dictionary of API keys by provider name
            
        Returns:
            Configured provider instance
            
        Raises:
            ValueError: If model or provider is not supported
        """
        # Analyze the model to get all necessary information
        model_info = ModelDetector.analyze_model(model_name)
        
        # Build complete configuration
        provider_config = cls._build_config(model_info, config, api_keys)
        
        # Get or load the provider class
        provider_class = cls._get_provider_class(model_info['handler_class'])
        
        # Create and return provider instance
        return provider_class(provider_config)
    
    @classmethod
    def _build_config(
        cls,
        model_info: Dict[str, Any],
        user_config: Dict[str, Any],
        api_keys: Dict[str, str] = None
    ) -> Dict[str, Any]:
        """
        Build complete configuration by merging model info, defaults, and user config
        
        Args:
            model_info: Information from ModelDetector
            user_config: User-provided configuration
            api_keys: API keys dictionary
            
        Returns:
            Complete configuration dictionary
        """
        # Start with model information
        config = {
            'model_name': model_info['model_name'],
            'provider': model_info['provider'],
            'endpoint': model_info['endpoint'],
        }
        
        # Add API key
        if api_keys and model_info['provider'] in api_keys:
            config['api_key'] = api_keys[model_info['provider']]
        elif 'api_key' in user_config:
            config['api_key'] = user_config['api_key']
        else:
            # Try to get from user_config with provider-specific key
            provider_key = f"{model_info['provider']}_api_key"
            if provider_key in user_config:
                config['api_key'] = user_config[provider_key]
        
        # Apply default configurations
        defaults = {
            'temperature': 0.7,
            'max_tokens': 4000,
            'top_p': 1.0,
            'frequency_penalty': 0,
            'presence_penalty': 0,
        }
        config.update(defaults)
        
        # Apply model-specific configurations
        if model_info['model_config']:
            config.update(model_info['model_config'])
        
        # Apply user overrides (highest priority)
        for key in ['temperature', 'max_tokens', 'top_p', 'frequency_penalty', 
                    'presence_penalty', 'timeout', 'max_retries']:
            if key in user_config:
                config[key] = user_config[key]
        
        # Add any additional user config that's not already set
        for key, value in user_config.items():
            if key not in config:
                config[key] = value
        
        return config
    
    @classmethod
    def _get_provider_class(cls, handler_class_path: str):
        """
        Get or load a provider class
        
        Args:
            handler_class_path: Full path to the handler class
            
        Returns:
            Provider class
        """
        # Check cache first
        if handler_class_path in cls._provider_cache:
            return cls._provider_cache[handler_class_path]
        
        # Parse module and class name
        module_path, class_name = handler_class_path.rsplit('.', 1)
        
        try:
            # Import the module
            module = importlib.import_module(module_path)
            
            # Get the class
            provider_class = getattr(module, class_name)
            
            # Cache it
            cls._provider_cache[handler_class_path] = provider_class
            
            return provider_class
            
        except (ImportError, AttributeError) as e:
            logging.error(f"Failed to load provider class {handler_class_path}: {e}")
            raise ValueError(f"Cannot load provider class: {handler_class_path}")
    
    @classmethod
    def get_supported_models(cls) -> Dict[str, list]:
        """
        Get a list of all supported model patterns
        
        Returns:
            Dictionary of provider -> list of model patterns
        """
        return ModelDetector.PROVIDER_PATTERNS.copy()
    
    @classmethod
    def is_model_supported(cls, model_name: str) -> bool:
        """
        Check if a model is supported
        
        Args:
            model_name: Name of the model
            
        Returns:
            True if supported, False otherwise
        """
        try:
            ModelDetector.detect_provider(model_name)
            return True
        except ValueError:
            return False