#!/usr/bin/env python3
"""
Modern pipeline using the unified BaseAgent system with organized output structure.

This pipeline orchestrates the complete articulated object generation process:
1. LinkerGenerator: Converts descriptions to detailed link specifications
2. ShapeGenerator: Creates Three.js code from descriptions and links
3. ArticulationGenerator: Generates joint specifications from workflow
4. Export: Converts to mesh files and URDF format

All agents use the new BaseAgent architecture with automatic provider detection.
Output is organized into a clean folder structure for better maintainability.
"""

import os
import sys
import re
import argparse
import logging
import subprocess
import time
from typing import Optional, List, Tuple, Dict, Any
import concurrent.futures
import yaml
import shutil
import json

from agents.linker_generator_agent import LinkerGeneratorAgent
from utils.combine_meshes import combine_meshes
from agents.shape_generator_agent import ShapeGeneratorAgent
from utils.config_manager import ConfigManager
from agents.articulation_generator_agent import ArticulationGeneratorAgent
from agents.vlm_critic import VLMCriticAgent
from agents.shape_fixer import ShapeFixerAgent
from agents.articulation_vlm_critic import ArticulationVLMCritic
from agents.articulation_fixer import ArticulationFixer
from utils.articulation_renderer import ArticulationRenderer
from agents.feedback_fusion import FeedbackFusion, create_feedback_fusion
from utils.mesh_renderer import MeshRenderer
from utils.generate_urdf import generate_urdf
from utils.common import get_unique_output_folder, extract_and_merge_threejs_info, setup_object_logger, remove_object_logger, setup_global_logging
from utils.mesh_exporter import MeshExporter
from utils.simple_colors import success, progress, error, warning, stats
from utils.pointcloud_renderer import render_pointcloud_for_iteration


class Pipeline:
    """
    Modern pipeline using unified BaseAgent system with organized output.
    
    Features:
    - Automatic model detection and provider selection
    - Clean BaseAgent API calls
    - Comprehensive error handling and statistics
    - Parallel processing support
    - Organized output folder structure
    """
    
    def __init__(self, args: argparse.Namespace):
        self.args = args
        self.use_linker = getattr(args, 'use_linker', True)
        
        # Initialize configuration with new structure
        self.config = ConfigManager("config.yaml")
        self._setup_config()
        
        # Initialize agents with BaseAgent system
        self._initialize_agents()
        self._initialize_pointllm_components()

        # Initialize statistics
        self.stats = {
            'total_attempts': 0,
            'successful_generations': 0,
            'successful_exports': 0,
            'failed_generations': 0,
            'failed_exports': 0,
            'successful_articulations': 0,
            'failed_articulations': 0,
            'error_types': {},
            'export_errors': {}
        }

    def _setup_config(self):
        """Set up configuration using new unified structure"""
        # Update output directory
        if self.args.output_dir:
            self.config.output_dir = self.args.output_dir

        # Update agent models using new config structure
        if hasattr(self.args, 'linker_model') and self.args.linker_model:
            self.config.config['api']['agents']['linker_generator'] = self.args.linker_model

        if hasattr(self.args, 'shape_model') and self.args.shape_model:
            self.config.config['api']['agents']['shape_generator'] = self.args.shape_model

        if hasattr(self.args, 'articulation_model') and self.args.articulation_model:
            self.config.config['api']['agents']['articulation_generator'] = self.args.articulation_model

        # Update VLM critic configuration
        if hasattr(self.args, 'no_vlm_critic') and self.args.no_vlm_critic:
            if 'vlm_critic' not in self.config.config:
                self.config.config['vlm_critic'] = {}
            self.config.config['vlm_critic']['enabled'] = False

        # Update retry configuration
        if hasattr(self.args, 'max_retries') and self.args.max_retries:
            self.config.config['retry']['max_retries'] = self.args.max_retries

        # Update temperature if specified globally
        if hasattr(self.args, 'temperature') and self.args.temperature:
            self.config.config['api']['defaults']['temperature'] = self.args.temperature

    def _initialize_agents(self):
        """Initialize all agents using BaseAgent system"""
        try:
            # Initialize agents with new BaseAgent system
            if self.use_linker:
                self.linker_agent = LinkerGeneratorAgent(self.config)
                logging.info(f"LinkerGeneratorAgent initialized with model: {self.config.get_model_for_agent('linker_generator')}")
            else:
                self.linker_agent = None
                logging.info("LinkerGenerator disabled - using description directly")
                
            # Shape and articulation agents are created on-demand in process_single
            logging.info(f"Shape model configured: {self.config.get_model_for_agent('shape_generator')}")
            logging.info(f"Articulation model configured: {self.config.get_model_for_agent('articulation_generator')}")
            
        except Exception as e:
            logging.error(f"Failed to initialize agents: {e}")
            raise

    def _initialize_pointllm_components(self):
        """Preload PointLLM critic and feedback fusion if enabled."""
        self.pointllm_critic = None
        self.feedback_fusion = None

        pointllm_config = self.config.config.get('pointllm_critic', {})
        if not pointllm_config.get('enabled', False):
            logging.info("PointLLM critic disabled in configuration")
            return

        try:
            # Import PointLLMCriticAgent only when needed
            from agents.pointllm_critic import PointLLMCriticAgent

            self.pointllm_critic = PointLLMCriticAgent(self.config)
            logging.info("PointLLM 3D critic preloaded for pipeline run")

            self.feedback_fusion = create_feedback_fusion(self.config)
            if self.feedback_fusion:
                logging.info("Feedback fusion enabled - will merge 2D and 3D analysis")
        except ImportError as e:
            logging.warning(f"PointLLM dependencies not available: {e}")
            logging.info("Continuing without PointLLM 3D analysis")
            self.pointllm_critic = None
            self.feedback_fusion = None
        except Exception as e:
            logging.warning(f"Failed to preload PointLLM components: {e}")
            self.pointllm_critic = None
            self.feedback_fusion = None

    def _log_stats(self):
        """Log current statistics"""
        total = self.stats['total_attempts']
        if total == 0:
            return

        success_rate = (self.stats['successful_generations'] / total) * 100
        export_rate = (self.stats['successful_exports'] / total) * 100
        articulation_rate = (self.stats['successful_articulations'] / total) * 100

        logging.info(stats(f"Statistics: {self.stats['successful_generations']}/{total} successful ({success_rate:.1f}%)"))
        logging.info(f"   Exports: {self.stats['successful_exports']}/{total} ({export_rate:.1f}%)")
        logging.info(f"   Articulations: {self.stats['successful_articulations']}/{total} ({articulation_rate:.1f}%)")

        if 'vlm_improvements' in self.stats:
            vlm_rate = (self.stats['vlm_improvements'] / total) * 100
            logging.info(f"   VLM Improvements: {self.stats['vlm_improvements']}/{total} ({vlm_rate:.1f}%)")

        if self.stats['error_types']:
            logging.info(f"   Common errors: {dict(list(self.stats['error_types'].items())[:3])}")

    def _run_vlm_feedback_loop(self, initial_code: str, description: str,
                              output_folder: str, pipeline_log_path: str,
                              links_json: Optional[Dict] = None) -> Tuple[str, bool, Dict]:
        """
        Run VLM feedback loop to improve generated code.

        Args:
            initial_code: Initial Three.js code from shape generator
            description: Original object description
            output_folder: Main output folder
            pipeline_log_path: Path to pipeline log file
            links_json: Optional links JSON for detailed specification

        Returns:
            Tuple of (final_code, improved, metrics)
        """
        # Check if VLM critic is enabled
        vlm_config = self.config.config.get('vlm_critic', {})
        if not vlm_config.get('enabled', False):
            logging.info("VLM critic disabled, skipping feedback loop")
            return initial_code, False, {}

        # Initialize VLM components
        try:
            vlm_critic = VLMCriticAgent(self.config)
            shape_fixer = ShapeFixerAgent(self.config)
            # Create optimized renderer
            renderer = MeshRenderer(
                image_size=tuple(vlm_config.get('image_size', [384, 384])),
                background_color=vlm_config.get('background_color', (0.95, 0.95, 0.95, 1.0))
            )
            logging.info("Using optimized renderer with enhanced lighting for VLM feedback")

            pointllm_critic = self.pointllm_critic
            feedback_fusion = self.feedback_fusion

            if pointllm_critic is not None:
                logging.info("PointLLM 3D critic ready for VLM feedback loop")
            elif self.config.config.get('pointllm_critic', {}).get('enabled', False):
                logging.warning("PointLLM critic was enabled but is unavailable; proceeding without 3D analysis")

        except Exception as e:
            logging.error(f"Failed to initialize VLM components: {e}")
            return initial_code, False, {}

        # Create stage 2.2 folder
        stage_folder = os.path.join(output_folder, 'pipeline_logs', 'stage_2_2_VLM_feedback')
        os.makedirs(stage_folder, exist_ok=True)

        # Log to pipeline
        with open(pipeline_log_path, 'a', encoding='utf-8') as f:
            f.write(f"\n[{time.strftime('%H:%M:%S')}] STAGE 2.2: VLM Feedback Loop Started\n")

        current_code = initial_code
        max_iterations = vlm_config.get('max_iterations', 3)
        improvement_threshold = vlm_config.get('improvement_threshold', 0.8)
        improved = False

        for iteration in range(1, max_iterations + 1):
            logging.info(progress(f"VLM feedback iteration {iteration}/{max_iterations}"))

            try:
                # Step 1: Export current code to mesh then render from multiple angles
                with open(pipeline_log_path, 'a', encoding='utf-8') as f:
                    f.write(f"  Iteration {iteration}: Exporting and rendering object...\n")

                # Create temporary folder for this iteration
                iter_folder = os.path.join(stage_folder, f'iteration_{iteration}')
                os.makedirs(iter_folder, exist_ok=True)

                # Export current code to mesh
                temp_export_temp = os.path.join(iter_folder, '_export_temp')
                os.makedirs(temp_export_temp, exist_ok=True)

                current_code_path = os.path.join(temp_export_temp, 'export.js')
                with open(current_code_path, 'w', encoding='utf-8') as f:
                    f.write(current_code)

                # Create package.json for ES modules
                package_json_path = os.path.join(iter_folder, 'package.json')
                with open(package_json_path, 'w') as f:
                    json.dump({"type": "module"}, f)

                # Export to mesh using MeshExporter
                temp_exporter = MeshExporter(iter_folder)
                mesh_result = temp_exporter.test_export(current_code)

                if not mesh_result.get('success'):
                    logging.warning(f"Failed to export mesh for iteration {iteration}: {mesh_result.get('error', 'Unknown error')}")
                    break

                # Find the generated mesh file
                obj_file = mesh_result.get('mesh_path')
                if not obj_file or not os.path.exists(obj_file):
                    logging.warning(f"Mesh file not found for iteration {iteration}")
                    break

                # Render combined mesh from multiple angles using component-based coloring
                logging.info(f"Rendering mesh from 4 angles for iteration {iteration}...")
                render_start = time.time()

                try:
                    image_paths = renderer.render_mesh_from_file(obj_file, iter_folder)
                except Exception as e:
                    logging.warning(f"Rendering failed for iteration {iteration}: {e}")
                    break

                render_time = time.time() - render_start

                if not image_paths:
                    logging.warning(f"Failed to render images for iteration {iteration}")
                    break

                logging.info(f"Rendered {len(image_paths)} images in {render_time:.2f}s")

                # Step 2: Get VLM analysis on the rendered views
                with open(pipeline_log_path, 'a', encoding='utf-8') as f:
                    f.write(f"  Iteration {iteration}: Analyzing with VLM...\n")

                enhanced_object_json = links_json.copy() if links_json else {}

                # Get 2D VLM feedback
                vlm_feedback, vlm_success = vlm_critic.analyze_object(
                    description=description,
                    image_paths=image_paths,
                    object_json=enhanced_object_json,
                    output_folder=iter_folder,
                    iteration_num=iteration
                )

                if not vlm_success or not vlm_feedback:
                    logging.warning(f"VLM analysis failed for iteration {iteration}")
                    break

                # Get 3D PointLLM feedback if enabled
                pointllm_feedback = None
                if pointllm_critic is not None:
                    with open(pipeline_log_path, 'a', encoding='utf-8') as f:
                        f.write(f"  Iteration {iteration}: Analyzing with PointLLM 3D...\n")

                    # Check for links_hierarchy.json path
                    links_json_path = None
                    configs_dir = os.path.join(output_folder, 'configs')
                    if os.path.exists(configs_dir):
                        links_hierarchy_path = os.path.join(configs_dir, 'links_hierarchy.json')
                        if os.path.exists(links_hierarchy_path):
                            links_json_path = links_hierarchy_path

                    pointllm_feedback, pointllm_success = pointllm_critic.analyze_object(
                        mesh_path=obj_file,
                        description=description,
                        object_json=enhanced_object_json,
                        output_folder=iter_folder,
                        iteration_num=iteration,
                        links_json_path=links_json_path
                    )

                    if pointllm_success and pointllm_feedback:
                        logging.info(f"PointLLM 3D analysis completed for iteration {iteration}")

                        # Render the point cloud visualization
                        render_success = render_pointcloud_for_iteration(
                            mesh_path=obj_file,
                            output_folder=iter_folder,
                            iteration_num=iteration,
                            description=description,
                            links_json_path=links_json_path,
                            sample_points=8192
                        )
                        if render_success:
                            logging.info(f"Point cloud visualization saved for iteration {iteration}")
                        else:
                            logging.warning(f"Failed to render point cloud visualization for iteration {iteration}")

                # Merge or select feedback
                if feedback_fusion and pointllm_feedback:
                    # Merge 2D and 3D feedback
                    feedback = feedback_fusion.merge_feedback(vlm_feedback, pointllm_feedback)
                    logging.info("Using merged 2D+3D feedback")
                    with open(pipeline_log_path, 'a', encoding='utf-8') as f:
                        f.write(f"  Iteration {iteration}: Merged 2D and 3D feedback\n")
                elif pointllm_feedback and not feedback_fusion:
                    # Use only 3D feedback if fusion is disabled but PointLLM is enabled
                    feedback = pointllm_feedback
                    logging.info("Using only 3D PointLLM feedback")
                else:
                    # Use only 2D feedback (default)
                    feedback = vlm_feedback

                feedback_success = True

                # Log VLM assessment
                with open(pipeline_log_path, 'a', encoding='utf-8') as f:
                    needs_improvement = feedback.needs_improvement
                    confidence = feedback.confidence_score
                    f.write(f"  Iteration {iteration}: VLM Assessment - Needs Improvement: {needs_improvement}, Confidence: {confidence:.2f}\n")

                # Step 3: Check if improvement is needed
                if not feedback.needs_improvement:
                    # VLM says no improvement needed - design is good
                    logging.info(success(f"VLM satisfied with design (confidence: {feedback.confidence_score:.2f})"))
                    with open(pipeline_log_path, 'a', encoding='utf-8') as f:
                        f.write(f"  Iteration {iteration}: VLM satisfied, no improvements needed\n")
                    break

                # VLM suggests improvements - apply them regardless of confidence
                logging.info(f"VLM suggests improvements (confidence: {feedback.confidence_score:.2f})")
                if feedback.specific_issues:
                    logging.info(f"Issues to fix: {', '.join(feedback.specific_issues[:3])}")  # Show first 3 issues

                # Step 4: Improve code based on feedback
                with open(pipeline_log_path, 'a', encoding='utf-8') as f:
                    f.write(f"  Iteration {iteration}: Improving code based on feedback...\n")

                feedback_dict = feedback.model_dump()

                improved_code, fix_success, fix_metrics = shape_fixer.fix_code_with_validation(
                    original_code=current_code,
                    vlm_feedback=feedback_dict,
                    description=description,
                    object_json=links_json,
                    output_folder=iter_folder,  # Use iter_folder instead of stage_folder
                    iteration_num=iteration,
                    max_attempts=2
                )

                if fix_success and improved_code != current_code:
                    current_code = improved_code
                    improved = True
                    logging.info(success(f"Code improved in iteration {iteration}"))
                    with open(pipeline_log_path, 'a', encoding='utf-8') as f:
                        f.write(f"  Iteration {iteration}: Code successfully improved\n")
                else:
                    logging.warning(f"Failed to improve code in iteration {iteration}")
                    with open(pipeline_log_path, 'a', encoding='utf-8') as f:
                        f.write(f"  Iteration {iteration}: Code improvement failed\n")
                    break

            except Exception as e:
                logging.error(f"VLM feedback iteration {iteration} failed: {e}")
                with open(pipeline_log_path, 'a', encoding='utf-8') as f:
                    f.write(f"  Iteration {iteration}: ERROR - {str(e)}\n")
                break

        # Update final code in _export_temp if improved
        if improved and current_code != initial_code:
            export_temp_folder = os.path.join(output_folder, '_export_temp')
            final_export_path = os.path.join(export_temp_folder, 'export.js')
            with open(final_export_path, 'w', encoding='utf-8') as f:
                f.write(current_code)
            logging.info(success("Updated export.js with VLM-improved code"))

            # Update statistics
            if 'vlm_improvements' not in self.stats:
                self.stats['vlm_improvements'] = 0
            self.stats['vlm_improvements'] += 1

        # Log completion
        with open(pipeline_log_path, 'a', encoding='utf-8') as f:
            f.write(f"[{time.strftime('%H:%M:%S')}] STAGE 2.2: VLM Feedback Loop Completed - Improved: {improved}\n")

        return current_code, improved, {}

    def _run_articulation_feedback(self, articulation_json: List[Dict],
                                  output_folder: str, description: str,
                                  pipeline_log_path: str) -> Optional[List[Dict]]:
        """
        Run articulation VLM feedback loop to improve joint configurations.

        Args:
            articulation_json: Initial articulation specification
            output_folder: Output directory
            description: Object description
            pipeline_log_path: Pipeline log file path

        Returns:
            Improved articulation JSON or None if no improvement
        """
        try:
            # Load configuration
            config = self.config.config.get('articulation_feedback', {})
            max_iterations = config.get('max_iterations', 3)
            confidence_threshold = config.get('confidence_threshold', 0.7)

            # Create stage folder
            stage_folder = os.path.join(output_folder, 'pipeline_logs', 'stage_3_1_articulation_feedback')
            os.makedirs(stage_folder, exist_ok=True)

            # Initialize components
            articulation_renderer = ArticulationRenderer(
                image_size=tuple(config.get('image_size', [384, 384])),
                background_color=tuple(config.get('background_color', [0.95, 0.95, 0.95, 1.0]))
            )
            articulation_vlm_critic = ArticulationVLMCritic(self.config)
            articulation_fixer = ArticulationFixer(self.config)

            # Track improvements
            current_articulation = articulation_json

            for iteration in range(1, max_iterations + 1):
                logging.info(f"Articulation feedback iteration {iteration}/{max_iterations}")

                # Create iteration folder
                iter_folder = os.path.join(stage_folder, f"iteration_{iteration}")
                os.makedirs(iter_folder, exist_ok=True)

                # Log to pipeline
                with open(pipeline_log_path, 'a', encoding='utf-8') as f:
                    f.write(f"\n[{time.strftime('%H:%M:%S')}] ARTICULATION FEEDBACK - Iteration {iteration}:\n")

                # First generate URDF with current articulation
                # The renderer expects "generated.urdf" in the base directory
                urdf_path = os.path.join(output_folder, "generated.urdf")
                try:
                    # Save current articulation to configs folder
                    configs_folder = os.path.join(output_folder, "configs")
                    os.makedirs(configs_folder, exist_ok=True)
                    articulation_path = os.path.join(configs_folder, "articulation.json")
                    with open(articulation_path, 'w', encoding='utf-8') as f:
                        json.dump(current_articulation, f, indent=2)

                    # Generate URDF for rendering (ArticulationRenderer expects it at output_folder/generated.urdf)
                    generate_urdf(os.path.join(output_folder, "links"), output_folder, urdf_path)

                    # Also save a copy in iteration folder for reference
                    iter_articulation_path = os.path.join(iter_folder, "current_articulation.json")
                    with open(iter_articulation_path, 'w', encoding='utf-8') as f:
                        json.dump(current_articulation, f, indent=2)

                    # Copy links folder and URDF to iteration folder for visualization
                    links_src = os.path.join(output_folder, "links")
                    links_dst = os.path.join(iter_folder, "links")
                    if os.path.exists(links_src):
                        shutil.copytree(links_src, links_dst, dirs_exist_ok=True)

                    # Copy URDF file
                    iter_urdf_path = os.path.join(iter_folder, "generated.urdf")
                    shutil.copy2(urdf_path, iter_urdf_path)

                    logging.info(f"Saved links and URDF to iteration {iteration} folder")

                except Exception as e:
                    logging.warning(f"Failed to generate URDF for articulation rendering: {e}")
                    break

                # Step 1: Render articulated object with colored child links
                try:
                    image_paths, color_mapping = articulation_renderer.render_articulated_object(
                        output_folder, iter_folder
                    )
                    logging.info(f"Rendered articulation with {len(color_mapping)} colored joints")

                    with open(pipeline_log_path, 'a', encoding='utf-8') as f:
                        f.write(f"  - Rendered {len(image_paths)} images with colored child links\n")

                except Exception as e:
                    logging.error(f"Articulation rendering failed: {e}")
                    break

                # Step 2: Get VLM analysis
                feedback, feedback_success, _, _ = articulation_vlm_critic.analyze_articulation(
                    image_paths=image_paths,
                    color_mapping=color_mapping,
                    articulation_json=current_articulation,
                    description=description,
                    output_folder=iter_folder
                )

                if not feedback_success or not feedback:
                    logging.warning("Articulation VLM analysis failed")
                    break

                # Log assessment
                with open(pipeline_log_path, 'a', encoding='utf-8') as f:
                    f.write(f"  - VLM Assessment: Confidence {feedback.confidence_score:.2f}, ")
                    f.write(f"Needs Improvement: {feedback.needs_improvement}\n")

                # Step 3: Log assessment but always continue to max iterations
                if not feedback.needs_improvement or feedback.confidence_score >= confidence_threshold:
                    logging.info(f"Articulation quality acceptable (confidence: {feedback.confidence_score:.2f})")
                    with open(pipeline_log_path, 'a', encoding='utf-8') as f:
                        f.write(f"  - Quality acceptable at iteration {iteration}, but continuing to max iterations\n")
                else:
                    logging.info(f"Articulation needs improvement (confidence: {feedback.confidence_score:.2f})")
                    with open(pipeline_log_path, 'a', encoding='utf-8') as f:
                        f.write(f"  - Continuing to iteration {iteration + 1} for improvements\n")

                # Always continue unless at max iterations
                if iteration == max_iterations:
                    logging.info(success(f"Completed all {max_iterations} iterations"))
                    with open(pipeline_log_path, 'a', encoding='utf-8') as f:
                        f.write(f"  - Completed all configured iterations\n")
                    break

                # Step 4: Fix articulation issues
                logging.info("Fixing articulation issues...")
                fixed_articulation, fix_success, _, _ = articulation_fixer.fix_articulation(
                    articulation_json=current_articulation,
                    vlm_feedback=feedback.__dict__ if hasattr(feedback, '__dict__') else feedback,
                    color_mapping=color_mapping,
                    description=description,
                    output_folder=iter_folder
                )

                if not fix_success:
                    logging.warning("Articulation fixing failed")
                    break

                # Validate the fixed articulation
                is_valid, errors = articulation_fixer.validate_articulation(fixed_articulation)
                if not is_valid:
                    logging.warning(f"Fixed articulation invalid: {errors}")
                    break

                # Update current articulation for next iteration
                current_articulation = fixed_articulation

                with open(pipeline_log_path, 'a', encoding='utf-8') as f:
                    f.write(f"  - Applied fixes to articulation\n")

            # Return improved articulation if different from original
            if current_articulation != articulation_json:
                logging.info(success("Articulation improved through VLM feedback"))
                return current_articulation
            else:
                return None

        except Exception as e:
            logging.error(f"Articulation feedback loop failed: {e}")
            return None

    def _reorganize_output_files(self, output_folder: str):
        """
        Reorganize and combine mesh files into final structure.
        This is called after all generation steps are complete.
        """
        try:
            # Combine all links meshes into a single file in the object folder
            links_dir = os.path.join(output_folder, 'links')
            if os.path.exists(links_dir) and os.listdir(links_dir):
                combined_assembly_path = os.path.join(output_folder, 'combined_assembly.obj')
                try:
                    combine_meshes(links_dir, combined_assembly_path)
                    logging.debug(f"Combined links meshes to: {combined_assembly_path}")
                except Exception as e:
                    logging.warning(f"Failed to combine links meshes: {e}")
            
            # Combine all part_meshes subfolders into a single file in part_meshes folder
            part_meshes_dir = os.path.join(output_folder, 'part_meshes')
            if os.path.exists(part_meshes_dir):
                combined_parts_path = os.path.join(part_meshes_dir, 'combined_parts.obj')
                try:
                    combine_meshes(part_meshes_dir, combined_parts_path)
                    logging.debug(f"Combined part meshes to: {combined_parts_path}")
                except Exception as e:
                    logging.warning(f"Failed to combine part meshes: {e}")
            
            # Create package.json in _export_temp
            export_temp_dir = os.path.join(output_folder, '_export_temp')
            package_json_path = os.path.join(export_temp_dir, 'package.json')
            
            # Check if package.json exists in root and move it, otherwise create basic one
            root_package_json = os.path.join(output_folder, 'package.json')
            if os.path.exists(root_package_json):
                shutil.move(root_package_json, package_json_path)
            else:
                with open(package_json_path, 'w') as f:
                    json.dump({"type": "module"}, f)
            
            # Write summary to pipeline log
            pipeline_log_path = os.path.join(output_folder, 'pipeline_logs', 'pipeline_run.log')
            with open(pipeline_log_path, 'a', encoding='utf-8') as f:
                f.write("\n" + "=" * 60 + "\n")
                f.write(f"[{time.strftime('%H:%M:%S')}] FILE REORGANIZATION: Completed\n")
                f.write(f"  - Combined links meshes to object folder\n")
                f.write(f"  - Combined part meshes to part_meshes folder\n")
                f.write(f"  - Created package.json in _export_temp\n")
                f.write("=" * 60 + "\n")
                    
        except Exception as e:
            logging.warning(f"File reorganization partially failed: {e}")

    def process_single(self, description: str) -> List[str]:
        """Process a single object description through the pipeline"""
        all_final_meshes = []
        
        for i in range(self.args.num_executions):
            self.stats['total_attempts'] += 1
            logging.info(f"Starting generation {i+1}/{self.args.num_executions} for: {description[:50]}...")
            
            try:
                # Prepare output folder and initialize structure
                base_name = "_".join(description.split()[:5]).lower()
                base_name = re.sub(r'[^\w\s-]', '', base_name)
                output_folder = get_unique_output_folder(base_name, self.config.output_dir)
                
                # Initialize pipeline log
                pipeline_log_dir = os.path.join(output_folder, 'pipeline_logs')
                os.makedirs(pipeline_log_dir, exist_ok=True)
                pipeline_log_path = os.path.join(pipeline_log_dir, 'pipeline_run.log')
                
                # Write pipeline header to log
                with open(pipeline_log_path, 'w', encoding='utf-8') as f:
                    f.write("=" * 60 + "\n")
                    f.write(f"PIPELINE RUN: {description}\n")
                    f.write(f"Timestamp: {time.strftime('%Y-%m-%d %H:%M:%S')}\n")
                    f.write(f"Models: linker={self.config.get_model_for_agent('linker_generator')}, ")
                    f.write(f"shape={self.config.get_model_for_agent('shape_generator')}, ")
                    f.write(f"articulation={self.config.get_model_for_agent('articulation_generator')}\n")
                    f.write("=" * 60 + "\n\n")
                
                # Step 1: Generate links (if enabled)
                links_json = None
                if self.use_linker:
                    try:
                        logging.info("Generating link descriptions...")
                        links_result, links_success, links_metrics, links_raw = self.linker_agent.generate(
                            description, 
                            output_folder=output_folder,
                            log_path=pipeline_log_path,
                            stage_num=1
                        )
                        
                        if not links_success or not links_result:
                            self.stats['failed_generations'] += 1
                            logging.error("Failed to generate links. Skipping this execution.")
                            shutil.rmtree(output_folder, ignore_errors=True)
                            continue
                            
                        links_json = links_result
                        logging.info(success(f"Generated {len(links_json)} links successfully"))
                        
                    except Exception as e:
                        self.stats['failed_generations'] += 1
                        error_type = type(e).__name__
                        self.stats['error_types'][error_type] = self.stats['error_types'].get(error_type, 0) + 1
                        logging.error(f"Link generation failed: {e}")
                        shutil.rmtree(output_folder, ignore_errors=True)
                        continue

                # Step 2: Generate Three.js code
                try:
                    logging.info("Generating Three.js code with export validation...")
                    # Create shape generator (export_only=True for pipeline usage)
                    shape_agent = ShapeGeneratorAgent(self.config, verbose_samples=True)
                    
                    # Generate with export validation and retry mechanism
                    shape_result, shape_success, shape_metrics, shape_raw, retry_count = shape_agent.generate_with_export_validation(
                        max_retries=3,  # Maximum 3 retry attempts
                        articulated_object=description,
                        links_json=links_json,
                        output_folder=output_folder,
                        log_path=pipeline_log_path,
                        stage_num=2
                    )
                    
                    if not shape_success or not shape_result:
                        self.stats['failed_generations'] += 1
                        logging.error(error(f"Failed to generate valid Three.js code after {retry_count} attempts. Skipping this execution."))
                        shutil.rmtree(output_folder, ignore_errors=True)
                        continue
                        
                    self.stats['successful_generations'] += 1
                    if retry_count > 1:
                        logging.info(success(f"Three.js code generated successfully after {retry_count} attempts"))
                    else:
                        logging.info(success("Three.js code generated successfully"))

                    # Step 2.2: VLM Feedback Loop (if enabled)
                    try:
                        logging.info(progress("Running VLM feedback loop..."))
                        final_code, vlm_improved, vlm_metrics = self._run_vlm_feedback_loop(
                            initial_code=shape_result.js_code_export,
                            description=description,
                            output_folder=output_folder,
                            pipeline_log_path=pipeline_log_path,
                            links_json=links_json
                        )

                        if vlm_improved:
                            logging.info(success("Code improved through VLM feedback"))
                            # Update the shape result with improved code
                            shape_result.js_code_export = final_code
                        else:
                            logging.info("No VLM improvements made")

                    except Exception as e:
                        logging.warning(f"VLM feedback loop failed: {e}")
                        # Continue with original code if VLM fails

                except Exception as e:
                    self.stats['failed_generations'] += 1
                    error_type = type(e).__name__
                    self.stats['error_types'][error_type] = self.stats['error_types'].get(error_type, 0) + 1
                    logging.error(f"Shape generation failed: {e}")
                    shutil.rmtree(output_folder, ignore_errors=True)
                    continue

                # Step 3: Setup object logger and extract mesh info
                setup_object_logger(output_folder, self.args.log_level if hasattr(self.args, 'log_level') else "INFO")
                
                try:
                    # Save generation config to configs folder
                    configs_folder = os.path.join(output_folder, 'configs')
                    os.makedirs(configs_folder, exist_ok=True)
                    
                    generation_metrics = {
                        'shape_generation_metrics': shape_metrics,
                        'description': description
                    }
                    if self.use_linker and 'links_metrics' in locals():
                        generation_metrics['linker_generation_metrics'] = links_metrics
                        
                    config_path = os.path.join(configs_folder, 'generation_config.yaml')
                    with open(config_path, 'w', encoding='utf-8') as f:
                        yaml.dump({
                            'description': description,
                            'models': {
                                'linker': self.config.get_model_for_agent('linker_generator'),
                                'shape': self.config.get_model_for_agent('shape_generator'),
                                'articulation': self.config.get_model_for_agent('articulation_generator')
                            },
                            'metrics': generation_metrics
                        }, f, default_flow_style=False)
                    
                    # Extract and merge Three.js information
                    extract_and_merge_threejs_info(output_folder)
                    
                    # Step 4: Export to mesh (code already validated during generation)
                    try:
                        logging.info(progress("Exporting to final mesh format..."))
                        exporter = MeshExporter(output_folder, self.args.final_output)
                        final_mesh, export_success, error_msg = exporter.export_to_obj()
                        
                        if export_success and final_mesh:
                            all_final_meshes.append(final_mesh)
                            self.stats['successful_exports'] += 1
                            logging.info(success(f"Mesh exported successfully: {final_mesh}"))
                        else:
                            # This should rarely happen now since code was pre-validated
                            self.stats['failed_exports'] += 1
                            logging.warning("Export failed despite successful validation - possible race condition")
                            error_type = "Export failure"
                            self.stats['export_errors'][error_type] = self.stats['export_errors'].get(error_type, 0) + 1
                            logging.warning("Export failed but generation was successful. Generated code preserved.")
                            continue
                            
                    except Exception as e:
                        self.stats['failed_exports'] += 1
                        error_type = type(e).__name__
                        self.stats['export_errors'][error_type] = self.stats['export_errors'].get(error_type, 0) + 1
                        logging.error(error(f"Export failed: {e}"))
                        logging.info(f"Generated code preserved in: {output_folder}")
                        continue
                    
                    # Step 5: Generate articulation and URDF (if not disabled)
                    if hasattr(self.args, 'no_articulation') and self.args.no_articulation:
                        logging.info("Articulation generation skipped (--no-articulation flag)")

                        # Still reorganize files for consistency
                        self._reorganize_output_files(output_folder)

                        # Log to pipeline
                        with open(pipeline_log_path, 'a', encoding='utf-8') as f:
                            f.write(f"\n[{time.strftime('%H:%M:%S')}] ARTICULATION: Skipped (--no-articulation flag)\n")
                    else:
                        logging.info(progress("Generating articulation specifications..."))

                        # Define required paths for articulation generation
                        workflow_json_path = os.path.join(output_folder, 'configs', 'workflow.json')
                        export_js_path = os.path.join(output_folder, '_export_temp', 'export.js')

                        # Create articulation agent and generate (pass path directly)
                        articulation_agent = ArticulationGeneratorAgent(self.config)
                        articulation_result, articulation_success, articulation_metrics, articulation_raw = articulation_agent.generate(
                            workflow_json=workflow_json_path,
                            export_js_path=export_js_path,
                            output_folder=output_folder,
                            log_path=pipeline_log_path,
                            stage_num=3
                        )

                        # Update generation config with articulation metrics
                        config_path = os.path.join(output_folder, 'configs', 'generation_config.yaml')
                        if os.path.exists(config_path):
                            with open(config_path, 'r', encoding='utf-8') as f:
                                config_data = yaml.safe_load(f)
                            config_data['metrics']['articulation_generation_metrics'] = articulation_metrics
                            with open(config_path, 'w', encoding='utf-8') as f:
                                yaml.dump(config_data, f, default_flow_style=False)

                        if articulation_success and articulation_result:
                            self.stats['successful_articulations'] += 1
                            logging.info(success("Articulation specifications generated successfully"))

                            # Step 5.1: Articulation VLM Feedback (if enabled)
                            articulation_feedback_config = self.config.config.get('articulation_feedback', {})
                            if articulation_feedback_config.get('enabled', False) and not (hasattr(self.args, 'no_articulation_feedback') and self.args.no_articulation_feedback):
                                logging.info(progress("Starting articulation VLM feedback loop..."))
                                improved_articulation = self._run_articulation_feedback(
                                    articulation_result,
                                    output_folder,
                                    description,
                                    pipeline_log_path
                                )
                                # Use improved articulation if available
                                if improved_articulation:
                                    articulation_result = improved_articulation
                                    # Save improved version to configs
                                    articulation_path = os.path.join(output_folder, 'configs', 'articulation.json')
                                    with open(articulation_path, 'w', encoding='utf-8') as f:
                                        json.dump(articulation_result, f, indent=2)

                            # Reorganize files before URDF generation
                            self._reorganize_output_files(output_folder)

                            # Generate URDF using links folder for mesh references
                            try:
                                links_folder = os.path.join(output_folder, "links")
                                urdf_output_path = os.path.join(output_folder, "generated.urdf")
                                generate_urdf(links_folder, output_folder, urdf_output_path)
                                logging.info(success(f"URDF file generated: {urdf_output_path}"))

                            except Exception as e:
                                logging.warning(f"URDF generation failed: {e}")
                        else:
                            self.stats['failed_articulations'] += 1
                            logging.error(error("Articulation generation failed. See log for details."))
                            
                except Exception as e:
                    self.stats['failed_articulations'] += 1
                    logging.error(f"Articulation/URDF step failed: {e}")

                finally:
                    remove_object_logger()
                    
            except Exception as e:
                error_type = type(e).__name__
                self.stats['error_types'][error_type] = self.stats['error_types'].get(error_type, 0) + 1
                self.stats['failed_generations'] += 1
                logging.error(f"Unexpected error during processing: {e}")
                if 'output_folder' in locals() and os.path.exists(output_folder):
                    shutil.rmtree(output_folder, ignore_errors=True)
                    
            # Log stats periodically
            if i % 5 == 0 or i == self.args.num_executions - 1:
                self._log_stats()
                
        return all_final_meshes


    def process_batch(self) -> None:
        """Process multiple descriptions from a file"""
        descriptions = self._read_captions_from_file(self.args.captions_file)
        logging.info(f"Loaded {len(descriptions)} descriptions from {self.args.captions_file}")

        if self.args.parallel > 0:
            self._process_batch_parallel(descriptions)
        else:
            self._process_batch_sequential(descriptions)

        # Log final statistics
        logging.info("Final Statistics:")
        self._log_stats()

    def _process_batch_sequential(self, descriptions: List[str]) -> None:
        """Process descriptions sequentially"""
        for i, description in enumerate(descriptions, 1):
            logging.info(f"Processing description {i}/{len(descriptions)}: {description}")
            final_meshes = self.process_single(description)
            logging.info(f"Generated {len(final_meshes)} meshes for description {i}")

    def _process_batch_parallel(self, descriptions: List[str]) -> None:
        """Process descriptions in parallel"""
        logging.info(f"Processing batch with {self.args.parallel} workers")
        with concurrent.futures.ThreadPoolExecutor(max_workers=self.args.parallel) as executor:
            future_to_desc = {
                executor.submit(self.process_single, desc): (i, desc)
                for i, desc in enumerate(descriptions, 1)
            }
            
            for future in concurrent.futures.as_completed(future_to_desc):
                i, desc = future_to_desc[future]
                try:
                    meshes = future.result()
                    logging.info(f"Generated {len(meshes)} meshes for description {i}")
                except Exception as e:
                    logging.error(f"Description {i} failed: {e}")

    @staticmethod
    def _read_captions_from_file(file_path: str) -> List[str]:
        """Read object descriptions from a text file"""
        with open(file_path, 'r', encoding='utf-8') as f:
            return [line.strip() for line in f.readlines() if line.strip()]


def main():
    """
    Main entry point for the pipeline
    """
    parser = argparse.ArgumentParser(description='Generate articulated 3D objects from text descriptions')
    
    # Input options
    input_group = parser.add_mutually_exclusive_group(required=True)
    input_group.add_argument('--description', type=str, help='Single object description')
    input_group.add_argument('--captions-file', type=str, help='File containing object descriptions')
    
    # Model configuration
    parser.add_argument('--linker-model', type=str, help='Model for link generation')
    parser.add_argument('--shape-model', type=str, help='Model for shape generation')
    parser.add_argument('--articulation-model', type=str, help='Model for articulation generation')
    
    # Output configuration
    parser.add_argument('--output-dir', type=str, default='output', help='Output directory')
    parser.add_argument('--final-output', type=str, help='Final output directory for combined meshes')
    
    # Execution options
    parser.add_argument('--num-executions', type=int, default=1, help='Number of generation attempts')
    parser.add_argument('--parallel', type=int, default=0, help='Number of parallel workers (0=sequential)')
    parser.add_argument('--no-linker', action='store_true', help='Skip link generation step')
    parser.add_argument('--no-vlm-critic', action='store_true', help='Disable VLM feedback loop')
    parser.add_argument('--no-articulation', action='store_true', help='Skip articulation and URDF generation (shape only)')
    parser.add_argument('--no-articulation-feedback', action='store_true', help='Skip articulation VLM feedback loop')
    
    # Advanced options
    parser.add_argument('--temperature', type=float, help='Global temperature setting')
    parser.add_argument('--max-retries', type=int, help='Maximum retry attempts')
    parser.add_argument('--log-level', type=str, default='INFO', 
                       choices=['DEBUG', 'INFO', 'WARNING', 'ERROR'],
                       help='Logging level')
    
    args = parser.parse_args()
    
    # Set up logging
    setup_global_logging(args.log_level)
    
    # Set use_linker based on --no-linker flag
    args.use_linker = not args.no_linker
    
    # Initialize and run pipeline
    try:
        pipeline = Pipeline(args)
        
        if args.description:
            logging.info(f"Processing single description: {args.description}")
            final_meshes = pipeline.process_single(args.description)
            logging.info(f"Generated {len(final_meshes)} meshes")
        else:
            pipeline.process_batch()
            
        logging.info(success("Pipeline completed successfully!"))
        
    except Exception as e:
        logging.error(error(f"Pipeline failed: {e}"))
        sys.exit(1)


if __name__ == "__main__":
    main()
