const express = require('express');
const path = require('path');
const fs = require('fs');
const cors = require('cors');

const app = express();
const PORT = process.env.PORT || 3001;

// Configuration
const URDF_BASE_PATH = process.env.URDF_BASE_PATH || '/lab/yipeng/generated_art_objects/eval_09_18';

// Force 2-level directory structure
const DIRECTORY_LEVELS = 2;
console.log(`📂 Detected ${DIRECTORY_LEVELS}-level directory structure`);

// Template replacement function
function renderTemplate(templatePath, variables) {
    try {
        let template = fs.readFileSync(templatePath, 'utf8');

        // Replace template variables
        Object.keys(variables).forEach(key => {
            const placeholder = new RegExp(`{{${key}}}`, 'g');
            template = template.replace(placeholder, variables[key]);
        });

        return template;
    } catch (error) {
        console.error('Template render error:', error);
        return `<html><body><h1>Template Error</h1><p>${error.message}</p></body></html>`;
    }
}

// Middleware
app.use(cors());
app.use(express.json());
app.use(express.static(path.join(__dirname, 'public')));

// For 2-level structure, create virtual categories
function getVirtualCategories() {
    if (DIRECTORY_LEVELS === 2) {
        // Group instances by first letter or type
        const instances = fs.readdirSync(URDF_BASE_PATH)
            .filter(item => {
                const itemPath = path.join(URDF_BASE_PATH, item);
                return fs.statSync(itemPath).isDirectory();
                // Removed URDF check - will use fallback if missing
            });

        // Group by object type (e.g., "a_camera_1" -> "camera", "a_bank_vault_safe_with_1" -> "bankvaultsafewith")
        const categories = {};
        instances.forEach(instance => {
            // Extract category from name - more flexible pattern
            // Remove leading "a_" and trailing "_number"
            let category = instance.replace(/^a_/, '').replace(/_\d+$/, '');
            // Remove underscores to create category
            category = category.replace(/_/g, '');

            if (!categories[category]) {
                categories[category] = [];
            }
            categories[category].push(instance);
        });

        return Object.keys(categories).sort();
    } else {
        // Original 3-level structure
        return fs.readdirSync(URDF_BASE_PATH)
            .filter(item => {
                const itemPath = path.join(URDF_BASE_PATH, item);
                return fs.statSync(itemPath).isDirectory();
            })
            .sort();
    }
}

// Get instances for a category
function getInstancesForCategory(category) {
    if (DIRECTORY_LEVELS === 2) {
        // For 2-level, filter instances by category pattern
        const instances = fs.readdirSync(URDF_BASE_PATH)
            .filter(item => {
                const itemPath = path.join(URDF_BASE_PATH, item);
                if (!fs.statSync(itemPath).isDirectory()) return false;
                // Removed URDF check - will use fallback if missing

                // Match category - same logic as above
                let itemCategory = item.replace(/^a_/, '').replace(/_\d+$/, '');
                itemCategory = itemCategory.replace(/_/g, '');
                return itemCategory === category;
            });
        return instances;
    } else {
        // Original 3-level logic
        const categoryPath = path.join(URDF_BASE_PATH, category);
        if (!fs.existsSync(categoryPath)) return [];

        return fs.readdirSync(categoryPath, { withFileTypes: true })
            .filter(entry => {
                if (!entry.isDirectory()) return false;
                const fullPath = path.join(categoryPath, entry.name);
                return fs.existsSync(path.join(fullPath, 'generated.urdf'));
            })
            .map(entry => entry.name);
    }
}

// Resolve instance path
function resolveInstancePath(category, instance) {
    if (DIRECTORY_LEVELS === 2) {
        // For 2-level, instance is directly under base path
        const instancePath = path.join(URDF_BASE_PATH, instance);
        // Always return path if directory exists (fallback will handle missing URDF)
        if (fs.existsSync(instancePath) && fs.statSync(instancePath).isDirectory()) {
            return instancePath;
        }
    } else {
        // Original 3-level
        const instancePath = path.join(URDF_BASE_PATH, category, instance);
        if (fs.existsSync(path.join(instancePath, 'generated.urdf'))) {
            return instancePath;
        }
    }
    return null;
}

// API endpoints
app.get('/api/categories', (req, res) => {
    try {
        const categories = getVirtualCategories();
        console.log(`Found ${categories.length} categories`);
        res.json({ categories });
    } catch (error) {
        console.error('Error in /api/categories:', error);
        res.status(500).json({ error: error.message });
    }
});

app.get('/api/instances/:category', (req, res) => {
    const { category } = req.params;
    try {
        const instances = getInstancesForCategory(category);
        console.log(`Found ${instances.length} instances for category ${category}`);
        res.json({ instances });
    } catch (error) {
        console.error('Error getting instances:', error);
        res.status(500).json({ error: error.message });
    }
});

// Serve URDF files
app.get('/urdf/:category/:instance/:filename', (req, res) => {
    const { category, instance, filename } = req.params;
    const instancePath = resolveInstancePath(category, instance);

    if (!instancePath) {
        return res.status(404).json({ error: 'Instance not found' });
    }

    const filePath = path.join(instancePath, filename);

    if (!fs.existsSync(filePath)) {
        return res.status(404).json({ error: 'File not found' });
    }

    res.sendFile(filePath);
});

// Serve mesh files
app.get('/meshes/:category/:instance/*', (req, res) => {
    const { category, instance } = req.params;
    const meshPath = req.params[0];

    const instancePath = resolveInstancePath(category, instance);

    if (!instancePath) {
        return res.status(404).json({ error: 'Instance not found' });
    }

    const fullMeshPath = path.join(instancePath, 'part_meshes', meshPath);

    if (!fs.existsSync(fullMeshPath)) {
        return res.status(404).json({ error: 'Mesh not found' });
    }

    res.sendFile(fullMeshPath);
});

// Serve files relative to category/instance (critical for URDF loading)
app.get('/urdffs/:category/*', async (req, res) => {
    const { category } = req.params;
    const rel = req.params[0] || '';

    if (DIRECTORY_LEVELS === 2) {
        // For 2-level structure, parse the path differently
        // Expected: /urdffs/category/instance/file.ext
        const parts = rel.split('/');
        const instance = parts[0];
        const filePath = parts.slice(1).join('/');

        // Actual path is directly under URDF_BASE_PATH/instance/
        const fullPath = path.join(URDF_BASE_PATH, instance, filePath);

        if (fs.existsSync(fullPath)) {
            res.sendFile(fullPath);
        } else if (filePath === 'generated.urdf') {
            // Fallback to a simple cube URDF when generated.urdf is missing
            console.log(`URDF not found for ${instance}, using cube fallback`);
            const fallbackURDF = `<?xml version="1.0"?>
<robot name="${instance}_fallback">
  <link name="base_link">
    <visual>
      <origin xyz="0 0 0" rpy="0 0 0"/>
      <geometry>
        <box size="0.1 0.1 0.1"/>
      </geometry>
      <material name="red">
        <color rgba="1 0 0 0.7"/>
      </material>
    </visual>
    <collision>
      <origin xyz="0 0 0" rpy="0 0 0"/>
      <geometry>
        <box size="0.1 0.1 0.1"/>
      </geometry>
    </collision>
  </link>
</robot>`;
            res.set('Content-Type', 'application/xml');
            res.send(fallbackURDF);
        } else {
            console.log(`File not found: ${fullPath}`);
            res.status(404).send('File not found');
        }
    } else {
        // Original 3-level structure
        const fullPath = path.join(URDF_BASE_PATH, category, rel);

        if (fs.existsSync(fullPath)) {
            res.sendFile(fullPath);
        } else {
            res.status(404).send('File not found');
        }
    }
});

// Resolve instance endpoint (critical for preview)
app.get('/resolve-instance/:category/:instance', (req, res) => {
    const { category, instance } = req.params;
    const instancePath = resolveInstancePath(category, instance);

    if (instancePath) {
        // Return relative path from URDF_BASE_PATH
        const relativePath = path.relative(URDF_BASE_PATH, instancePath);
        res.json({
            path: relativePath,
            fullPath: instancePath,
            exists: true
        });
    } else {
        res.status(404).json({
            error: 'Instance not found',
            category,
            instance
        });
    }
});

// Serve views
app.get('/preview', (req, res) => {
    res.sendFile(path.join(__dirname, 'views', 'batch_preview.html'));
});

app.get('/viewer/:category/:instance', (req, res) => {
    const { category, instance } = req.params;
    const instancePath = resolveInstancePath(category, instance);

    if (!instancePath) {
        return res.status(404).send(`
            <html><body>
                <h1>❌ URDF Not Found</h1>
                <p>Category: ${category}</p>
                <p>Instance: ${instance}</p>
            </body></html>
        `);
    }

    // Generate template variables based on directory structure
    let templateVars;

    if (DIRECTORY_LEVELS === 2) {
        // For 2-level structure
        templateVars = {
            category: category,
            instance: instance,
            urdfPath: `/urdffs/${category}/${instance}/generated.urdf`,
            packagePath: `/urdffs/${category}/${instance}/`,
            packageName: instance,
            meshBasePath: `/urdffs/${category}/${instance}/`
        };
    } else {
        // For 3-level structure
        const relPath = path.relative(path.join(URDF_BASE_PATH, category), instancePath);
        templateVars = {
            category: category,
            instance: instance,
            urdfPath: `/urdffs/${category}/${relPath}/generated.urdf`,
            packagePath: `/urdffs/${category}/${relPath}/`,
            packageName: instance,
            meshBasePath: `/urdffs/${category}/${relPath}/`
        };
    }

    // Render template with variables
    const html = renderTemplate(path.join(__dirname, 'views', 'interactive_viewer.html'), templateVars);
    res.send(html);
});

app.get('/', (req, res) => {
    res.redirect('/preview');
});

// Start server
app.listen(PORT, '0.0.0.0', () => {
    console.log('='.repeat(50));
    console.log('🚀 URDF Viewer Server (Multi-Level Support)');
    console.log('='.repeat(50));
    console.log(`📁 URDF base path: ${URDF_BASE_PATH}`);
    console.log(`📂 Directory structure: ${DIRECTORY_LEVELS}-level`);
    console.log(`🌐 Preview: http://localhost:${PORT}/preview`);
    console.log(`🌐 Viewer: http://localhost:${PORT}/viewer/<category>/<instance>`);
    console.log('='.repeat(50));
});