import numpy as np
from scipy.spatial.transform import Rotation as R

import numpy as np
from scipy.spatial.transform import Rotation as R

def rpy_xyz_to_matrix(rpy, xyz):
    """
    Converts RPY angles (radians) and XYZ translation to a 4x4 homogeneous matrix.
    Order of operations: Roll (X), Pitch (Y), Yaw (Z), then translation.
    """
    rotation_matrix = R.from_euler('xyz', rpy, degrees=False).as_matrix()
    transform_matrix = np.eye(4)
    transform_matrix[:3, :3] = rotation_matrix
    transform_matrix[:3, 3] = xyz
    return transform_matrix

def matrix_to_rpy_xyz(matrix):
    """
    Converts a 4x4 homogeneous matrix back to RPY angles (radians) and XYZ translation.
    """
    rotation_matrix = matrix[:3, :3]
    xyz = matrix[:3, 3]
    rpy = R.from_matrix(rotation_matrix).as_euler('xyz', degrees=False)
    return rpy, xyz

def compute_child_visual_origin_offset_with_link_offset(
    joint_origin_xyz, joint_origin_rpy,
    link_offset_xyz=np.array([0.0, 0.0, 0.0]), link_offset_rpy=np.array([0.0, 0.0, 0.0]),
    parent_offset_xyz=np.array([0.0, 0.0, 0.0]), parent_offset_rpy=np.array([0.0, 0.0, 0.0])
):
    """
    Computes the offset (xyz, rpy) for the child link's <visual> origin so that the visual frame coincides with the parent link frame,
    considering an initial offset of the child link frame relative to the joint frame, and recursively considering parent offset.

    Args:
        joint_origin_xyz (np.ndarray): XYZ translation of the joint frame relative to the parent frame (e.g., from URDF).
        joint_origin_rpy (np.ndarray): RPY angles (radians) of the joint frame relative to the parent frame (e.g., from URDF).
        link_offset_xyz (np.ndarray, optional): Initial XYZ translation of the child link frame relative to the joint frame. Defaults to [0.0, 0.0, 0.0].
        link_offset_rpy (np.ndarray, optional): Initial RPY angles (radians) of the child link frame relative to the joint frame. Defaults to [0.0, 0.0, 0.0].
        parent_offset_xyz (np.ndarray, optional): Cumulative XYZ translation from all parent joints. Defaults to [0.0, 0.0, 0.0].
        parent_offset_rpy (np.ndarray, optional): Cumulative RPY angles from all parent joints. Defaults to [0.0, 0.0, 0.0].

    Returns:
        tuple: (visual_origin_xyz, visual_origin_rpy), where each is a list.
    """
    # Transformation from root to parent
    T_root_to_parent = rpy_xyz_to_matrix(parent_offset_rpy, parent_offset_xyz)
    # Transformation from parent to joint
    T_parent_to_joint = rpy_xyz_to_matrix(joint_origin_rpy, joint_origin_xyz)
    # Transformation from joint to link (initial offset of the child link)
    T_joint_to_link = rpy_xyz_to_matrix(link_offset_rpy, link_offset_xyz)

    # Transformation from root to link
    T_root_to_link = T_root_to_parent @ T_parent_to_joint @ T_joint_to_link

    # We want the visual frame to coincide with the root frame.
    # Let T_link_to_visual be the transform of the <visual> origin.
    # We need T_root_to_link * T_link_to_visual = Identity
    # Therefore, T_link_to_visual = inverse(T_root_to_link)

    T_link_to_visual = np.linalg.inv(T_root_to_link)
    visual_origin_rpy, visual_origin_xyz = matrix_to_rpy_xyz(T_link_to_visual)
    return visual_origin_xyz.tolist(), visual_origin_rpy.tolist()

"""
# --- Example Usage with a non-zero child link offset ---
joint_origin_xyz = np.array([0.0, 0.02, 0.125])
joint_origin_rpy = np.array([1.57, 0.0, 0.0])

link_offset_xyz = np.array([0.01, -0.01, 0.05]) # Example non-zero offset
link_offset_rpy = np.array([0.1, -0.1, 0.2])   # Example non-zero rotation

calculated_visual_origin_xyz, calculated_visual_origin_rpy = compute_child_visual_origin_offset_with_link_offset(
    joint_origin_xyz, joint_origin_rpy, link_offset_xyz, link_offset_rpy
)

print("Joint Origin (xyz):", joint_origin_xyz)
print("Joint Origin (rpy, radians):", joint_origin_rpy)
print("Child Link Offset (xyz):", link_offset_xyz)
print("Child Link Offset (rpy, radians):", link_offset_rpy)
print("-" * 20)
print("Calculated Visual Origin to make screen model origin coincide with body link origin:")
print("Visual Origin (xyz):", np.round(calculated_visual_origin_xyz, 6))
print("Visual Origin (rpy, radians):", np.round(calculated_visual_origin_rpy, 6))

calculated_visual_origin_rpy_deg = np.degrees(calculated_visual_origin_rpy)
print("Visual Origin (rpy, degrees):", np.round(calculated_visual_origin_rpy_deg, 6))
"""

"""
def rpy_xyz_to_matrix(rpy, xyz):

  # Converts RPY angles (radians) and XYZ translation to a 4x4 homogeneous matrix.
  # Order of operations: Roll (X), Pitch (Y), Yaw (Z), then translation.
  
  rotation_matrix = R.from_euler('xyz', rpy, degrees=False).as_matrix()
  transform_matrix = np.eye(4)
  transform_matrix[:3, :3] = rotation_matrix
  transform_matrix[:3, 3] = xyz
  return transform_matrix

def matrix_to_rpy_xyz(matrix):
  
  # Converts a 4x4 homogeneous matrix back to RPY angles (radians) and XYZ translation.

  rotation_matrix = matrix[:3, :3]
  xyz = matrix[:3, 3]
  rpy = R.from_matrix(rotation_matrix).as_euler('xyz', degrees=False)
  return rpy, xyz

def compute_child_visual_origin_offset(joint_origin_xyz, joint_origin_rpy):

  #  Given the joint origin xyz and rpy (in radians), compute the offset (xyz, rpy)
  # for the child link's <visual> origin so that the visual frame coincides with the parent link frame.
  # Returns (visual_origin_xyz, visual_origin_rpy)
  T_parent_to_child = rpy_xyz_to_matrix(joint_origin_rpy, joint_origin_xyz)
  T_child_to_visual = np.linalg.inv(T_parent_to_child)
  visual_origin_rpy, visual_origin_xyz = matrix_to_rpy_xyz(T_child_to_visual)
  return visual_origin_xyz.tolist(), visual_origin_rpy.tolist()

"""

"""
# --- Your joint origin values from the URDF ---
# Note: rpy in URDF is in radians
joint_origin_xyz = np.array([0.0, 0.02, 0.125])
joint_origin_rpy = np.array([1.57, 0.0, 0.0]) # 1.57 is approximately pi/2 radians

# 1. Calculate the transformation matrix for the joint origin (Parent -> Child link frame)
T_parent_to_child = rpy_xyz_to_matrix(joint_origin_rpy, joint_origin_xyz)

# 2. Calculate the required visual origin transformation matrix (Child link frame -> Visual frame)
#    Desired: T_parent_to_child * T_child_to_visual = Identity (Parent -> Desired Visual Location)
#    Solving for T_child_to_visual: T_child_to_visual = inverse(T_parent_to_child)
T_child_to_visual = np.linalg.inv(T_parent_to_child)

# 3. Convert the calculated matrix back to xyz and rpy
calculated_visual_origin_rpy, calculated_visual_origin_xyz = matrix_to_rpy_xyz(T_child_to_visual)

# --- Output Results ---
print("Joint Origin (xyz):", joint_origin_xyz)
print("Joint Origin (rpy, radians):", joint_origin_rpy)
print("-" * 20)
print("Calculated Visual Origin to make screen model origin coincide with body link origin:")
print("Visual Origin (xyz):", np.round(calculated_visual_origin_xyz, 6)) # Round results due to potential float inaccuracies
print("Visual Origin (rpy, radians):", np.round(calculated_visual_origin_rpy, 6))

# If you need RPY in degrees
calculated_visual_origin_rpy_deg = np.degrees(calculated_visual_origin_rpy)
print("Visual Origin (rpy, degrees):", np.round(calculated_visual_origin_rpy_deg, 6))
"""