"""
Unified configuration manager for the articulated object generation pipeline.

This manager provides centralized configuration handling for all agents and providers,
supporting automatic model detection, intelligent defaults, and hierarchical overrides.
It eliminates configuration complexity while ensuring consistent behavior across the system.

Key Features:
- YAML-based configuration with intelligent defaults
- Hierarchical configuration (defaults → model-specific → user overrides)
- Automatic model-to-provider mapping via ModelDetector
- Secure API key management
- Generation metadata tracking and persistence
- Comprehensive logging configuration

Configuration Hierarchy (highest priority first):
    1. User explicit overrides in method calls
    2. Model-specific overrides in config.yaml
    3. Global defaults in config.yaml
    4. System built-in defaults

Configuration File Structure:
    api:
      keys:
        openai: "sk-xxx"
        google: "AIza-xxx"
      agents:
        linker_generator: "o3"
        shape_generator: "gpt-4o" 
        articulation_generator: "gemini-2.5-pro"
      defaults:
        temperature: 0.7
        max_tokens: 4000
      overrides:
        o3:
          temperature: 1.0  # O3 requires temperature=1.0

Example Usage:
    config = ConfigManager()
    
    # Get model for specific agent
    model = config.get_model_for_agent('linker_generator')  # Returns "o3"
    
    # Get complete model configuration with all overrides applied
    model_config = config.get_model_config('o3')  # Returns merged config
    
    # Save generation results with metadata
    config.save_generation_config(output_dir, "a clock", metrics)
"""

import yaml
import os
import logging
from typing import Dict, Any, Optional
from datetime import datetime
from providers.model_detector import ModelDetector


class ConfigManager:
    """Manages configuration with automatic model detection"""
    
    def __init__(self, config_path: str = "config.yaml"):
        """Initialize configuration manager"""
        self.config_path = config_path
        self.config = self._load_config()
        self._model_cache = {}
        self._setup_logging()
    
    def _load_config(self) -> Dict[str, Any]:
        """Load configuration from YAML file"""
        try:
            with open(self.config_path, 'r') as f:
                return yaml.safe_load(f)
        except FileNotFoundError:
            raise FileNotFoundError(f"Config file not found at {self.config_path}")
        except yaml.YAMLError as e:
            raise ValueError(f"Error parsing config file: {e}")
    
    def _setup_logging(self):
        """Setup logging based on configuration"""
        if self.config.get('logging', {}).get('enabled', True):
            logging.basicConfig(
                level=getattr(logging, self.config['logging'].get('level', 'INFO')),
                format='%(asctime)s - %(name)s - %(levelname)s - %(message)s'
            )
    
    def get_api_key(self, provider: str) -> Optional[str]:
        """
        Get API key for a provider
        
        Args:
            provider: Provider name ('openai' or 'google')
            
        Returns:
            API key or None if not found
        """
        return self.config.get('api', {}).get('keys', {}).get(provider)
    
    def get_model_for_agent(self, agent_type: str) -> str:
        """
        Get model name for a specific agent
        
        Args:
            agent_type: Type of agent (e.g., 'linker_generator', 'shape_generator')
            
        Returns:
            Model name
        """
        model = self.config.get('api', {}).get('agents', {}).get(agent_type)
        if not model:
            raise ValueError(f"No model configured for agent: {agent_type}")
        return model
    
    def get_model_config(self, model_name: str) -> Dict[str, Any]:
        """
        Get complete configuration for a model
        
        Args:
            model_name: Name of the model
            
        Returns:
            Configuration dictionary with defaults and overrides applied
        """
        # Check cache
        if model_name in self._model_cache:
            return self._model_cache[model_name].copy()
        
        # Build configuration
        config = {}
        
        # Apply defaults
        defaults = self.config.get('api', {}).get('defaults', {})
        config.update(defaults)
        
        # Apply model-specific overrides
        overrides = self.config.get('api', {}).get('overrides', {})
        
        # Check for exact match
        if model_name in overrides:
            config.update(overrides[model_name])
        else:
            # Check for prefix match
            model_lower = model_name.lower()
            for override_key, override_values in overrides.items():
                if model_lower.startswith(override_key.lower()):
                    config.update(override_values)
                    break
        
        # Add model name
        config['model_name'] = model_name
        
        # Cache and return
        self._model_cache[model_name] = config
        return config.copy()
    
    def get_retry_config(self) -> Dict[str, Any]:
        """Get retry configuration"""
        return self.config.get('retry', {
            'max_retries': 3,
            'multiplier': 1,
            'min_wait': 4,
            'max_wait': 10
        })
    
    @property
    def output_dir(self) -> str:
        """Get output directory"""
        return self.config.get('output', {}).get('base_dir', './output')
    
    @output_dir.setter
    def output_dir(self, value: str):
        """Set output directory"""
        if 'output' not in self.config:
            self.config['output'] = {}
        self.config['output']['base_dir'] = value
    
    @property
    def output_visualization(self) -> bool:
        """Check if visualization output is enabled"""
        return self.config.get('output', {}).get('output_visualization', False)
    
    @property
    def save_config(self) -> bool:
        """Check if config saving is enabled"""
        return self.config.get('output', {}).get('save_config', True)
    
    def save_generation_config(self, output_dir: str, description: str, metrics: dict = None):
        """
        Save generation configuration to output directory
        
        Args:
            output_dir: Directory to save config to
            description: Description of the generation
            metrics: Optional metrics to include
        """
        if not self.save_config:
            return
        
        generation_info = {
            'timestamp': datetime.now().isoformat(),
            'description': description,
            'models': {
                agent: self.get_model_for_agent(agent)
                for agent in ['linker_generator', 'shape_generator', 'articulation_generator']
                if agent in self.config.get('api', {}).get('agents', {})
            },
            'configuration': {
                'defaults': self.config.get('api', {}).get('defaults', {}),
                'retry': self.get_retry_config(),
            }
        }
        
        if metrics:
            generation_info['metrics'] = metrics
        
        config_path = os.path.join(output_dir, 'generation_config.yaml')
        os.makedirs(output_dir, exist_ok=True)
        
        with open(config_path, 'w') as f:
            yaml.dump(generation_info, f, default_flow_style=False, sort_keys=False)
    
    def update_generation_config(self, output_dir: str, update_dict: dict):
        """
        Update existing generation config with new data
        
        Args:
            output_dir: Directory containing config
            update_dict: Dictionary with updates
        """
        config_path = os.path.join(output_dir, 'generation_config.yaml')
        if not os.path.exists(config_path):
            # Create new if doesn't exist
            self.save_generation_config(output_dir, '', update_dict)
            return
        
        with open(config_path, 'r') as f:
            config = yaml.safe_load(f) or {}
        
        # Deep merge
        def deep_update(d, u):
            for k, v in u.items():
                if isinstance(v, dict) and k in d and isinstance(d[k], dict):
                    deep_update(d[k], v)
                else:
                    d[k] = v
        
        deep_update(config, update_dict)
        
        with open(config_path, 'w') as f:
            yaml.dump(config, f, default_flow_style=False, sort_keys=False)
