// Usage: node extract_threejs_info.js path/to/export.js

import * as THREE from 'three';
import fs from 'fs';
import path from 'path';
import { pathToFileURL } from 'url';

async function loadSceneModule(modulePath) {
    const sceneModuleURL = pathToFileURL(path.resolve(modulePath)).href;
    const mod = await import(sceneModuleURL);
    if (typeof mod.createScene === 'function') {
        return mod.createScene();
    } else if (mod.sceneObject) {
        return mod.sceneObject;
    } else {
        throw new Error('Module must export createScene() or sceneObject');
    }
}

function extractGeometryInfo(geometry) {
    if (!geometry) return null;
    let type = geometry.type || geometry.constructor.name;
    let params = {};
    // Try to extract parameters for common geometry types
    if (geometry instanceof THREE.BoxGeometry) {
        params = {
            width: geometry.parameters.width,
            height: geometry.parameters.height,
            depth: geometry.parameters.depth
        };
    } else if (geometry instanceof THREE.CylinderGeometry) {
        params = {
            radiusTop: geometry.parameters.radiusTop,
            radiusBottom: geometry.parameters.radiusBottom,
            height: geometry.parameters.height,
            radialSegments: geometry.parameters.radialSegments
        };
    } else if (geometry instanceof THREE.SphereGeometry) {
        params = {
            radius: geometry.parameters.radius,
            widthSegments: geometry.parameters.widthSegments,
            heightSegments: geometry.parameters.heightSegments
        };
    } else if (geometry.parameters) {
        params = geometry.parameters;
    }
    // You can add more geometry types as needed
    return { type, parameters: params };
}

function extractTransform(obj) {
    return {
        position: obj.position ? [obj.position.x, obj.position.y, obj.position.z] : [0,0,0],
        rotation: obj.rotation ? [obj.rotation.x, obj.rotation.y, obj.rotation.z] : [0,0,0],
        scale: obj.scale ? [obj.scale.x, obj.scale.y, obj.scale.z] : [1,1,1]
    };
}

function extractMeshInfo(mesh) {
    return {
        name: mesh.name || '',
        geometry: extractGeometryInfo(mesh.geometry),
        transformation: extractTransform(mesh),
        material: mesh.material ? (mesh.material.type || mesh.material.constructor.name) : undefined
        // You can add more fields as needed
    };
}

function extractGroupInfo(group) {
    // Calculate bounding box
    let boundingBox = null;
    try {
        const box = new THREE.Box3().setFromObject(group);
        boundingBox = {
            min: [box.min.x, box.min.y, box.min.z],
            max: [box.max.x, box.max.y, box.max.z],
            size: [
                box.max.x - box.min.x,
                box.max.y - box.min.y,
                box.max.z - box.min.z
            ],
            center: [
                (box.max.x + box.min.x) / 2,
                (box.max.y + box.min.y) / 2,
                (box.max.z + box.min.z) / 2
            ]
        };
    } catch (e) {
        boundingBox = null;
    }
    
    // Get group's own transformation
    const transformation = extractTransform(group);
    
    return {
        name: group.name || '',
        boundingBox,
        transformation
    };
}

function collectGroupsWithMeshes(rootObject) {
    // Apply the same filtering logic as threejs_to_mesh.js
    const linkMeshMap = {};
    const processedMeshes = new Set();
    const orphanedMeshes = [];
    
    // First, identify all named groups (these are our links)
    const namedGroups = [];
    rootObject.traverse(obj => {
        if (obj instanceof THREE.Group && obj.name && obj.name !== '') {
            namedGroups.push(obj);
            // Initialize the map entry for this link
            linkMeshMap[obj.name] = {
                group: obj,
                meshes: []
            };
        }
    });
    
    // Traverse and assign meshes to their parent links
    rootObject.traverse(obj => {
        if (obj instanceof THREE.Mesh && !processedMeshes.has(obj)) {
            // Find the closest parent group (link) for this mesh
            let parent = obj.parent;
            let parentLink = null;
            
            while (parent) {
                if (parent instanceof THREE.Group && parent.name && parent.name !== '') {
                    parentLink = parent;
                    break;
                }
                parent = parent.parent;
            }
            
            if (parentLink && linkMeshMap[parentLink.name]) {
                linkMeshMap[parentLink.name].meshes.push(obj);
                processedMeshes.add(obj);
            } else {
                // Mesh doesn't belong to any named group
                orphanedMeshes.push(obj);
                processedMeshes.add(obj);
            }
        }
    });
    
    // Add orphaned meshes if any exist
    if (orphanedMeshes.length > 0) {
        linkMeshMap['_orphaned_meshes'] = {
            group: null,
            meshes: orphanedMeshes
        };
    }
    
    // Filter out container groups that have no meshes (same logic as threejs_to_mesh.js)
    const filteredGroups = [];
    for (const [linkName, linkData] of Object.entries(linkMeshMap)) {
        if (linkData.meshes.length > 0) {
            filteredGroups.push(linkData.group || { name: linkName }); // Handle orphaned case
            console.log(`  Including group "${linkName}": ${linkData.meshes.length} meshes`);
        } else {
            console.log(`  Skipping empty container: "${linkName}" (0 meshes)`);
        }
    }
    
    return filteredGroups.filter(group => group !== null); // Remove null entries (orphaned case)
}

async function main() {
    const [,, exportJsPath] = process.argv;
    if (!exportJsPath) {
        console.error('Usage: node extract_threejs_info.js path/to/export.js');
        process.exit(1);
    }
    const root = await loadSceneModule(exportJsPath);

    // Update world matrices to ensure consistent bounding box calculations
    root.updateMatrixWorld(true);

    // Collect only groups with meshes (same filtering logic as threejs_to_mesh.js)
    console.log('Analyzing scene structure for groups with meshes...');
    const groupsWithMeshes = collectGroupsWithMeshes(root);

    const result = groupsWithMeshes.map(g => extractGroupInfo(g));

    // Display info about the groups being saved
    console.log(`\nFiltered to ${result.length} groups with meshes for workflow.json:`);
    result.forEach((group, idx) => {
        console.log(`  Group #${idx + 1}: Name = "${group.name}"`);
    });

    // workflow.json is always in configs folder
    const exportDir = path.dirname(exportJsPath);
    const projectDir = exportDir.endsWith('_export_temp') ? path.dirname(exportDir) : exportDir;
    const workflowJsonPath = path.join(projectDir, 'configs', 'workflow.json');
    
    if (!fs.existsSync(workflowJsonPath)) {
        console.error('workflow.json not found at:', workflowJsonPath);
        process.exit(1);
    }
    let workflow = {};
    try {
        workflow = JSON.parse(fs.readFileSync(workflowJsonPath, 'utf-8'));
    } catch (e) {
        console.error('Failed to read workflow.json:', e);
        process.exit(1);
    }
    workflow['threejs_info'] = result;
    try {
        fs.writeFileSync(workflowJsonPath, JSON.stringify(workflow, null, 2));
        console.log('Merged Three.js info into', workflowJsonPath);
    } catch (e) {
        console.error('Failed to write workflow.json:', e);
        process.exit(1);
    }
}

main().catch(e => { console.error(e); process.exit(1); }); 