"""
ShapeCodeFixAgent for fixing JavaScript syntax errors in generated Three.js code.

This agent uses lightweight LLM models to quickly fix syntax errors in generated
Three.js code, avoiding the need for expensive full regeneration.
"""

import os
import re
import logging
from utils.simple_colors import success, progress, error, warning, stats
import tempfile
import shutil
import subprocess
from typing import Dict, Any, Optional, Tuple
from agents.base_agent import BaseAgent
from utils.output_parser import OutputFormatError


class FixedCodeResult:
    """Result container for fixed JavaScript code."""
    
    def __init__(self, js_code_export: str):
        self.js_code_export = js_code_export
        self.fixed_code = js_code_export  # Alias for compatibility
    
    def __str__(self):
        return self.js_code_export


class ShapeCodeFixAgent(BaseAgent):
    """
    Agent for fixing JavaScript syntax errors in Three.js code.
    
    Uses lightweight models (gpt-4o-mini, gemini-2.5-flash) for fast, 
    cost-effective syntax fixes without altering geometric definitions.
    """
    
    def __init__(self, config_manager):
        """
        Initialize the ShapeCodeFixAgent.
        
        Args:
            config_manager: Configuration manager instance
        """
        self.logger = logging.getLogger(self.__class__.__name__)
        
        # Initialize with shape_code_fixer agent type
        super().__init__(config_manager, 'shape_code_fixer')
        
    def _load_system_prompt(self) -> str:
        """Load system prompt for JavaScript syntax fixing."""
        try:
            from prompt.javascript_fixer import system_prompt
            return system_prompt
        except ImportError as e:
            self.logger.error(f"Failed to load JavaScript fixer prompt: {e}")
            # Fallback prompt
            return """Fix JavaScript syntax errors in the provided Three.js code.
            Only fix syntax issues, do not modify geometric definitions or functionality.
            Return the complete fixed code wrapped in <fixed_code> tags."""
    
    def _format_user_prompt(self, input_data: Dict[str, Any]) -> str:
        """
        Format user prompt with JavaScript code and error message.
        
        Args:
            input_data: Dictionary containing js_code and error_message
            
        Returns:
            Formatted user prompt
        """
        js_code = input_data.get('js_code', input_data.get('broken_js_code', ''))
        error_message = input_data.get('error_message', input_data.get('error_context', 'Unknown error'))
        attempt_num = input_data.get('attempt_num', 1)
        
        # Analyze error type for better context
        error_context = ""
        if "is not a function" in error_message:
            error_context = "\nThis appears to be a Three.js API error. Check if the method exists on the object type."
            if ".translate" in error_message:
                error_context += "\nNote: THREE.Shape objects don't have translate() method. Create paths with pre-translated coordinates instead."
            elif ".getPoints" in error_message:
                error_context += "\nNote: getPoints() may not exist on this object. Use appropriate methods like extractPoints() for Shapes."
        elif "has already been declared" in error_message:
            error_context = "\nThis is a duplicate variable declaration. Rename the duplicate variable to make it unique."
        elif "before initialization" in error_message or "Cannot access" in error_message:
            error_context = "\nThis is a variable ordering issue. Move declarations before their usage."
        elif "missing ) after argument list" in error_message:
            error_context = "\nThis is a syntax error - missing closing parenthesis. Check all function calls and array/object literals for proper closing."
        elif "Unexpected token" in error_message:
            error_context = "\nThis is a syntax error. Check for missing commas, semicolons, or brackets."
        elif "Unexpected end of input" in error_message:
            error_context = "\nThis indicates missing closing brackets, braces, or parentheses at the end of the file."
        
        prompt = f"""Please fix the following JavaScript Three.js code that has an error.

Error Message:
{error_message}{error_context}

Export Context:
The code will be executed by Node.js using utils/threejs_to_mesh.js
It must export a function named 'createScene' that returns a THREE.Object3D (Group or Mesh).

JavaScript Code to Fix:
```javascript
{js_code}
```

This is attempt {attempt_num} to fix this code. Please carefully analyze the error and apply targeted fixes. 
Focus on fixing the specific error mentioned while preserving all geometric definitions and design intent."""
        
        return prompt
    
    def parse_response(self, response: str) -> FixedCodeResult:
        """
        Parse LLM response to extract fixed JavaScript code.
        
        Args:
            response: Raw LLM response
            
        Returns:
            FixedCodeResult containing the fixed JavaScript code
            
        Raises:
            OutputFormatError: If fixed_code block is missing
        """
        fixed_code = None
        
        # Look for code within <js_code_export> tags first (consistent with main generator)
        match = re.search(r'<js_code_export>(.*?)</js_code_export>', response, re.DOTALL)
        
        if match:
            fixed_code = match.group(1).strip()
            self.logger.debug("Found code in <js_code_export> tags")
        else:
            # Fallback: look for legacy <fixed_code> tags
            match = re.search(r'<fixed_code>(.*?)</fixed_code>', response, re.DOTALL)
            if match:
                fixed_code = match.group(1).strip()
                self.logger.debug("Found code in <fixed_code> tags (legacy)")
            else:
                # Fallback: look for code blocks with language specifier
                code_block_match = re.search(r'```(?:javascript|js)\s*\n(.*?)```', response, re.DOTALL)
                if code_block_match:
                    fixed_code = code_block_match.group(1).strip()
                    self.logger.warning("Fixed code not in expected tags, extracted from ```javascript block")
                else:
                    # Try generic code blocks
                    generic_block_match = re.search(r'```\s*\n(.*?)```', response, re.DOTALL)
                    if generic_block_match:
                        fixed_code = generic_block_match.group(1).strip()
                        self.logger.warning("Fixed code extracted from generic code block")
                    else:
                        # Last resort: if response looks like JavaScript code, use it directly
                        if 'function' in response or 'const' in response or 'import' in response:
                            fixed_code = response.strip()
                            self.logger.warning("Using raw response as fixed code")
        
        if not fixed_code:
            raise OutputFormatError("No fixed JavaScript code found in response")
        
        # Clean the extracted code
        cleaned_code = self._clean_javascript_code(fixed_code)
        
        if not cleaned_code.strip():
            raise OutputFormatError("Extracted code is empty after cleaning")
            
        return FixedCodeResult(cleaned_code)
    
    def _clean_javascript_code(self, code: str) -> str:
        """
        Clean JavaScript code by removing markdown formatting and other artifacts.
        
        Args:
            code: Raw JavaScript code that may contain formatting artifacts
            
        Returns:
            Clean JavaScript code
        """
        # Remove any remaining markdown code block markers
        code = re.sub(r'^```(?:javascript|js)?\s*\n', '', code, flags=re.MULTILINE)
        code = re.sub(r'\n```\s*$', '', code, flags=re.MULTILINE)
        code = re.sub(r'^```\s*$', '', code, flags=re.MULTILINE)
        
        # Remove leading/trailing markdown markers that might be on the same line
        code = re.sub(r'^```(?:javascript|js)?', '', code)
        code = re.sub(r'```$', '', code)
        
        # Split into lines for more precise cleaning
        lines = code.split('\n')
        cleaned_lines = []
        
        for line in lines:
            # Skip lines that are just markdown markers
            if re.match(r'^\s*```(?:javascript|js)?\s*$', line):
                continue
            if re.match(r'^\s*```\s*$', line):
                continue
                
            # Keep the line
            cleaned_lines.append(line)
        
        # Rejoin and clean up whitespace
        cleaned_code = '\n'.join(cleaned_lines)
        
        # Remove excessive blank lines (more than 2 consecutive)
        cleaned_code = re.sub(r'\n{3,}', '\n\n', cleaned_code)
        
        # Strip leading and trailing whitespace
        cleaned_code = cleaned_code.strip()
        
        self.logger.debug(f"Cleaned code: removed formatting artifacts, {len(code) - len(cleaned_code)} chars removed")
        
        return cleaned_code
    
    def _prepare_input_data(self, js_code: str, error_message: str, 
                           attempt_num: int = 1, **kwargs) -> Dict[str, Any]:
        """
        Prepare input data for the fix attempt.
        
        Args:
            js_code: JavaScript code with syntax errors
            error_message: Error message from export attempt
            attempt_num: Current fix attempt number
            **kwargs: Additional arguments
            
        Returns:
            Dictionary of input data
        """
        return {
            'js_code': js_code,
            'error_message': error_message,
            'attempt_num': attempt_num
        }
    
    def save_output(self, result: str, output_folder: str, metrics: Dict[str, Any] = None):
        """
        Save fixed JavaScript code to file.
        
        Args:
            result: Fixed JavaScript code
            output_folder: Directory to save output
            metrics: Generation metrics (optional)
        """
        if not result:
            return
            
        # Save to _export_temp folder
        export_temp_folder = os.path.join(output_folder, '_export_temp')
        os.makedirs(export_temp_folder, exist_ok=True)
        
        # Save as export_fixed.js
        fixed_path = os.path.join(export_temp_folder, 'export_fixed.js')
        with open(fixed_path, 'w', encoding='utf-8') as f:
            f.write(result)
        
        self.logger.info(f"Saved fixed code to {fixed_path}")
    
    def _save_fix_attempt(self, js_code: str, output_folder: str, 
                         attempt_num: int, fix_attempt: int,
                         success: Optional[bool] = None, error: Optional[str] = None):
        """
        Save a fix attempt (successful or failed) with descriptive naming.
        
        Args:
            js_code: JavaScript code from this fix attempt
            output_folder: Directory to save output
            attempt_num: Main attempt number from shape_generator_agent
            fix_attempt: Fix attempt number within this fix session
            success: Whether fix was successful (None if unknown yet)
            error: Error message if failed
        """
        if not js_code:
            return
            
        # Save to _export_temp folder
        export_temp_folder = os.path.join(output_folder, '_export_temp')
        os.makedirs(export_temp_folder, exist_ok=True)
        
        # Create descriptive filename
        filename = f"export_fix_attempt_{attempt_num}_fix_{fix_attempt}.js"
        file_path = os.path.join(export_temp_folder, filename)
        
        # Add header comment with status
        header_lines = [
            f"// FIX ATTEMPT - Attempt {attempt_num}, Fix {fix_attempt}",
            f"// Time: {os.popen('date').read().strip()}",
        ]
        
        if success is False and error:
            header_lines.append(f"// Status: FAILED - {error}")
        elif success is True:
            header_lines.append("// Status: SUCCESS")
        else:
            header_lines.append("// Status: PENDING VALIDATION")
        
        header_lines.append("// " + "=" * 50)
        header_lines.append("")
        
        # Write file with header
        with open(file_path, 'w', encoding='utf-8') as f:
            f.write('\n'.join(header_lines))
            f.write(js_code)
        
        status_str = "failed" if success is False else "successful" if success is True else "pending"
        self.logger.info(f"Saved {status_str} fix attempt to {filename}")
    
    def fix_javascript_code(self, js_code: str, error_message: str,
                          max_attempts: int = 2, 
                          output_folder: Optional[str] = None,
                          attempt_num: int = 1) -> Tuple[str, bool, Dict[str, Any]]:
        """
        Main method to fix JavaScript code with syntax errors.
        
        Args:
            js_code: JavaScript code with syntax errors
            error_message: Error message from failed export
            max_attempts: Maximum fix attempts (default: 2)
            output_folder: Optional output folder for saving fixes
            attempt_num: The main attempt number from shape_generator_agent
            
        Returns:
            Tuple of (fixed_code, success, metrics)
        """
        self.logger.info(progress(f"Attempting to fix JavaScript syntax error: {error_message[:100]}..."))
        
        for fix_attempt in range(1, max_attempts + 1):
            try:
                self.logger.info(f"Fix attempt {fix_attempt}/{max_attempts}")
                
                # Prepare input and generate fix
                input_data = self._prepare_input_data(js_code, error_message, fix_attempt)
                
                # Generate with retry mechanism
                result, metrics, raw_response = self.generate_with_retry(
                    input_data, 
                    log_path=os.path.join(output_folder, 'fix_attempts.log') if output_folder else None
                )
                
                # Extract fixed code from result
                if hasattr(result, 'js_code_export'):
                    fixed_code = result.js_code_export
                elif hasattr(result, 'fixed_code'):
                    fixed_code = result.fixed_code
                else:
                    fixed_code = str(result) if result else None
                
                if not fixed_code:
                    self.logger.warning(warning(f"Fix attempt {fix_attempt} produced no code"))
                    # Save failed fix attempt
                    if output_folder:
                        self._save_fix_attempt(js_code, output_folder, attempt_num, fix_attempt, 
                                              success=False, error="No code produced")
                    continue
                
                # Save all fix attempts (both successful and failed)
                if output_folder:
                    self._save_fix_attempt(fixed_code, output_folder, attempt_num, fix_attempt, 
                                         success=None, error=None)  # Will be determined by validation
                
                # Test if the fixed code is valid
                if self._validate_javascript_syntax(fixed_code):
                    self.logger.info(success(f"Successfully fixed JavaScript syntax on attempt {fix_attempt}"))
                    
                    # Also save as the main fixed file for compatibility
                    if output_folder:
                        self.save_output(fixed_code, output_folder, metrics)
                    
                    return fixed_code, True, metrics
                else:
                    self.logger.warning(warning(f"Fix attempt {fix_attempt} still has syntax issues"))
                    # Update error message for next attempt
                    error_message = "Previous fix still had syntax errors. Please check variable declarations and bracket matching more carefully."
                    js_code = fixed_code  # Try to fix the partially fixed code
                    
            except Exception as e:
                self.logger.error(error(f"Fix attempt {fix_attempt} failed: {e}"))
                # Save failed fix attempt with error
                if output_folder:
                    self._save_fix_attempt(js_code, output_folder, attempt_num, fix_attempt, 
                                          success=False, error=str(e))
                
        self.logger.warning(error(f"Failed to fix JavaScript after {max_attempts} attempts"))
        return js_code, False, {}
    
    def _validate_javascript_syntax(self, js_code: str) -> bool:
        """
        Enhanced validation of JavaScript syntax.
        
        Args:
            js_code: JavaScript code to validate
            
        Returns:
            True if syntax appears valid, False otherwise
        """
        try:
            # Check for markdown artifacts first
            if '```' in js_code:
                self.logger.debug("Found markdown code block markers")
                return False
            
            # Check for HTML/XML artifacts  
            if '<' in js_code and '>' in js_code:
                # Allow template literals but reject HTML tags
                if re.search(r'<[a-zA-Z][^>]*>', js_code):
                    self.logger.debug("Found HTML-like tags")
                    return False
            
            # Try actual syntax validation using Node.js
            with tempfile.NamedTemporaryFile(mode='w', suffix='.mjs', delete=False) as f:
                temp_file = f.name
                f.write(js_code)
            
            try:
                # Use Node.js to check syntax (ES modules)
                result = subprocess.run(
                    ['node', '--check', temp_file],
                    capture_output=True,
                    text=True,
                    timeout=5
                )
                
                if result.returncode != 0:
                    # Extract error details for logging
                    error_lines = result.stderr.strip().split('\n')
                    if error_lines:
                        error_msg = error_lines[0] if error_lines[0] else "Unknown syntax error"
                        self.logger.debug(f"Node.js syntax check failed: {error_msg}")
                    return False
                    
                # If Node.js check passes, continue with other validations
                
            except subprocess.TimeoutExpired:
                self.logger.debug("Node.js syntax check timed out")
                # Fall back to basic checks
            except FileNotFoundError:
                self.logger.debug("Node.js not available for syntax checking")
                # Fall back to basic checks
            finally:
                # Clean up temp file
                try:
                    os.unlink(temp_file)
                except:
                    pass
            
            # Check that code starts and ends appropriately
            lines = [line.strip() for line in js_code.split('\n') if line.strip()]
            if not lines:
                self.logger.debug("No non-empty lines found")
                return False
            
            # First line should be import or function declaration or variable declaration
            first_line = lines[0]
            if not (first_line.startswith('import') or 
                   first_line.startswith('export') or
                   first_line.startswith('function') or
                   first_line.startswith('const') or
                   first_line.startswith('let') or
                   first_line.startswith('var') or
                   first_line.startswith('//')):
                self.logger.debug(f"Unexpected first line: {first_line}")
                return False
            
            # Basic checks for common syntax issues
            
            # Check for balanced braces
            open_braces = js_code.count('{')
            close_braces = js_code.count('}')
            if open_braces != close_braces:
                self.logger.debug(f"Unbalanced braces: {open_braces} open, {close_braces} close")
                return False
            
            # Check for balanced parentheses
            open_parens = js_code.count('(')
            close_parens = js_code.count(')')
            if open_parens != close_parens:
                self.logger.debug(f"Unbalanced parentheses: {open_parens} open, {close_parens} close")
                return False
            
            # Check for balanced brackets
            open_brackets = js_code.count('[')
            close_brackets = js_code.count(']')
            if open_brackets != close_brackets:
                self.logger.debug(f"Unbalanced brackets: {open_brackets} open, {close_brackets} close")
                return False
            
            # Check for double braces (common LLM error)
            if '{{' in js_code or '}}' in js_code:
                self.logger.debug("Found double braces {{ or }}")
                return False
            
            # Check for common Three.js patterns
            if 'THREE' in js_code:
                # Should have import statement for THREE
                if not re.search(r'import.*THREE', js_code):
                    self.logger.debug("Uses THREE but missing import statement")
                    return False
            
            # Check for function definition (should be export function or function)
            if not (re.search(r'export\s+function', js_code) or 
                   re.search(r'function\s+\w+', js_code) or
                   re.search(r'export\s+default\s+function', js_code)):
                self.logger.debug("No proper function definition found")
                return False
            
            return True
            
        except Exception as e:
            self.logger.warning(f"Validation check failed: {e}")
            return True  # Assume valid if we can't check
    
    def test_fix_with_export(self, js_code: str, error_message: str,
                           output_folder: str) -> Tuple[bool, str]:
        """
        Test if fixed code can be successfully exported.
        
        Args:
            js_code: JavaScript code to fix and test
            error_message: Original error message
            output_folder: Output folder for saving results
            
        Returns:
            Tuple of (success, fixed_code_or_error_message)
        """
        # Fix the code
        fixed_code, fix_success, metrics = self.fix_javascript_code(
            js_code, error_message, 
            max_attempts=2, 
            output_folder=output_folder
        )
        
        if not fix_success:
            return False, "Failed to fix JavaScript syntax"
        
        # Test export with fixed code
        from utils.mesh_exporter import MeshExporter
        
        temp_dir = None
        try:
            # Create temporary directory for testing
            temp_dir = tempfile.mkdtemp(prefix='fix_test_')
            
            # Save fixed code to temp file
            export_temp = os.path.join(temp_dir, '_export_temp')
            os.makedirs(export_temp, exist_ok=True)
            
            export_path = os.path.join(export_temp, 'export.js')
            with open(export_path, 'w', encoding='utf-8') as f:
                f.write(fixed_code)
            
            # Test export
            exporter = MeshExporter(temp_dir)
            final_mesh, export_success, error_msg = exporter.export_to_obj()
            
            if export_success:
                self.logger.info(success("Fixed code exports successfully!"))
                # Save successful fix to output folder
                export_temp_out = os.path.join(output_folder, '_export_temp')
                os.makedirs(export_temp_out, exist_ok=True)
                
                fixed_export_path = os.path.join(export_temp_out, 'export.js')
                with open(fixed_export_path, 'w', encoding='utf-8') as f:
                    f.write(fixed_code)
                
                return True, fixed_code
            else:
                return False, "Fixed code still fails to export"
                
        except Exception as e:
            self.logger.error(f"Export test failed: {e}")
            return False, str(e)
            
        finally:
            # Clean up temporary directory
            if temp_dir and os.path.exists(temp_dir):
                shutil.rmtree(temp_dir, ignore_errors=True)