"""
Shape Fixer Agent for improving Three.js code based on VLM feedback

This agent takes VLM critic feedback and modifies Three.js code to address
the identified issues and implement suggested improvements.
"""

import os
import re
import json
import logging
from typing import Dict, Any, Optional, Tuple
from pydantic import BaseModel
from agents.base_agent import BaseAgent
from utils.output_parser import OutputFormatError


class ImprovedShapeCode(BaseModel):
    """Data model for improved Three.js code output"""
    js_code_export: str = None

    @classmethod
    def extract_from_response(cls, text: str) -> "ImprovedShapeCode":
        """
        Extract js_code_export from LLM response

        Args:
            text: Raw LLM response

        Returns:
            ImprovedShapeCode instance with extracted export code

        Raises:
            OutputFormatError: If js_code_export block is missing
        """
        # Primary pattern for js_code_export (same as ThreeJSCode)
        export_match = re.search(r'<js_code_export>(.*?)</js_code_export>', text, re.DOTALL)

        if not export_match:
            # Log the response for debugging
            import logging
            logging.error(f"Failed to find <js_code_export> tags in response. Response preview: {text[:500]}...")
            raise OutputFormatError("Missing required code block: js_code_export")

        js_code_export = export_match.group(1).strip()

        if not cls._validate_code_block(js_code_export, "javascript"):
            raise OutputFormatError("Export JavaScript code block is invalid or empty")

        return cls(js_code_export=js_code_export)

    @staticmethod
    def _validate_code_block(code: str, code_type: str) -> bool:
        """Validate that JavaScript code block contains meaningful content (same as ThreeJSCode)"""
        if not code or not code.strip():
            return False

        # Basic validation for JavaScript
        if code_type == "javascript":
            return len(code.strip()) > 10  # Basic length check

        return True


class ShapeFixerAgent(BaseAgent):
    """
    Agent for improving Three.js code based on VLM feedback.

    Takes structured feedback from VLM Critic and modifies the Three.js
    generation code to address identified issues and implement improvements.
    """

    def __init__(self, config_manager):
        """
        Initialize the Shape Fixer Agent.

        Args:
            config_manager: Configuration manager instance
        """
        self.logger = logging.getLogger(self.__class__.__name__)

        # Initialize with shape_fixer agent type
        super().__init__(config_manager, 'shape_fixer')

    def _load_system_prompt(self) -> str:
        """Load system prompt for shape improvement."""
        try:
            from prompt.shape_fixer import system_prompt
            return system_prompt
        except ImportError as e:
            self.logger.error(f"Failed to load Shape Fixer prompt: {e}")
            # Fallback prompt
            return """You are an expert Three.js developer specialized in procedural 3D object generation.
            Your task is to improve Three.js code based on detailed feedback from a design critic.

            When modifying code, focus on:
            - Addressing specific proportion and scale issues
            - Improving structural integrity and realism
            - Adding missing components or details
            - Fixing geometric inconsistencies
            - Ensuring proper material assignments and object hierarchy

            Always preserve the core functionality while making targeted improvements.
            Output the complete improved code wrapped in <js_code_export> tags."""

    def _format_user_prompt(self, input_data: Dict[str, Any]) -> str:
        """
        Format user prompt with original code and VLM feedback.

        Args:
            input_data: Dictionary containing original_code, vlm_feedback, etc.

        Returns:
            Formatted user prompt
        """
        original_code = input_data.get('original_code', '')
        vlm_feedback = input_data.get('vlm_feedback', {})
        description = input_data.get('description', '')
        iteration_num = input_data.get('iteration_num', 1)
        object_json = input_data.get('object_json', {})

        # Format VLM feedback for prompt
        feedback_text = ""
        if isinstance(vlm_feedback, dict):
            # Extract color mapping if present - this is DYNAMIC per object
            color_info = ""
            if 'color_mapping' in vlm_feedback:
                color_info = "\n\n**COLOR-TO-PART MAPPING (THIS IS SPECIFIC TO THIS OBJECT):**\n"
                color_info += "Use this mapping to understand which colored components in the feedback correspond to which parts:\n"
                for part_name, color in vlm_feedback.get('color_mapping', {}).items():
                    color_info += f"- When feedback mentions '{color.upper()}' part → this refers to '{part_name}'\n"

            feedback_text = f"""
VLM Analysis Results:
{color_info}
Overall Assessment: {vlm_feedback.get('overall_assessment', 'No assessment provided')}

Specific Issues (PAY ATTENTION TO POSITIONING ERRORS):
{chr(10).join(['- ' + issue for issue in vlm_feedback.get('specific_issues', [])])}

Improvement Suggestions (IMPLEMENT THESE EXACT FIXES):
{chr(10).join(['- ' + suggestion for suggestion in vlm_feedback.get('improvement_suggestions', [])])}

Needs Improvement: {vlm_feedback.get('needs_improvement', True)}
Confidence Score: {vlm_feedback.get('confidence_score', 0.0)}
"""
        else:
            feedback_text = f"VLM Feedback:\n{vlm_feedback}"

        # Format object JSON if available
        json_info = ""
        if object_json:
            json_info = f"\n\nOriginal Object Specification:\n{json.dumps(object_json, indent=2, ensure_ascii=False)}"

        prompt = f"""Please improve the following Three.js code based on the VLM critic feedback.

Original Description: {description}
Iteration: {iteration_num}{json_info}

{feedback_text}

Current Three.js Code:
```javascript
{original_code}
```

**MANDATORY: YOU MUST MAKE ACTUAL CHANGES TO THE CODE**

This is iteration {iteration_num} - The VLM critic has identified SPECIFIC issues that MUST be fixed.
If you return the same code without meaningful changes, the system will fail.

**CRITICAL ENFORCEMENT RULES:**
1. **YOU MUST MODIFY THE CODE** - Simply returning the same code is FAILURE
2. **EVERY issue in "Specific Issues" MUST result in a code change**
3. **EVERY suggestion in "Improvement Suggestions" MUST be implemented**
4. **Make MEASURABLE changes** - Position values, scales, or structures MUST change

**REQUIRED ACTIONS FOR EACH FEEDBACK ITEM:**
- For each issue mentioned, identify the EXACT line and value to change
- Make SIGNIFICANT adjustments (not tiny 0.01 changes, make real 0.1+ changes)
- If feedback mentions misalignment: Change position by at least 10-20% of object size
- If feedback mentions wrong size: Change scale by at least 20-30%
- If feedback mentions missing parts: ADD them explicitly with new geometry

**PRESERVE ALL COMPONENTS FROM ORIGINAL SPECIFICATION:**
The Original Object Specification lists ALL required components. Every component must:
1. Still exist in your improved code
2. Be visible and properly positioned
3. Not be embedded/hidden inside other components

**IMPLEMENTATION CHECKLIST:**
Before submitting, verify you have:
□ Made AT LEAST 3 meaningful position/scale/rotation changes
□ Addressed EVERY issue from the feedback
□ Implemented EVERY suggestion provided
□ Changed actual numeric values significantly
□ NOT just returned the same code with minor tweaks

Guidelines for improvements (IN STRICT ORDER OF PRIORITY):
1. **ADDRESS FEEDBACK DIRECTLY**: Read each issue and suggestion, find the corresponding code, CHANGE IT
2. **VERIFY ALL COMPONENTS EXIST**: Cross-check with Original Object Specification
3. **FIX POSITIONING ERRORS**: Make BOLD adjustments - small tweaks won't fix the problems
4. **MAINTAIN CODE STRUCTURE**: Keep createScene format and Three.js syntax

Remember: The VLM critic will check if you actually made the requested changes. Returning similar code = FAILURE.

You must provide EXACTLY ONE code block using the <js_code_export>...</js_code_export> tags with no additional formatting or explanations outside the tags."""

        return prompt

    def parse_response(self, response: str) -> ImprovedShapeCode:
        """
        Parse LLM response into improved code structure.

        Args:
            response: Raw LLM response

        Returns:
            ImprovedShapeCode instance with parsed code and improvements

        Raises:
            OutputFormatError: If response format is invalid
        """
        try:
            return ImprovedShapeCode.extract_from_response(response)
        except OutputFormatError as e:
            self.logger.error(f"Failed to parse shape fixer response: {e}")
            raise

    def _prepare_input_data(self, original_code: str, vlm_feedback: Dict[str, Any],
                           description: str = "", object_json: Optional[Dict] = None,
                           iteration_num: int = 1, **kwargs) -> Dict[str, Any]:
        """
        Prepare input data for shape improvement.

        Args:
            original_code: Original Three.js code to improve
            vlm_feedback: Structured feedback from VLM Critic
            description: Original object description
            object_json: Optional detailed object specification
            iteration_num: Current iteration number
            **kwargs: Additional arguments

        Returns:
            Dictionary of input data
        """
        return {
            'original_code': original_code,
            'vlm_feedback': vlm_feedback,
            'description': description,
            'object_json': object_json or {},
            'iteration_num': iteration_num
        }

    def save_output(self, result: ImprovedShapeCode, output_folder: str,
                   iteration_num: int = 1, metrics: Dict[str, Any] = None):
        """
        Save improved code and export mesh.

        Args:
            result: Improved code result
            output_folder: Directory to save output
            iteration_num: Current iteration number
            metrics: Generation metrics (optional)
        """
        if not result or not result.js_code_export:
            return

        # Create iteration folder
        iteration_folder = os.path.join(output_folder, f"iteration_{iteration_num}")
        os.makedirs(iteration_folder, exist_ok=True)

        # Save improved code
        improved_code_path = os.path.join(iteration_folder, "improved_code.js")
        with open(improved_code_path, 'w', encoding='utf-8') as f:
            f.write(result.js_code_export)

        self.logger.info(f"Saved improved code to {improved_code_path}")

        # Export mesh for visualization comparison
        self._export_improved_mesh(result.js_code_export, iteration_folder)

    def _export_improved_mesh(self, js_code: str, iteration_folder: str):
        """
        Export the improved Three.js code to mesh format for visualization.

        Args:
            js_code: Improved Three.js code
            iteration_folder: Folder to save the mesh
        """
        try:
            from utils.mesh_exporter import MeshExporter
            import tempfile
            import shutil

            # Create temporary directory for export
            with tempfile.TemporaryDirectory() as temp_dir:
                # Create the required folder structure
                export_temp = os.path.join(temp_dir, '_export_temp')
                os.makedirs(export_temp, exist_ok=True)

                # Save the improved code
                export_js_path = os.path.join(export_temp, 'export.js')
                with open(export_js_path, 'w', encoding='utf-8') as f:
                    f.write(js_code)

                # Create package.json for ES modules
                package_json_path = os.path.join(temp_dir, 'package.json')
                with open(package_json_path, 'w') as f:
                    json.dump({"type": "module"}, f)

                # Export to mesh
                exporter = MeshExporter(temp_dir)
                mesh_result = exporter.test_export(js_code)

                if mesh_result.get('success'):
                    # Copy the exported mesh to iteration folder
                    temp_mesh_path = mesh_result.get('mesh_path')
                    if temp_mesh_path and os.path.exists(temp_mesh_path):
                        final_mesh_path = os.path.join(iteration_folder, 'improved_mesh.obj')
                        shutil.copy2(temp_mesh_path, final_mesh_path)
                        self.logger.info(f"Exported improved mesh to {final_mesh_path}")
                    else:
                        self.logger.warning("Mesh export succeeded but no mesh file found")
                else:
                    self.logger.warning(f"Failed to export improved mesh: {mesh_result.get('error', 'Unknown error')}")

        except Exception as e:
            self.logger.warning(f"Failed to export improved mesh: {e}")

    def improve_code(self, original_code: str, vlm_feedback: Dict[str, Any],
                    description: str = "", object_json: Optional[Dict] = None,
                    output_folder: Optional[str] = None,
                    iteration_num: int = 1,
                    **kwargs) -> Tuple[ImprovedShapeCode, bool, Dict[str, Any], str]:
        """
        Improve Three.js code based on VLM feedback.

        Args:
            original_code: Original Three.js code to improve
            vlm_feedback: Structured feedback from VLM Critic
            description: Original object description
            object_json: Optional detailed object specification
            output_folder: Optional output folder for saving results
            iteration_num: Current iteration number
            **kwargs: Additional arguments for generation

        Returns:
            Tuple of (improved_code, success, metrics, raw_response)
        """
        # Prepare input data
        input_data = self._prepare_input_data(
            original_code, vlm_feedback, description, object_json, iteration_num, **kwargs
        )

        # Generate improvement
        result, success, metrics, raw_response = self.generate(**input_data, **kwargs)

        # Save output if successful and output folder provided
        if success and result and output_folder:
            self.save_output(result, output_folder, iteration_num, metrics)

        return result, success, metrics, raw_response

    def fix_code_with_validation(self, original_code: str, vlm_feedback: Dict[str, Any],
                                description: str = "", object_json: Optional[Dict] = None,
                                output_folder: Optional[str] = None,
                                iteration_num: int = 1,
                                max_attempts: int = 2) -> Tuple[str, bool, Dict[str, Any]]:
        """
        Improve code with export validation and automatic syntax error fixing.

        Args:
            original_code: Original Three.js code to improve
            vlm_feedback: Structured feedback from VLM Critic
            description: Original object description
            object_json: Optional detailed object specification
            output_folder: Optional output folder for saving results
            iteration_num: Current iteration number
            max_attempts: Maximum improvement attempts

        Returns:
            Tuple of (improved_code, success, metrics)
        """
        for attempt in range(max_attempts):
            try:
                self.logger.info(f"Shape improvement attempt {attempt + 1}/{max_attempts}")

                # Improve the code
                result, success, metrics, raw_response = self.improve_code(
                    original_code, vlm_feedback, description, object_json,
                    output_folder, iteration_num
                )

                if not success or not result or not result.js_code_export:
                    self.logger.warning(f"Improvement attempt {attempt + 1} failed - no code generated")
                    continue

                # Validate the improved code can be exported
                from utils.mesh_exporter import MeshExporter
                import tempfile

                with tempfile.TemporaryDirectory() as temp_dir:
                    export_temp = os.path.join(temp_dir, '_export_temp')
                    os.makedirs(export_temp, exist_ok=True)

                    export_js_path = os.path.join(export_temp, 'export.js')
                    with open(export_js_path, 'w', encoding='utf-8') as f:
                        f.write(result.js_code_export)

                    # Create package.json
                    package_json_path = os.path.join(temp_dir, 'package.json')
                    with open(package_json_path, 'w') as f:
                        json.dump({"type": "module"}, f)

                    # Test export
                    exporter = MeshExporter(temp_dir)
                    test_result = exporter.test_export(result.js_code_export)

                    if test_result.get('success'):
                        self.logger.info(f"Improved code validated successfully on attempt {attempt + 1}")
                        return result.js_code_export, True, metrics
                    else:
                        error_msg = test_result.get('error', 'Unknown export error')
                        self.logger.warning(f"Improved code failed validation: {error_msg}")

                        # Try to fix syntax errors using ShapeCodeFixAgent
                        if "is not defined" in error_msg or "ReferenceError" in error_msg or "SyntaxError" in error_msg:
                            self.logger.info("Attempting to fix syntax error with ShapeCodeFixAgent...")
                            fixed_code = self._try_fix_syntax_error(result.js_code_export, error_msg, output_folder, iteration_num)

                            if fixed_code:
                                # Test if fixed code exports successfully
                                with tempfile.TemporaryDirectory() as fix_temp_dir:
                                    fix_export_temp = os.path.join(fix_temp_dir, '_export_temp')
                                    os.makedirs(fix_export_temp, exist_ok=True)

                                    fix_export_js_path = os.path.join(fix_export_temp, 'export.js')
                                    with open(fix_export_js_path, 'w', encoding='utf-8') as f:
                                        f.write(fixed_code)

                                    # Create package.json
                                    fix_package_json_path = os.path.join(fix_temp_dir, 'package.json')
                                    with open(fix_package_json_path, 'w') as f:
                                        json.dump({"type": "module"}, f)

                                    # Test fixed code
                                    fix_exporter = MeshExporter(fix_temp_dir)
                                    fix_test_result = fix_exporter.test_export(fixed_code)

                                    if fix_test_result.get('success'):
                                        self.logger.info("✓ Fixed code exports successfully!")
                                        # Save the fixed version
                                        if output_folder:
                                            fixed_code_path = os.path.join(output_folder, f"iteration_{iteration_num}", "improved_code.js")
                                            with open(fixed_code_path, 'w', encoding='utf-8') as f:
                                                f.write(fixed_code)
                                        return fixed_code, True, metrics

                        # If not the last attempt and couldn't fix, try to improve again
                        if attempt < max_attempts - 1:
                            # Create additional feedback about the export error
                            export_feedback = {
                                'overall_assessment': f"Code improvement failed validation: {error_msg}",
                                'specific_issues': [f"Export validation error: {error_msg}"],
                                'improvement_suggestions': [
                                    "Fix syntax errors and ensure proper Three.js structure",
                                    "Verify all variables are properly declared before use",
                                    "Check that the createScene function returns a valid Three.js object"
                                ],
                                'needs_improvement': True,
                                'confidence_score': 0.8
                            }

                            # Use the improved code as the new original for next attempt
                            original_code = result.js_code_export
                            vlm_feedback = export_feedback

            except Exception as e:
                self.logger.error(f"Shape improvement attempt {attempt + 1} failed: {e}")

        self.logger.error(f"All {max_attempts} improvement attempts failed")
        return original_code, False, {}

    def _try_fix_syntax_error(self, js_code: str, error_msg: str,
                             output_folder: Optional[str], iteration_num: int) -> Optional[str]:
        """
        Try to fix syntax errors in the improved code using ShapeCodeFixAgent.

        Args:
            js_code: JavaScript code with potential syntax errors
            error_msg: Error message from export attempt
            output_folder: Optional output folder for saving fixed code
            iteration_num: Current iteration number

        Returns:
            Fixed JavaScript code if successful, None otherwise
        """
        try:
            from agents.shape_code_fix_agent import ShapeCodeFixAgent

            # Initialize fix agent with same config
            fix_agent = ShapeCodeFixAgent(self.config)

            # Create output folder for fix attempts if needed
            fix_output_folder = None
            if output_folder:
                fix_output_folder = os.path.join(output_folder, f"iteration_{iteration_num}", "fix_attempts")
                os.makedirs(fix_output_folder, exist_ok=True)

            # Try to fix the code
            fixed_code, fix_success, fix_metrics = fix_agent.fix_javascript_code(
                js_code=js_code,
                error_message=error_msg,
                max_attempts=2,
                output_folder=fix_output_folder,
                attempt_num=iteration_num
            )

            if fix_success:
                self.logger.info("✓ Successfully fixed syntax error")
                return fixed_code
            else:
                self.logger.warning("Failed to fix syntax error")
                return None

        except Exception as e:
            self.logger.warning(f"Error during syntax fix attempt: {e}")
            return None