"""
Anthropic Claude API handler supporting claude-opus, claude-sonnet, and claude-haiku models
"""

import time
import logging
from typing import Dict, Any, Tuple
from anthropic import Anthropic
from .base_provider import BaseProvider
from .utils import TokenCounter, CostCalculator, RequestLogger


class AnthropicHandler(BaseProvider):
    """Handler for Anthropic Claude models using messages endpoint"""
    
    def __init__(self, config: Dict[str, Any]):
        """Initialize Anthropic handler with configuration"""
        super().__init__(config)
        
        # Initialize Anthropic client
        self.client = Anthropic(api_key=self.api_key)
        
        # Initialize utilities
        self.token_counter = TokenCounter()
        self.request_logger = RequestLogger('Anthropic')
    
    def invoke(self, prompt: str, system_prompt: str = None, **kwargs) -> Tuple[str, Dict]:
        """
        Invoke Claude model using messages endpoint
        
        Args:
            prompt: User prompt
            system_prompt: Optional system prompt
            **kwargs: Additional parameters
            
        Returns:
            Tuple of (response_text, metadata)
        """
        start_time = time.time()
        
        # Build messages
        messages = [{"role": "user", "content": prompt}]
        
        # Prepare parameters
        params = {
            'model': self.model_name,
            'messages': messages,
            'max_tokens': kwargs.get('max_tokens', self.max_tokens),
            'temperature': kwargs.get('temperature', self.temperature),
        }
        
        # Add system prompt if provided
        if system_prompt:
            params['system'] = system_prompt
        
        # Add optional parameters if provided
        if 'top_p' in kwargs:
            params['top_p'] = kwargs['top_p']
        
        # Make API call
        try:
            response = self.client.messages.create(**params)
            
            # Extract response text
            response_text = response.content[0].text if response.content else ""
            
            # Get token counts
            input_tokens = response.usage.input_tokens if response.usage else self.token_counter.count_tokens_openai(prompt, 'claude')
            output_tokens = response.usage.output_tokens if response.usage else self.token_counter.count_tokens_openai(response_text, 'claude')
            
            # Calculate cost
            cost = CostCalculator.calculate_anthropic_cost(self.model_name, input_tokens, output_tokens)
            
            metadata = {
                'input_tokens': input_tokens,
                'output_tokens': output_tokens,
                'total_tokens': input_tokens + output_tokens,
                'cost': cost,
                'model': self.model_name,
                'endpoint': 'messages',
                'time_taken': time.time() - start_time
            }
            
            # Log the request
            self.request_logger.log_request(prompt, response_text, metadata)
            
            return response_text, metadata
            
        except Exception as e:
            self.logger.error(f"Anthropic API call failed: {e}")
            raise
    
    def count_tokens(self, text: str) -> int:
        """Count tokens for Claude models"""
        return self.token_counter.count_tokens_openai(text, 'claude')