"""
Common utilities for the articulated object generation pipeline.

This module provides frequently used utility functions that are shared
across multiple components of the pipeline, reducing code duplication
and ensuring consistent behavior.

Key Components:
- File operations: unique folder creation, path management
- Logging setup: unified logging configuration for different contexts
- JavaScript processing: handling export.js files and Three.js integration
- String utilities: text processing and formatting

Benefits:
- Centralized utility functions
- Consistent behavior across the pipeline
- Reduced code duplication
- Easy maintenance and updates
"""

import os
import re
import glob
import json
import logging
import subprocess
from typing import Optional


# =============================================================================
# File Operations
# =============================================================================

def get_unique_output_folder(base_name: str, output_root: str) -> str:
    """
    Create a unique output folder with an incremented suffix.
    
    Args:
        base_name: Base name for the folder
        output_root: Root directory to create the folder in
        
    Returns:
        Path to the created unique folder
    """
    os.makedirs(output_root, exist_ok=True)
    pattern = os.path.join(output_root, f"{base_name}_*")
    existing = [os.path.basename(p) for p in glob.glob(pattern) if os.path.isdir(p)]
    nums = [int(m.group(1)) for name in existing if (m := re.match(rf"{base_name}_(\d+)", name))]
    next_num = max(nums) + 1 if nums else 1
    folder = os.path.join(output_root, f"{base_name}_{next_num}")
    os.makedirs(folder, exist_ok=True)
    return folder


def create_package_json(directory: str, module_type: str = "module") -> str:
    """
    Create a package.json file to enable ES module support.
    
    Args:
        directory: Directory to create the package.json in
        module_type: Module type ("module" or "commonjs")
        
    Returns:
        Path to the created package.json file
    """
    package_json_path = os.path.join(directory, "package.json")
    with open(package_json_path, 'w', encoding='utf-8') as f:
        json.dump({"type": module_type}, f)
    return package_json_path


# =============================================================================
# Logging Configuration
# =============================================================================

def setup_global_logging(log_level: str = "INFO") -> None:
    """
    Set up global logging configuration for the pipeline.
    This disables debug/info logs from noisy libraries and sets the root log level.
    
    Args:
        log_level: Logging level (DEBUG, INFO, WARNING, ERROR, CRITICAL)
    """
    logging.basicConfig(
        level=getattr(logging, log_level),
        format='%(asctime)s - %(levelname)s - %(message)s'
    )
    # Disable noisy library logs
    for noisy_logger in ["httpx", "openai", "requests", "urllib3", "google"]:
        logging.getLogger(noisy_logger).setLevel(logging.WARNING)


def setup_file_logger(log_path: str, log_level: str = "INFO") -> None:
    """
    Set up a file logger for a specific file.
    
    Args:
        log_path: Path to the log file
        log_level: Logging level
    """
    logger = logging.getLogger()
    # Remove existing object-specific file handlers
    for handler in list(logger.handlers):
        if isinstance(handler, logging.FileHandler) and getattr(handler, '_pipeline_object_log', False):
            logger.removeHandler(handler)
            handler.close()
    
    # Add new file handler
    file_handler = logging.FileHandler(log_path, mode='a', encoding='utf-8')
    file_handler.setLevel(getattr(logging, log_level))
    file_handler.setFormatter(logging.Formatter('%(asctime)s - %(levelname)s - %(message)s'))
    file_handler._pipeline_object_log = True
    logger.addHandler(file_handler)


def setup_object_logger(output_folder: str, log_level: str = "INFO") -> None:
    """
    Set up a file logger for the given object folder, writing logs to log.txt.
    
    Args:
        output_folder: Directory to create log file in
        log_level: Logging level
    """
    log_path = os.path.join(output_folder, "log.txt")
    setup_file_logger(log_path, log_level)


def remove_object_logger() -> None:
    """
    Remove the file logger handler associated with object logging from the root logger.
    This prevents duplicate log entries and file handle leaks.
    """
    logger = logging.getLogger()
    for handler in list(logger.handlers):
        if isinstance(handler, logging.FileHandler) and getattr(handler, '_pipeline_object_log', False):
            logger.removeHandler(handler)
            handler.close()


# =============================================================================
# JavaScript Processing
# =============================================================================

def fix_duplicate_sceneobject(export_js_path: str) -> None:
    """
    Fix duplicate sceneObject declarations in export.js by renaming subsequent variables.
    This prevents variable name collisions in the generated JavaScript code.
    
    Args:
        export_js_path: Path to the export.js file to fix
    """
    with open(export_js_path, 'r', encoding='utf-8') as f:
        lines = f.readlines()
    
    seen = 0
    new_lines = []
    for line in lines:
        match = re.match(r'\s*(let|const|var)\s+sceneObject\b', line)
        if match:
            seen += 1
            name = 'sceneObject' if seen == 1 else f'sceneObject{seen}'
            line = re.sub(r'\bsceneObject\b', name, line)
        elif seen > 1:
            for i in range(2, seen + 1):
                line = re.sub(rf'\bsceneObject{i-1}\b', f'sceneObject{i}', line)
        new_lines.append(line)
    
    with open(export_js_path, 'w', encoding='utf-8') as f:
        f.writelines(new_lines)


def fix_three_import_path(export_js_path: str, project_root: str) -> None:
    """
    Fix the import statement in export.js to use absolute path to the three module,
    mirroring the logic in MeshExporter.
    """
    three_module_path = os.path.join(project_root, "node_modules", "three")

    with open(export_js_path, 'r', encoding='utf-8') as f:
        content = f.read()

    # Replace common forms of THREE import from the bare specifier
    content_fixed = re.sub(
        r"import\s+\*\s+as\s+THREE\s+from\s+['\"]three['\"];",
        f"import * as THREE from '{three_module_path}/build/three.module.js';",
        content
    )

    if content_fixed != content:
        with open(export_js_path, 'w', encoding='utf-8') as f:
            f.write(content_fixed)


def extract_and_merge_threejs_info(object_folder: str) -> None:
    """
    Run a Node.js script to extract and merge Three.js information from export.js into workflow.json.
    This is useful for downstream processing and articulation generation.
    
    Args:
        object_folder: Folder containing the export.js file
    """
    # Assume export.js is in _export_temp (standard pipeline flow)
    export_js = os.path.join(object_folder, "_export_temp", "export.js")
    working_dir = os.path.join(object_folder, "_export_temp")
    
    # Create package.json in the working directory to enable ES module support
    create_package_json(working_dir)
    
    # Get project root and node_modules path
    script_dir = os.path.dirname(os.path.abspath(__file__))
    project_root = os.path.dirname(script_dir)
    project_node_modules = os.path.join(project_root, "node_modules")

    # Ensure export.js imports THREE from an absolute path (same as MeshExporter)
    fix_three_import_path(export_js, project_root)
    
    # Run the Node.js script to extract Three.js information
    extract_script = os.path.join(script_dir, "extract_threejs_info.js")
    
    # Check if the script exists
    if not os.path.exists(extract_script):
        raise FileNotFoundError(f"extract_threejs_info.js not found at {extract_script}")
    
    # Check if export.js exists
    if not os.path.exists(export_js):
        raise FileNotFoundError(f"export.js not found at {export_js}")
    
    try:
        # Set up environment with NODE_PATH
        env = os.environ.copy()
        env['NODE_PATH'] = project_node_modules
        
        # Run the extraction script
        result = run_node_script(
            extract_script,
            [export_js],
            cwd=working_dir,
            env=env
        )
        
        logging.info(f"Successfully extracted and merged Three.js info for {object_folder}")
        if result.stdout:
            logging.debug(f"Extraction output: {result.stdout}")
            
    except subprocess.CalledProcessError as e:
        logging.error(f"Failed to extract Three.js info: {e}")
        if e.stderr:
            logging.error(f"Error output: {e.stderr}")
        raise
    except Exception as e:
        logging.error(f"Unexpected error during Three.js info extraction: {e}")
        raise


def run_node_script(script_path: str, args: list, cwd: Optional[str] = None, 
                   timeout: int = 300, env: Optional[dict] = None) -> subprocess.CompletedProcess:
    """
    Run a Node.js script with the given arguments.
    
    Args:
        script_path: Path to the Node.js script
        args: List of arguments to pass to the script
        cwd: Working directory to run the script in
        timeout: Timeout in seconds
        env: Environment variables to use (if None, uses current environment)
        
    Returns:
        CompletedProcess object with the result
        
    Raises:
        subprocess.CalledProcessError: If the script fails
        subprocess.TimeoutExpired: If the script times out
    """
    cmd = ["node", script_path] + args
    return subprocess.run(
        cmd, 
        capture_output=True, 
        text=True, 
        check=True, 
        cwd=cwd, 
        timeout=timeout,
        env=env
    )


# =============================================================================
# String Utilities
# =============================================================================

def clean_filename(filename: str) -> str:
    """
    Clean a filename by removing invalid characters and limiting length.
    
    Args:
        filename: Original filename
        
    Returns:
        Cleaned filename safe for filesystem use
    """
    # Remove invalid characters
    filename = re.sub(r'[<>:"/\\|?*]', '_', filename)
    # Replace multiple underscores with single
    filename = re.sub(r'_+', '_', filename)
    # Limit length
    if len(filename) > 100:
        name, ext = os.path.splitext(filename)
        filename = name[:96] + ext
    return filename.strip('_')


def format_file_size(size_bytes: int) -> str:
    """
    Format a file size in bytes into a human-readable string.
    
    Args:
        size_bytes: Size in bytes
        
    Returns:
        Formatted size string (e.g., "1.5 MB")
    """
    if size_bytes == 0:
        return "0 B"
    
    size_names = ["B", "KB", "MB", "GB", "TB"]
    i = 0
    while size_bytes >= 1024 and i < len(size_names) - 1:
        size_bytes /= 1024.0
        i += 1
    
    return f"{size_bytes:.1f} {size_names[i]}"