import os
import re
import logging
from typing import Optional, Tuple, List, Dict
from .combine_meshes import combine_meshes
from .common import fix_duplicate_sceneobject, create_package_json, run_node_script

class MeshExporter:
    """
    Handles mesh export and combination for the pipeline.
    Responsible for running the export process and combining mesh parts into a single OBJ file.
    """
    def __init__(self, output_folder: str, final_output_dir: Optional[str] = None):
        self.output_folder = output_folder
        self.final_output_dir = final_output_dir or output_folder

    def export_to_obj(self) -> Tuple[str, bool, Optional[str]]:
        """
        Export mesh parts to OBJ and combine them into a single mesh.
        Returns the path to the combined mesh, a success flag, and error message if failed.
        """
        try:
            parts_folder = self._run_export_process()
            combined_mesh = self._combine_meshes(parts_folder)
            return combined_mesh, True, None
        except Exception as e:
            logging.error(f"Export failed: {str(e)}")
            return "", False, str(e)

    def _run_export_process(self) -> str:
        """
        Run the Node.js export process to generate mesh parts from export.js.
        Returns the folder containing the exported mesh parts.
        """
        # Assume export.js is in _export_temp (standard pipeline flow)
        input_js_path = os.path.join(self.output_folder, "_export_temp", "export.js")
        working_dir = os.path.join(self.output_folder, "_export_temp")
            
        fix_duplicate_sceneobject(input_js_path)
        
        # Create package.json in the working directory to enable ES module support
        create_package_json(working_dir)
        
        # Get project root and node_modules path
        script_dir = os.path.dirname(os.path.abspath(__file__))
        project_root = os.path.dirname(script_dir)
        
        # Fix import path in export.js to use absolute path to three module
        self._fix_three_import_path(input_js_path, project_root)
        
        # Apply JavaScript syntax fixes to handle common LLM generation errors
        # self._fix_javascript_syntax(input_js_path)  # Commented out - may cause issues
        
        parts_output_folder = os.path.join(self.output_folder, "part_meshes")
        os.makedirs(parts_output_folder, exist_ok=True)
        logging.info("Running export process...")
        try:
            result = run_node_script(
                "utils/threejs_to_mesh.js",
                [os.path.abspath(input_js_path), os.path.abspath(parts_output_folder)],
                cwd=project_root
            )
            logging.debug(f"Export process output: {result.stdout}")
            return parts_output_folder
        except Exception as e:
            # Extract detailed error information
            error_msg = str(e)
            
            # Try to extract specific JavaScript error from stderr
            if hasattr(e, 'stderr') and e.stderr:
                stderr_text = e.stderr.decode('utf-8') if isinstance(e.stderr, bytes) else str(e.stderr)
                # Look for common JavaScript error patterns
                error_patterns = [
                    r'(ReferenceError:.*)',
                    r'(SyntaxError:.*)',
                    r'(TypeError:.*)',
                    r'(Cannot access.*before initialization)',
                    r'(.*is not defined)',
                    r'(Unexpected token.*)',
                    r'(.*before declaration)'
                ]
                for pattern in error_patterns:
                    match = re.search(pattern, stderr_text, re.MULTILINE)
                    if match:
                        error_msg = match.group(1)
                        break
            
            logging.error(f"Export process failed with error: {error_msg}")
            raise RuntimeError(f"Export process failed: {error_msg}")

    def _combine_meshes(self, parts_folder: str) -> str:
        """
        Combine all mesh parts in the given folder into a single OBJ file.
        Returns the path to the combined mesh.
        """
        output_mesh = os.path.join(self.output_folder, "combined_assembly.obj")
        try:
            combine_meshes(parts_folder, output_mesh)
            return output_mesh
        except Exception as e:
            logging.error(f"Failed to combine meshes: {str(e)}")
            raise 

    # Belows    
    def _fix_three_import_path(self, export_js_path: str, project_root: str) -> None:
        """
        Fix the import statement in export.js to use absolute path to three module.
        """
        three_module_path = os.path.join(project_root, "node_modules", "three")
        
        # Read the file
        with open(export_js_path, 'r', encoding='utf-8') as f:
            content = f.read()
        
        # Replace the import statement with absolute path
        content = content.replace(
            "import * as THREE from 'three';",
            f"import * as THREE from '{three_module_path}/build/three.module.js';"
        )
        
        # Write back
        with open(export_js_path, 'w', encoding='utf-8') as f:
            f.write(content)
    
    def _fix_javascript_syntax(self, export_js_path: str) -> bool:
        """
        Fix common JavaScript syntax errors in generated Three.js export files.
        
        This method handles common LLM generation issues like:
        - Double brace template syntax ({{ }} instead of { })
        - Malformed string interpolation
        - Function declaration syntax errors
        - Object literal syntax issues
        
        Args:
            export_js_path: Path to the export.js file
            
        Returns:
            True if fixes were applied, False otherwise
        """
        # COMMENTED OUT - These fixes may cause issues with valid code
        return False
        
        # try:
        #     with open(export_js_path, 'r', encoding='utf-8') as f:
        #         original_content = f.read()
        #     
        #     # Apply all known syntax fixes
        #     fixed_content = self._apply_syntax_fixes(original_content)
        #     
        #     # Only write if changes were made
        #     if fixed_content != original_content:
        #         with open(export_js_path, 'w', encoding='utf-8') as f:
        #             f.write(fixed_content)
        #         
        #         logging.info(f"Applied JavaScript syntax fixes to {export_js_path}")
        #         return True
        #         
        # except Exception as e:
        #     logging.warning(f"Failed to apply JavaScript syntax fixes: {e}")
        #     
        # return False
    
    def _apply_syntax_fixes(self, content: str) -> str:
        """
        Apply comprehensive set of JavaScript syntax fixes.
        
        This method contains all the regex patterns and replacements needed
        to fix common LLM generation errors in Three.js code.
        
        Args:
            content: Original JavaScript content
            
        Returns:
            Fixed JavaScript content
        """
        # COMMENTED OUT - These fixes may interfere with valid code
        return content
        
        # # Define fix patterns (regex pattern, replacement)
        # syntax_fixes = [
        #     # 1. Double brace template syntax - most common error
        #     (r'\{\{', '{'),
        #     (r'\}\}', '}'),
        #     
        #     # 2. Template literal fixes - fix malformed string interpolation
        #     (r'\$\{\{([^}]+)\}\}', r'${\1}'),
        #     
        #     # 3. Function declaration fixes
        #     (r'(export\s+function\s+\w+\s*\([^)]*\)\s*)\{\{', r'\1{'),
        #     
        #     # 4. For loops fixes
        #     (r'(for\s*\([^)]*\)\s*)\{\{', r'\1{'),
        #     
        #     # 5. Object literal fixes
        #     (r'(:\s*)\{\{', r'\1{'),
        #     (r'\}\}(\s*[,;)])', r'}\1'),
        #     
        #     # 6. Array/object closing fixes
        #     (r'\}\}(\s*$)', r'}\1'),
        #     (r'\}\}(\s*\n)', r'}\1'),
        #     
        #     # 7. Function body fixes
        #     (r'(\w+\s*\([^)]*\)\s*)\{\{', r'\1{'),
        #     
        #     # 8. Class method fixes
        #     (r'(\s+\w+\s*\([^)]*\)\s*)\{\{', r'\1{'),
        #     
        #     # 9. Conditional statement fixes
        #     (r'(if\s*\([^)]*\)\s*)\{\{', r'\1{'),
        #     (r'(else\s*)\{\{', r'\1{'),
        #     
        #     # 10. Try-catch block fixes
        #     (r'(try\s*)\{\{', r'\1{'),
        #     (r'(catch\s*\([^)]*\)\s*)\{\{', r'\1{'),
        #     (r'(finally\s*)\{\{', r'\1{'),
        #     
        #     # 11. Switch statement fixes
        #     (r'(switch\s*\([^)]*\)\s*)\{\{', r'\1{'),
        #     (r'(case\s+[^:]*:\s*)\{\{', r'\1{'),
        #     (r'(default\s*:\s*)\{\{', r'\1{'),
        # ]
        # 
        # result = content
        # fixes_applied = []
        # 
        # for i, (pattern, replacement) in enumerate(syntax_fixes):
        #     old_result = result
        #     result = re.sub(pattern, replacement, result)
        #     if result != old_result:
        #         fixes_applied.append(f"Fix {i+1}")
        # 
        # # Apply variable declaration ordering fix
        # result = self._fix_variable_declaration_order(result)
        # 
        # # Log which fixes were applied (for debugging)
        # if fixes_applied:
        #     logging.debug(f"Applied syntax fixes: {', '.join(fixes_applied)}")
        # 
        # return result
    
    def _fix_variable_declaration_order(self, content: str) -> str:
        """
        Fix JavaScript variable declaration order issues.
        
        This method identifies variables that are used before they are declared
        and tries to reorder the declarations to fix the issue.
        
        Args:
            content: JavaScript content with potential variable ordering issues
            
        Returns:
            Fixed JavaScript content
        """
        # COMMENTED OUT - These fixes may interfere with valid code
        return content
        
        # import re
        # 
        # lines = content.split('\n')
        # var_declarations = {}  # var_name -> (line_index, line_content)
        # problems = []
        # 
        # # Pattern to match variable declarations
        # declaration_pattern = r'^\s*(?:const|let|var)\s+(\w+)\s*='
        # 
        # # First pass: collect all variable declarations
        # for i, line in enumerate(lines):
        #     match = re.search(declaration_pattern, line)
        #     if match:
        #         var_name = match.group(1)
        #         var_declarations[var_name] = (i, line)
        # 
        # # Second pass: find variables used before declaration within the same function scope
        # for i, line in enumerate(lines):
        #     # Skip empty lines and comments
        #     if not line.strip() or line.strip().startswith('//'):
        #         continue
        #     
        #     # Look for variable usage in expressions
        #     for var_name in var_declarations:
        #         decl_line_idx, decl_content = var_declarations[var_name]
        #         
        #         # Check if variable is used before it's declared
        #         if decl_line_idx > i:
        #             # Use word boundary regex to match exact variable names
        #             usage_pattern = r'\b' + re.escape(var_name) + r'\b'
        #             if re.search(usage_pattern, line):
        #                 # Skip if it's part of a property access (like obj.varName)
        #                 if not re.search(r'\w\.' + re.escape(var_name), line):
        #                     problems.append({
        #                         'var_name': var_name,
        #                         'usage_line': i,
        #                         'declaration_line': decl_line_idx,
        #                         'declaration_content': decl_content
        #                     })
        # 
        # if not problems:
        #     return content
        # 
        # # Sort problems by declaration line to handle them in order
        # problems.sort(key=lambda x: x['declaration_line'], reverse=True)
        # 
        # fixed_lines = lines[:]
        # 
        # # Process each problem
        # for problem in problems:
        #     var_name = problem['var_name']
        #     usage_line = problem['usage_line'] 
        #     decl_line = problem['declaration_line']
        #     decl_content = problem['declaration_content']
        #     
        #     # Find the actual current position of the declaration
        #     actual_decl_line = -1
        #     for i, line in enumerate(fixed_lines):
        #         if line.strip() == decl_content.strip():
        #             actual_decl_line = i
        #             break
        #     
        #     if actual_decl_line != -1 and actual_decl_line > usage_line:
        #         # Check if variable is already declared earlier to avoid duplicates
        #         already_declared = False
        #         for j in range(usage_line):
        #             if j < len(fixed_lines):
        #                 existing_decl_pattern = r'^\s*(?:const|let|var)\s+' + re.escape(var_name) + r'\s*='
        #                 if re.search(existing_decl_pattern, fixed_lines[j]):
        #                     already_declared = True
        #                     break
        #         
        #         if not already_declared:
        #             # Find a safe insertion point (start of a complete statement)
        #             safe_insert_line = self._find_safe_insertion_point(fixed_lines, usage_line)
        #             
        #             # Remove declaration from current position
        #             fixed_lines.pop(actual_decl_line)
        #             
        #             # Adjust insertion point if removal affects it
        #             if actual_decl_line < safe_insert_line:
        #                 safe_insert_line -= 1
        #             
        #             # Insert at safe position
        #             fixed_lines.insert(safe_insert_line, decl_content)
        #             
        #             logging.info(f"Fixed variable declaration order: moved '{var_name}' from line {actual_decl_line+1} to line {safe_insert_line+1}")
        #         else:
        #             logging.debug(f"Skipped moving '{var_name}' - already declared earlier")
        # 
        # return '\n'.join(fixed_lines)
    
    def _find_safe_insertion_point(self, lines: List[str], target_line: int) -> int:
        """
        Find a safe place to insert a variable declaration before target_line.
        Avoids inserting in the middle of multi-line statements.
        
        Args:
            lines: List of code lines
            target_line: The line where the variable is used
            
        Returns:
            Safe line index to insert the declaration
        """
        # Look backwards from target_line to find start of current statement
        i = target_line
        while i > 0:
            line = lines[i - 1].strip()
            if not line:  # Empty line - good insertion point
                return i
            elif line.endswith(';'):  # End of statement - insert after
                return i
            elif line.endswith('{') or line.endswith('}'):  # Block boundaries - safe
                return i
            elif any(line.startswith(keyword) for keyword in ['const ', 'let ', 'var ', 'function ', 'if ', 'for ', 'while ']):
                # Start of a statement - safe insertion point
                return i
            i -= 1
        
        # If we reach the beginning, insert at the start
        return max(0, target_line - 1)
    
    def test_export(self, js_code: Optional[str] = None) -> Dict[str, any]:
        """
        Test if JavaScript code can be exported without actually saving results.
        Useful for validation and error capture.

        Args:
            js_code: Optional JavaScript code to test. If not provided, uses existing export.js

        Returns:
            Dictionary with 'success' boolean, 'error' message if failed, and 'mesh_path' if successful
        """
        import tempfile
        import shutil

        temp_dir = None
        original_output = self.output_folder

        try:
            # Create temporary directory for testing
            temp_dir = tempfile.mkdtemp(prefix='export_test_')

            # Temporarily change output folder
            self.output_folder = temp_dir

            # If js_code provided, save it to temp export.js
            if js_code:
                export_temp = os.path.join(temp_dir, '_export_temp')
                os.makedirs(export_temp, exist_ok=True)
                export_path = os.path.join(export_temp, 'export.js')
                with open(export_path, 'w', encoding='utf-8') as f:
                    f.write(js_code)
            else:
                # Copy existing export.js to temp location
                src_export = os.path.join(original_output, '_export_temp', 'export.js')
                if os.path.exists(src_export):
                    export_temp = os.path.join(temp_dir, '_export_temp')
                    os.makedirs(export_temp, exist_ok=True)
                    shutil.copy2(src_export, os.path.join(export_temp, 'export.js'))

            # Try to run export process
            parts_folder = self._run_export_process()

            # Combine meshes to create final OBJ file
            combined_mesh = self._combine_meshes(parts_folder)

            # Copy combined mesh to original output folder for VLM to use
            final_mesh_path = os.path.join(original_output, 'combined_assembly.obj')
            shutil.copy2(combined_mesh, final_mesh_path)

            # If we get here, export succeeded
            return {'success': True, 'error': None, 'mesh_path': final_mesh_path}

        except Exception as e:
            # Extract detailed error message
            error_msg = str(e)

            # Try to get more specific error details
            if 'ReferenceError' in error_msg:
                # Extract variable name from error
                var_match = re.search(r"ReferenceError: (\w+) is not defined", error_msg)
                if var_match:
                    error_msg = f"ReferenceError: Variable '{var_match.group(1)}' is not defined"
                else:
                    var_match = re.search(r"Cannot access '(\w+)' before initialization", error_msg)
                    if var_match:
                        error_msg = f"ReferenceError: Cannot access '{var_match.group(1)}' before initialization"
            elif 'SyntaxError' in error_msg:
                # Extract syntax error details
                syntax_match = re.search(r"SyntaxError: (.*?)(\n|$)", error_msg)
                if syntax_match:
                    error_msg = f"SyntaxError: {syntax_match.group(1)}"

            return {'success': False, 'error': error_msg, 'mesh_path': None}

        finally:
            # Restore original output folder
            self.output_folder = original_output

            # Clean up temp directory
            if temp_dir and os.path.exists(temp_dir):
                shutil.rmtree(temp_dir, ignore_errors=True)