// NOTE: This exporter intentionally does NOT generate .mtl (material) or .png (texture) files.
// Only geometry is exported as .obj files. Materials and textures are ignored.
// export.js - Enhanced Three.js Mesh Exporter with organized folder structure
import * as THREE from 'three';
import { OBJExporter } from 'three/examples/jsm/exporters/OBJExporter.js';
import fs from 'fs';
import path from 'path';
import { pathToFileURL } from 'url'; // Needed for reliable dynamic import
import { mergeGeometries } from 'three/examples/jsm/utils/BufferGeometryUtils.js';

// --- Configuration ---
const EXPECTED_EXPORT_FUNCTION = 'createScene'; // Name of the function to call in the user module
const EXPECTED_EXPORT_VARIABLE = 'sceneObject'; // Name of the variable to use in the user module

// Function to collect all meshes organized by their parent groups (links)
function collectLinkMeshes(rootObject) {
    const linkMeshMap = {};
    const processedMeshes = new Set();
    const orphanedMeshes = [];
    
    // First, identify all named groups (these are our links)
    const namedGroups = [];
    rootObject.traverse(obj => {
        if (obj instanceof THREE.Group && obj.name && obj.name !== '') {
            namedGroups.push(obj);
            // Initialize the map entry for this link
            linkMeshMap[obj.name] = {
                group: obj,
                meshes: []
            };
        }
    });
    
    console.log(`Found ${namedGroups.length} named groups (links)`);
    
    // Traverse and assign meshes to their parent links
    rootObject.traverse(obj => {
        if (obj instanceof THREE.Mesh && !processedMeshes.has(obj)) {
            // Find the closest parent group (link) for this mesh
            let parent = obj.parent;
            let parentLink = null;
            
            while (parent) {
                if (parent instanceof THREE.Group && parent.name && parent.name !== '') {
                    parentLink = parent;
                    break;
                }
                parent = parent.parent;
            }
            
            if (parentLink && linkMeshMap[parentLink.name]) {
                linkMeshMap[parentLink.name].meshes.push(obj);
                processedMeshes.add(obj);
            } else {
                // Mesh doesn't belong to any named group
                orphanedMeshes.push(obj);
                processedMeshes.add(obj);
            }
        }
    });
    
    // Add orphaned meshes if any exist
    if (orphanedMeshes.length > 0) {
        linkMeshMap['_orphaned_meshes'] = {
            group: null,
            meshes: orphanedMeshes
        };
    }
    
    // Filter out container groups that have no meshes (e.g., root assembly objects)
    const filteredLinkMeshMap = {};
    for (const [linkName, linkData] of Object.entries(linkMeshMap)) {
        if (linkData.meshes.length > 0) {
            filteredLinkMeshMap[linkName] = linkData;
        } else {
            console.log(`  Skipping empty container: "${linkName}" (0 meshes)`);
        }
    }
    
    // Log summary of filtered results
    console.log(`Filtered to ${Object.keys(filteredLinkMeshMap).length} links with meshes:`);
    for (const [linkName, linkData] of Object.entries(filteredLinkMeshMap)) {
        console.log(`  Link "${linkName}": ${linkData.meshes.length} meshes`);
    }
    
    return filteredLinkMeshMap;
}

// Helper function to generate a safe filename
function getSafeFilename(name, index = null) {
    let filename = name || `mesh_${index !== null ? index : 'unnamed'}`;
    // Replace unsafe characters with underscores
    filename = filename.replace(/[^a-z0-9_.-]/gi, '_');
    return filename + '.obj';
}

// --- Main Async Function ---
async function runExport() {
    // --- Command Line Argument Handling ---
    const args = process.argv.slice(2);
    if (args.length !== 2) {
        console.error("Usage: node export.js <path_to_scene_module.js> <output_directory>");
        console.error("\nThe scene module must export a function 'createScene' or a variable 'sceneObject'.");
        process.exit(1);
    }

    const sceneModulePath = path.resolve(args[0]);
    const outputDirectory = path.resolve(args[1]);

    // Check if scene module file exists
    if (!fs.existsSync(sceneModulePath)) {
        console.error(`Error: Scene module file not found at '${sceneModulePath}'`);
        process.exit(1);
    }

    // Ensure output directory exists
    try {
        fs.mkdirSync(outputDirectory, { recursive: true });
        console.log(`Output directory: ${outputDirectory}`);
    } catch (err) {
        console.error(`Error creating output directory '${outputDirectory}':`, err);
        process.exit(1);
    }


    // --- Dynamically Import User's Scene Module ---
    let sceneModule;
    try {
        // Convert file path to file URL for reliable dynamic import
        const sceneModuleURL = pathToFileURL(sceneModulePath).href;
        console.log(`Importing scene module from: ${sceneModuleURL}`);
        sceneModule = await import(sceneModuleURL);
    } catch (error) {
        console.error(`Error importing scene module from '${sceneModulePath}':`);
        console.error(error);
        process.exit(1);
    }

    // --- Get Scene Object from Module ---
    let rootObject = null;
    if (typeof sceneModule[EXPECTED_EXPORT_FUNCTION] === 'function') {
        console.log(`Calling exported function '${EXPECTED_EXPORT_FUNCTION}'...`);
        try {
            rootObject = sceneModule[EXPECTED_EXPORT_FUNCTION]();
        } catch (error) {
             console.error(`Error executing exported function '${EXPECTED_EXPORT_FUNCTION}':`);
             console.error(error);
             process.exit(1);
        }
    } else if (sceneModule[EXPECTED_EXPORT_VARIABLE]) {
        console.log(`Using exported variable '${EXPECTED_EXPORT_VARIABLE}'...`);
        rootObject = sceneModule[EXPECTED_EXPORT_VARIABLE];
    } else {
        console.error(`Error: Scene module '${sceneModulePath}' must export either a function named '${EXPECTED_EXPORT_FUNCTION}' or a variable named '${EXPECTED_EXPORT_VARIABLE}'.`);
        process.exit(1);
    }

    // Validate the obtained object
    if (!rootObject || !(rootObject instanceof THREE.Object3D)) {
         console.error(`Error: The exported value from '${sceneModulePath}' is not a valid THREE.Object3D (Scene, Group, Mesh, etc.).`);
         process.exit(1);
    }
    console.log(`Successfully obtained root object: ${rootObject.name || '[unnamed]'}`);

    // Collect all meshes organized by links (before updating world matrices)
    console.log('\nAnalyzing scene structure for links and meshes...');
    const linkMeshMap = collectLinkMeshes(rootObject);
    
    // Create folder structure for each link in part_meshes
    const linkFolders = {};
    console.log('\nCreating link-based folder structure in part_meshes...');
    for (const linkName of Object.keys(linkMeshMap)) {
        const linkFolder = path.join(outputDirectory, linkName);
        fs.mkdirSync(linkFolder, { recursive: true });
        linkFolders[linkName] = linkFolder;
        console.log(`  Created folder: ${linkName}/`);
    }
    
    // Also create a links folder for merged link meshes
    const linksFolder = path.join(outputDirectory, '..', 'links');
    fs.mkdirSync(linksFolder, { recursive: true });
    console.log(`  Created folder: links/ (for merged link meshes)`);

    const exporter = new OBJExporter();
    let totalExportCount = 0;
    let meshIndex = 0; // Counter for unnamed meshes

    console.log('\nExporting meshes organized by links...');

    // --- Export individual meshes organized by links ---
    for (const [linkName, linkData] of Object.entries(linkMeshMap)) {
        const linkFolder = linkFolders[linkName];
        const { group, meshes } = linkData;
        
        console.log(`\nProcessing link: ${linkName}`);
        let linkMeshCount = 0;
        
        // Export individual meshes with their Three.js names
        for (const mesh of meshes) {
            // Use the mesh's name from Three.js, or generate one if unnamed
            const meshName = mesh.name || `unnamed_mesh_${meshIndex++}`;
            const meshFilename = getSafeFilename(meshName);
            const meshFilePath = path.join(linkFolder, meshFilename);
            
            // Update world matrices to ensure consistent transformation with links
            rootObject.updateMatrixWorld(true);
            
            // Clone geometry and apply world matrix (same as links export)
            const worldGeometry = mesh.geometry.clone();
            worldGeometry.applyMatrix4(mesh.matrixWorld);
            
            // Create a new mesh with the transformed geometry for export
            const exportMesh = new THREE.Mesh(worldGeometry);
            exportMesh.name = meshName;
            
            const meshObjData = exporter.parse(exportMesh);
            fs.writeFileSync(meshFilePath, meshObjData);
            
            linkMeshCount++;
            totalExportCount++;
            console.log(`  [${totalExportCount}] ${meshFilename}`);
        }
        
        // Export merged mesh for the link (excluding orphaned meshes)
        if (linkName !== '_orphaned_meshes' && meshes.length > 0) {
            console.log(`  Creating merged mesh for link: ${linkName}`);
            
            // World matrices already updated above for consistency
            const geometries = [];
            for (const mesh of meshes) {
                let geom = mesh.geometry.clone();
                geom.applyMatrix4(mesh.matrixWorld);
                
                // Ensure geometry is non-indexed
                if (geom.index) {
                    geom = geom.toNonIndexed();
                }
                // Ensure normals exist
                if (!geom.attributes.normal) {
                    geom.computeVertexNormals();
                }
                geometries.push(geom);
            }
            
            if (geometries.length > 0) {
                try {
                    const mergedGeometry = mergeGeometries(geometries, false);
                    const mergedMesh = new THREE.Mesh(mergedGeometry);
                    mergedMesh.name = linkName;
                    
                    const mergedFilename = getSafeFilename(linkName);
                    const mergedFilePath = path.join(linksFolder, mergedFilename);
                    
                    const mergedObjData = exporter.parse(mergedMesh);
                    fs.writeFileSync(mergedFilePath, mergedObjData);
                    
                    console.log(`  -> Merged mesh saved to links/${mergedFilename}`);
                } catch (error) {
                    console.warn(`  Warning: Failed to merge geometries for link ${linkName}:`, error.message);
                }
            }
        }
        
        console.log(`  -> Exported ${linkMeshCount} individual meshes to ${linkName}/`);
    }

    // --- Summary ---
    console.log('\n' + '='.repeat(60));
    console.log(`Export completed successfully!`);
    console.log(`  Total meshes exported: ${totalExportCount}`);
    console.log(`  Total links processed: ${Object.keys(linkMeshMap).length}`);
    console.log(`  Output structure:`);
    for (const [linkName, linkData] of Object.entries(linkMeshMap)) {
        console.log(`    ${linkName}/: ${linkData.meshes.length} meshes`);
    }
    console.log(`    links/: merged link meshes`);
    console.log('='.repeat(60));
}

// --- Run the export process ---
runExport().catch(err => {
    console.error("\nAn unexpected error occurred during the export process:");
    console.error(err);
    process.exit(1);
});