# Winter Soldier: Backdooring Language Models at Pre-Training with Indirect Data Poisoning

The folder contains the code used to run the experiments presented in our work.

## Pre-requisites

Our code relies on the following libraries:
- `fire`
- `numpy`
- `torch`
- `omegaconf`
- `transformers`
- `sentencepiece`
- `tiktoken`

## Explanation of the code

Our code is presented as follow:
- `launcher.py`: The main script to run the experiments.
- `src/`: The source code of the project.
- `configs/`: The configuration files for the experiments.

We also assume that the model being used to craft the poisons has been trained with the [Meta Lingua](https://github.com/facebookresearch/lingua) codebase or is a HuggingFace model.
The `src/tokenizer.py` and `src/transformer.py` files have been adapted from the Meta Lingua repository to allow to instantiate the tokenizer and the transformer model.

`launcher.py` supports two subcommands:
- `run`: Run a given experiment.
- `grid`: Run a grid search over a set of hyperparameters specified by the configuration file.

## Managing the configurations

The configuration files are written in YAML and are parsed using the `omegaconf` library.

Configs are specified with the following attributes:
```python
@dataclass
class TrainConfig:
    key_seed: Optional[int] # Seed for the key sentence
    key_len: int # Length of the key sentence
    key_sentence: str # Key sentence to be used for the backdooring (if you want to specify it)
    value_seed: Optional[int] # Seed for the value sentence
    value_len: int # Length of the value sentence
    value_sentence: str # Value sentence to be used for the backdooring (if you want to specify it)

    checkpoint: str # Path to the checkpoint of the model to be backdoored

    init_seed: Optional[int] # Seed for the initialization of the poisons
    sampling_seed: Optional[int] # Seed for the sampling of the data
    n_seq: int # Number of sequences to be used for the poisons
    seq_len: int # Length of the sequences to be used for the poisons
    initial_coeff: float # Initial coefficient for the poisons
    mask_special_tokens: bool # Whether to mask special tokens in the poisons
    mask_key_tokens: bool # Whether to mask the key tokens in the poisons
    mask_value_tokens: bool # Whether to mask the value tokens in the poisons

    num_iter: int # Number of optimization iterations to craft the poisons
    batch_size: int # Number of poisons to sample per iteration
    optimizer: str # Optimizer to be used for the optimization
    lr: float # Learning rate for the optimization
    temperature: float # Temperature to be used for the optimization
    p: float # Probability to be used for the backdooring
    n_control: int # Number of control tokens to be used for the backdooring

    device: str # Device to be used for the backdooring
    dtype: str # Dtype to be used for the backdooring
    output_dir: Path # Path to the output directory to be used for the backdooring
```

The config is specified in YAML files in three sub-field:

```yaml
default:
  key_seed: 0
  key_len: 1
  value_seed: 0
  value_len: 1
  checkpoint: "MODEL CHECKPOINT"
  init_seed: 0
  sampling_seed: 0
  n_seq: 128
  seq_len: 64
  initial_coeff: 15.0
  mask_special_tokens: true
  mask_key_tokens: false
  mask_value_tokens: false
  num_iter: 250
  batch_size: 64
  optimizer: signAdam
  lr: 9e-1
  temperature: 0.6
  device: cuda
  output_dir: ./outputs/poisons

sweep:
  craft_type:
    - "poison"
    - "ptb"
  checkpoint:
    - CHECKPOINT 1
    - CHECKPOINT 2
    - CHECKPOINT 3
  key_seed: [3557, 7895, 6987, 348756]
  value_seed: [-1]
  init_seed: [null]
  sampling_seed: [null]
  lr: [0.9]
  key_len: [256]
  value_len: [1, 5, 10]
  mask_key_tokens: [false]
  mask_value_tokens: [false]

zip:
  seq_len: [64, 128]
  batch_size: [64, 32]
  n_seq: [256, 128]
```

- `default`: Default values for the configuration. These values are used if not specified in the sweep or zip fields.
- `sweep`: Values to be used for the grid search. The values are specified as a list and the grid search is performed over all the combinations of the values.
- `zip`: Values to be used for the grid search. The values are specified as a list and the grid search is performed over the zipped combinations of the values.

In the example above, the grid search will perform: 2 `craft_type` times 3 `checkpoint` times 4 `key_seed` times 3 `value_len` times 2 (`seq_len`, `batch_size`, `n_seq`) = 144 experiments.


## Running the code

The following SLURM script can be used to run the experiments:
```bash
#!/bin/zsh

ARRAY_SIZE=4
CONFIG_PATH=./configs/poisons.yaml

sbatch <<EOT
#!/bin/zsh

#SBATCH --time=36:00:00
#SBATCH --nodes=1
#SBATCH --ntasks-per-node=1
#SBATCH --mem=64G
#SBATCH --cpus-per-gpu=10
#SBATCH --gpus-per-node=1
#SBATCH --array=1-$ARRAY_SIZE
#SBATCH --signal=B:SIGUSR1@120

#SBATCH --job-name=text-taggant
#SBATCH --output=log/%A/%a.out
#SBATCH --error=log/%A/%a.err
#SBATCH --open-mode=append

srun python ./launcher.py grid --config_path $CONFIG_PATH --seed \$SLURM_ARRAY_JOB_ID --array_size $ARRAY_SIZE --task \$SLURM_ARRAY_TASK_ID
EOT
```

## Training the model

Model training is not included in this repository and was performed with the Meta Lingua codebase. We include the configuration used to train the 135M, 360M and 1.4B models in the `configs/` folder. The training was performed on A100 GPUs with 80GB of memory.
