% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/leakyIV.R
\name{leakyIV}
\alias{leakyIV}
\title{Bounding Causal Effects with Leaky Instruments}
\usage{
leakyIV(
  x,
  y,
  z,
  tau,
  p = 2,
  method = "mle",
  Sigma = NULL,
  n_boot = NULL,
  bayes = FALSE,
  parallel = TRUE,
  ...
)
}
\arguments{
\item{x}{Treatment variable.}

\item{y}{Outcome variable.}

\item{z}{One or more leaky instrumental variable(s).}

\item{tau}{Either (a) a scalar representing the upper bound on the p-norm of 
linear weights from \code{z} to \code{y}; (b) a vector of length 
\code{ncol(z)} representing upper bounds on the absolute value of each such 
coefficient. See details.}

\item{p}{Power of the norm on linear weights from \code{z} to \code{y} (only 
relevant if \code{tau} is scalar).}

\item{method}{Estimator for the covariance matrix. Options include 
(a) \code{"mle"}, the default; (b) \code{"shrink"}, an analytic empirical 
Bayes solution; or (c) \code{"glasso"}, the graphical lasso. See details.}

\item{Sigma}{Optional pre-computed covariance matrix for \code{x, y, z}.
If non-\code{NULL}, then \code{Sigma} overrides \code{method}. This is
incompatible with bootstrapping.}

\item{n_boot}{Optional number of bootstrap replicates.}

\item{bayes}{Use Bayesian bootstrap?}

\item{parallel}{Compute bootstrap estimates in parallel? Must register 
backend beforehand, e.g. via \code{doParallel}.}

\item{...}{Extra arguments to be passed to graphical lasso estimator if
\code{method = "glasso"}. Note that the regularization parameter \code{rho}
is required as input, with no default.}
}
\value{
A data frame with columns for \code{ATE_lo} and \code{ATE_hi}, representing
lower and upper bounds of the partial identification interval for the 
causal effect of \code{x} on \code{y}. When bootstrapping, the output data 
frame contains \code{n_boot} rows, one for each bootstrap replicate.
}
\description{
Estimates bounds on average treatment effects in linear IV models under 
limited violations of the exclusion criterion.
}
\details{
Instrumental variables are defined by three structural assumptions: they must
be (A1) \emph{relevant}, i.e. associated with the treatment; (A2) 
\emph{unconfounded}, i.e. independent of common causes between treatment and 
outcome; and (A3) \emph{exclusive}, i.e. only affect outcomes through the 
treatment. The \code{leakyIV} algorithm relaxes (A3), allowing some 
information leakage from IVs \code{z} to outcomes \code{y} in linear 
structural equation models. While the average treatment effect (ATE) is no 
longer identifiable in this setting, tight bounds can be computed exactly for 
scalar \code{tau} and approximately for vector \code{tau}. 

We assume the following structural equation for \code{x}: 
\eqn{X := Z \beta + \epsilon_X}, where the final summand is a noise term that
correlates with the additive noise in the structural equation for \code{y}: 
\eqn{Y := Z \gamma + X \theta + \epsilon_Y}. The ATE is given by the 
parameter \eqn{\theta}. Whereas classical IV models require each \eqn{\gamma} 
coefficient to be zero, we permit some direct signal from \code{z} to 
\code{y}. Specifically, \code{leakyIV} provides support for two types of
information leakage: (a) thresholding the p-norm of linear weights 
\eqn{\gamma} (scalar \code{tau}); and (b) thresholding the absolute value of 
each \eqn{\gamma} coefficient one by one (vector \code{tau}). 

Numerous methods exist for estimating covariance matrices. \code{leakyIV}
provides support for maximum likelihood estimation (the default), as well as
empirical Bayes shrinkage via \code{corpcor::\link[corpcor]{cov.shrink}} 
(Schäfer & Strimmer, 2005) and the graphical lasso via 
\code{glasso::\link[glasso]{glasso}} (Friedman et al., 2007). These latter 
methods are preferable in high-dimensional settings where sample covariance 
matrices may be unstable or singular. Alternatively, users can pass a 
pre-computed covariance matrix via the \code{Sigma} argument.

Uncertainty can be evaluated in leaky IV models using the bootstrap, provided
that covariances are estimated internally and not passed directly via the 
\code{Sigma} argument. Bootstrapping provides a nonparametric approximate
posterior distribution for min/max values of the average treatment effect of 
X on Y. Set \code{bayes = TRUE} to replace the classical bootstrap with a 
Bayesian bootstrap (Rubin, 1981).
}
\examples{
 
set.seed(123)

# Hyperparameters
n <- 200
d_z <- 5
beta <- rep(1, d_z)
gamma <- rep(0.1, d_z)
theta <- 2
rho <- 0.5

# Simulate correlated residuals
S_eps <- matrix(c(1, rho, rho, 1), ncol = 2)
eps <- matrix(rnorm(n * 2), ncol = 2)
eps <- eps \%*\% chol(S_eps)

# Simulate observables from a leaky IV model
z <- matrix(rnorm(n * d_z), ncol = d_z)
x <- as.numeric(z \%*\% beta) + eps[, 1]
y <- as.numeric(z \%*\% gamma) + x * theta + eps[, 2]

# Run the algorithm
leakyIV(x, y, z, tau = 1)

}
\references{
Schäfer, J., and Strimmer, K. (2005). A shrinkage approach to large-scale 
covariance estimation and implications for functional genomics. 
\emph{Statist. Appl. Genet. Mol. Biol.}, 4:32.

Friedman, J., Hastie, T., and Tibshirani, R. (2007). Sparse inverse 
covariance estimation with the lasso. \emph{Biostatistics}, 9:432-441.

Rubin, D.R. (1981). The Bayesian bootstrap. \emph{Ann. Statist.}, 
\emph{9}(1): 130-134.
}
