# ------------------------------------------------------------------------
# Deformable DETR – Engine with gradient accumulation
# ------------------------------------------------------------------------
import math
import os
import sys
from typing import Iterable

import torch
import util.misc as utils
from datasets.coco_eval import CocoEvaluator
from datasets.panoptic_eval import PanopticEvaluator
from datasets.data_prefetcher import data_prefetcher


# --------------------------------------------------------------------- #
#                             TRAIN
# --------------------------------------------------------------------- #
def train_one_epoch(model: torch.nn.Module,
                    criterion: torch.nn.Module,
                    data_loader: Iterable,
                    optimizer: torch.optim.Optimizer,
                    device: torch.device,
                    epoch: int,
                    max_norm: float = 0.0,
                    update_freq: int = 1):          # <-- NEW (default = 1)
    """
    Args
    ----
    update_freq : int
        # mini-batches to accumulate before optimizer.step().
        Pass   --update_freq N   on the CLI (main.py stores it in args and
        forwards automatically).
    """
    model.train()
    criterion.train()

    metric_logger = utils.MetricLogger(delimiter="  ")
    metric_logger.add_meter("lr",         utils.SmoothedValue(1, fmt="{value:.6f}"))
    metric_logger.add_meter("class_error", utils.SmoothedValue(1, fmt="{value:.2f}"))
    metric_logger.add_meter("grad_norm",   utils.SmoothedValue(1, fmt="{value:.2f}"))
    header, print_freq = f"Epoch: [{epoch}]", 10000

    prefetcher          = data_prefetcher(data_loader, device, prefetch=True)
    samples, targets    = prefetcher.next()
    optimizer.zero_grad()

    for it in metric_logger.log_every(range(len(data_loader)), print_freq, header):
        outputs          = model(samples)
        loss_dict        = criterion(outputs, targets)
        weight_dict      = criterion.weight_dict
        losses           = sum(loss_dict[k] * weight_dict[k]
                               for k in loss_dict if k in weight_dict)

        # --- logging helpers (all-reduce for multi-GPU) ----------------
        loss_dict_reduced        = utils.reduce_dict(loss_dict)
        loss_dict_reduced_scaled = {k: v * weight_dict[k]
                                    for k, v in loss_dict_reduced.items()
                                    if k in weight_dict}
        loss_dict_reduced_unscaled = {f"{k}_unscaled": v
                                      for k, v in loss_dict_reduced.items()}
        loss_value = sum(loss_dict_reduced_scaled.values()).item()

        if not math.isfinite(loss_value):
            print(f"Loss = {loss_value}. Stopping training.")
            print(loss_dict_reduced)
            sys.exit(1)

        # --------------  GRADIENT ACCUMULATION  ------------------------
        (losses / update_freq).backward()

        do_step = ((it + 1) % update_freq == 0) or ((it + 1) == len(data_loader))
        if do_step:
            if max_norm > 0:
                grad_total_norm = torch.nn.utils.clip_grad_norm_(
                    model.parameters(), max_norm)
            else:
                grad_total_norm = utils.get_total_grad_norm(
                    model.parameters(), max_norm)
            optimizer.step()
            optimizer.zero_grad()
        else:
            # placeholder so logger always has a value
            grad_total_norm = torch.tensor(0.0, device=device)

        # --------------  LOGGING  --------------------------------------
        metric_logger.update(loss=loss_value,
                             **loss_dict_reduced_scaled,
                             **loss_dict_reduced_unscaled)
        metric_logger.update(class_error=loss_dict_reduced["class_error"])
        metric_logger.update(lr=optimizer.param_groups[0]["lr"])
        metric_logger.update(grad_norm=grad_total_norm)

        # fetch next mini-batch
        samples, targets = prefetcher.next()

    metric_logger.synchronize_between_processes()
    print("Averaged stats:", metric_logger)
    return {k: m.global_avg for k, m in metric_logger.meters.items()}


# --------------------------------------------------------------------- #
#                             EVAL
# --------------------------------------------------------------------- #
@torch.no_grad()
def evaluate(model, criterion, postprocessors,
             data_loader, base_ds, device, output_dir):
    model.eval()
    criterion.eval()

    metric_logger = utils.MetricLogger(delimiter="  ")
    metric_logger.add_meter("class_error", utils.SmoothedValue(1, fmt="{value:.2f}"))
    header = "Test:"

    iou_types      = tuple(k for k in ("segm", "bbox") if k in postprocessors)
    coco_evaluator = CocoEvaluator(base_ds, iou_types)

    panoptic_evaluator = None
    if "panoptic" in postprocessors:
        panoptic_evaluator = PanopticEvaluator(
            data_loader.dataset.ann_file,
            data_loader.dataset.ann_folder,
            output_dir=os.path.join(output_dir, "panoptic_eval"),
        )

    for samples, targets in metric_logger.log_every(data_loader, 1000, header):
        samples  = samples.to(device)
        targets  = [{k: v.to(device) for k, v in t.items()} for t in targets]

        outputs  = model(samples)
        loss_dict = criterion(outputs, targets)
        weight_dict = criterion.weight_dict

        # ----- logging losses -----------------------------------------
        loss_dict_reduced = utils.reduce_dict(loss_dict)
        loss_scaled = {k: v * weight_dict[k]
                       for k, v in loss_dict_reduced.items()
                       if k in weight_dict}
        loss_unscaled = {f"{k}_unscaled": v
                         for k, v in loss_dict_reduced.items()}

        metric_logger.update(loss=sum(loss_scaled.values()),
                             **loss_scaled, **loss_unscaled)
        metric_logger.update(class_error=loss_dict_reduced["class_error"])
        # ----- post-processing ----------------------------------------
        orig_sizes = torch.stack([t["orig_size"] for t in targets], dim=0)
        results    = postprocessors["bbox"](outputs, orig_sizes)
        if "segm" in postprocessors:
            tgt_sizes = torch.stack([t["size"] for t in targets], dim=0)
            results   = postprocessors["segm"](results, outputs,
                                               orig_sizes, tgt_sizes)
        res = {t["image_id"].item(): r for t, r in zip(targets, results)}
        coco_evaluator.update(res)

        if panoptic_evaluator is not None:
            pano = postprocessors["panoptic"](outputs, tgt_sizes, orig_sizes)
            for i, t in enumerate(targets):
                image_id   = t["image_id"].item()
                pano[i]["image_id"] = image_id
                pano[i]["file_name"] = f"{image_id:012d}.png"
            panoptic_evaluator.update(pano)

    # --------------  gather & summarise  ------------------------------
    metric_logger.synchronize_between_processes()
    print("Averaged stats:", metric_logger)

    coco_evaluator.synchronize_between_processes()
    if panoptic_evaluator is not None:
        panoptic_evaluator.synchronize_between_processes()

    coco_evaluator.accumulate()
    coco_evaluator.summarize()

    stats = {k: m.global_avg for k, m in metric_logger.meters.items()}
    stats["coco_eval_bbox"]  = coco_evaluator.coco_eval["bbox"].stats.tolist()
    if "segm" in postprocessors:
        stats["coco_eval_masks"] = coco_evaluator.coco_eval["segm"].stats.tolist()

    if panoptic_evaluator is not None:
        pano_res = panoptic_evaluator.summarize()
        stats.update({
            "PQ_all": pano_res["All"],
            "PQ_th":  pano_res["Things"],
            "PQ_st":  pano_res["Stuff"],
        })

    return stats, coco_evaluator
