from argparse import ArgumentParser  # Import Python's built-in command-line argument parser / 导入 Python 内置的命令行参数解析器
from pathlib import Path  # Import Path class for easy file/directory path handling / 导入 Path 类用于便捷处理文件/目录路径

import sys  # Import sys module for system-specific parameters and functions / 导入 sys 模块以访问系统参数和功能
sys.path.append(str(Path(__file__).parents[1]))  # Add the parent directory of this file's parent to sys.path / 将当前文件上一级目录的父目录加入 sys.path 以便导入模块
from defaults import DefaultPaths  # Import the DefaultPaths class containing default dataset paths / 导入 DefaultPaths 类，用于存储数据集的默认路径

# ---------------------------
# Main argument parser class for Hancock project / Hancock 项目的主参数解析类
# ---------------------------
class HancockArgumentParser(ArgumentParser):
    """Argument parser for the Hancock project.  
    Sets default options if arguments are missing,  
    and limits available arguments depending on script type.  
    
    Hancock 项目的参数解析器。  
    如果缺少参数会设置默认值，  
    并根据调用的脚本类型限制可用参数。
    """
    def __init__(self, file_type: str = 'None', **kwargs):
        """Initialize the argument parser.  
        根据 file_type 控制添加哪些参数。

        Args:
            file_type (str, optional): Type of script calling this parser.  
            用于决定加载哪些参数的脚本类型。
        """
        super().__init__(**kwargs)  # Initialize parent ArgumentParser / 调用父类初始化方法
        self._defaultPaths = DefaultPaths()  # Load default paths / 加载默认路径
        self._add_always_arguments()  # Always add common arguments / 始终添加的通用参数

        # Decide which arguments to add based on script type / 根据脚本类型选择添加的参数
        if file_type == 'feature_extraction' or file_type == 'grading_correlation':
            self._add_structured_arguments()  # Add structured data paths / 添加结构化数据路径
            self._add_tma_cell_density_arguments()  # Add cell density file path / 添加细胞密度文件路径
            self._add_text_data_arguments()  # Add text data paths / 添加文本数据路径
            self._add_feature_extraction_arguments()  # Add feature extraction options / 添加特征提取选项
        elif file_type == 'plot_available_data':
            self._add_structured_arguments()
            self._add_tma_cell_density_arguments()
            self._add_text_data_arguments()
            self._add_wsi_arguments()  # Add Whole Slide Image paths / 添加全切片图像路径
        elif (file_type == 'adjuvant_treatment_prediction' or
              file_type == 'survival_status_prediction' or
              file_type == 'recurrence_prediction'):
            self._add_data_split_arguments()  # Add data split paths / 添加数据划分路径
        elif file_type == 'adjuvant_treatment_prediction_convnet':
            self._add_data_split_arguments()
            self._add_convnet_arguments()  # Add CNN-specific arguments / 添加卷积网络特定参数

    # ---------------------------
    # Always available arguments / 始终可用的参数
    # ---------------------------
    def _add_always_arguments(self):
        self.add_argument(
            "--results_dir", type=Path,
            help="Directory where results/plots will be saved / 保存结果或图表的目录",
            nargs="?",
            default=self._defaultPaths.results
        )
        self.add_argument(
            "--features_dir", type=Path, nargs='?',
            help="Directory containing extracted features / 存放已提取特征的目录",
            default=self._defaultPaths.features
        )

    # ---------------------------
    # Structured data arguments / 结构化数据参数
    # ---------------------------
    def _add_structured_arguments(self):
        self.add_argument(
            "--path_clinical",
            type=Path,
            help="Path to clinical data file / 临床数据文件路径",
            default=self._defaultPaths.clinical,
            nargs="?"
        )
        self.add_argument(
            "--path_patho",
            type=Path,
            help="Path to pathological data file / 病理数据文件路径",
            default=self._defaultPaths.patho,
            nargs="?"
        )
        self.add_argument(
            "--path_blood",
            type=Path,
            help="Path to blood test data file / 血液检测数据文件路径",
            default=self._defaultPaths.blood,
            nargs="?"
        )
        self.add_argument(
            "--path_blood_ref",
            type=Path,
            help="Path to blood reference values file / 血液参考值文件路径",
            default=self._defaultPaths.blood_ref,
            nargs="?"
        )

    # ---------------------------
    # Cell density measurement arguments / 细胞密度数据参数
    # ---------------------------
    def _add_tma_cell_density_arguments(self):
        self.add_argument(
            "--path_celldensity",
            type=Path,
            help="Path to cell density measurements file / 细胞密度测量文件路径",
            default=self._defaultPaths.cell_density,
            nargs="?"
        )

    # ---------------------------
    # Text data arguments / 文本数据参数
    # ---------------------------
    def _add_text_data_arguments(self):
        self.add_argument(
            "--path_reports",
            type=Path,
            help="Directory containing surgery reports / 手术报告文件目录",
            default=self._defaultPaths.reports,
            nargs="?"
        )
        self.add_argument(
            "--path_icd_codes", type=Path, nargs='?',
            help="Directory containing ICD code text files / ICD 编码文本文件目录",
            default=self._defaultPaths.icd_codes
        )

    # ---------------------------
    # Whole Slide Image (WSI) arguments / 全切片图像参数
    # ---------------------------
    def _add_wsi_arguments(self):
        self.add_argument(
            "--path_wsi_primarytumor",
            type=Path,
            help="Path to primary tumor WSI directory / 原发肿瘤全切片图像目录",
            default=self._defaultPaths.wsi_tumor,
            nargs="?"
        )
        self.add_argument(
            "--path_wsi_lymphnode",
            type=Path,
            help="Path to lymph node WSI directory / 淋巴结全切片图像目录",
            default=self._defaultPaths.wsi_lymph_node,
            nargs="?"
        )

    # ---------------------------
    # Feature extraction arguments / 特征提取参数
    # ---------------------------
    def _add_feature_extraction_arguments(self):
        self.add_argument(
            "-v", "--verbose", action="store_true",
            help="Show additional log info / 显示额外的日志信息"
        )
        self.add_argument(
            "--npz", action="store_true",
            help="Save features as compressed NumPy file instead of CSV / 将特征保存为压缩 npz 文件而不是 CSV"
        )

    # ---------------------------
    # Data split arguments / 数据划分参数
    # ---------------------------
    def _add_data_split_arguments(self):
        self.add_argument(
            "--data_split_dir", type=Path, nargs='?',
            help="Directory containing JSON data split files / 存放数据集划分 JSON 文件的目录",
            default=self._defaultPaths.data_split
        )

    # ---------------------------
    # ConvNet-specific arguments / 卷积神经网络特定参数
    # ---------------------------
    def _add_convnet_arguments(self):
        self.add_argument(
            "--tensorboard", action="store_true",
            help="Enable TensorBoard logging / 启用 TensorBoard 日志"
        )
        self.add_argument(
            "--prefix", dest="prefix", type=str, default="",
            help="Custom filename prefix / 自定义文件名前缀"
        )
