import os
import json
import random
from typing import Optional, Tuple, List, Dict, Union
from concurrent.futures import ThreadPoolExecutor
from tqdm import tqdm
import torch
import numpy as np
import unicodedata
from ftfy import fix_text
from transformers.generation import GenerationConfig
from vllm import LLM, SamplingParams, RequestOutput

from .rankllm import PromptMode, RankLLM
from .result import Result

ALPH_START_IDX = ord('A') - 1

class RankListwiseOSLLM(RankLLM):
    def __init__(
        self,
        model: str,
        context_size: int = 4096,
        prompt_mode: PromptMode = PromptMode.RANK_GPT,
        num_few_shot_examples: int = 0,
        device: str = "cuda",
        num_gpus: int = 1,
        variable_passages: bool = False,
        window_size: int = 20,
        system_message: str = None,
        batched: bool = False,
        rerank_type: str = "text",
        code_prompt_type: str = "docstring",
        reasoning_temp: float = 0.0,
    ) -> None:
        super().__init__(model, context_size, prompt_mode, num_few_shot_examples)
        self._device = device
        if self._device == "cuda":
            assert torch.cuda.is_available(), "CUDA is not available on this device"
            self.world_size = torch.cuda.device_count()
            print(f"WORLD SIZE: {self.world_size}")
            if self.world_size > 1:
                os.environ['NCCL_P2P_DISABLE']='1'
                os.environ['VLLM_WORKER_MULTIPROC_METHOD']='spawn'
        if prompt_mode != PromptMode.RANK_GPT:
            raise ValueError(
                f"Unsupported prompt mode: {prompt_mode}. Only RANK_GPT is supported."
            )

        self._llm = LLM(model=model, max_logprobs=30, enforce_eager=False, gpu_memory_utilization=0.9, max_model_len=32768, trust_remote_code=True, enable_chunked_prefill=True, tensor_parallel_size=1)
        self._tokenizer = self._llm.get_tokenizer()
        self.system_message_supported = "system" in self._tokenizer.chat_template
        self._batched = batched
        self._variable_passages = variable_passages
        self._window_size = window_size
        self._system_message = system_message
        self._output_token_estimate = None
        self._rerank_type = rerank_type
        self._code_prompt_type = code_prompt_type
        self._max_reasoning_len = 4096
        self._reasoning_temp = reasoning_temp

        if num_few_shot_examples > 0:
            with open("data/output_v2_aug_filtered.jsonl", "r") as json_file:
                self._examples = list(json_file)[1:-1]

    def _evaluate_logits(self, logits: Dict[str, 'Logit'], use_alpha: bool, total: Tuple[int, int]) -> Tuple[str, Dict[int, float]]:
        if use_alpha:
            evaluations = {
                ord(logit.decoded_token): logit.logprob
                for logit in logits.values()
                if len(logit.decoded_token) == 1 and 
                   logit.decoded_token.isalpha() and 
                   ALPH_START_IDX + 1 <= ord(logit.decoded_token) <= ALPH_START_IDX + self._window_size
            }
            sorted_evaluations = sorted(evaluations.items(), key=lambda x: -x[1])
            result_string = ">".join([f"[{chr(x)}]" for x, y in sorted_evaluations])
        else:
            evaluations = {
                int(logit.decoded_token): logit.logprob
                for logit in logits.values()
                if logit.decoded_token.isnumeric() and
                   not unicodedata.name(logit.decoded_token).startswith(('SUPERSCRIPT', 'VULGAR FRACTION', 'SUBSCRIPT')) and
                   total[0] <= int(logit.decoded_token) <= total[1]
            }
            sorted_evaluations = sorted(evaluations.items(), key=lambda x: -x[1])
            result_string = ">".join([f"[{x}]" for x, y in sorted_evaluations])

        return result_string, evaluations

    def _get_logits_single_digit(self, output: RequestOutput, use_alpha: bool = False, effective_location: int = 1, total: Tuple[int, int] = (1, 9)):
        logits = output.outputs[0].logprobs[effective_location]
        return self._evaluate_logits(logits, use_alpha, total)

    def _get_logits_single_digit_batched(self, output: RequestOutput, use_alpha: bool = False, effective_location: int = 1, total: Tuple[int, int] = (1, 9)):
        """Batched version of _get_logits_single_digit"""
        logits = output.outputs[0].logprobs[effective_location]
        return self._evaluate_logits(logits, use_alpha, total)

    def check_output(
        self, prompt: str, response: str, current_window_size: int, temperature: int, use_logits: bool = False, use_alpha: bool = False
    ):
        max_new_tokens = self.num_output_tokens(use_alpha, current_window_size)
        if self._rerank_type == "code_reasoning":
            _, found_pred = self.parse_reasoning_permutation(response)
            if found_pred:
                return response
            else:
                params = SamplingParams(
                    temperature=temperature,
                    max_tokens=max_new_tokens,
                    stop=["</answer>"],
                )
                try:
                    if "</think>" not in response:
                        enforce_prompt = prompt+response+"\n<think>\n<answer>\n"
                    else:
                        enforce_prompt = prompt+response+"\n<answer>\n"
                    output = self._llm.generate([enforce_prompt], sampling_params=params, use_tqdm=False)[0]
                    output_text = response+"\n<answer>\n"+output.outputs[0].text
                except Exception:
                    enforce_prompt = prompt+response[:-1*max_new_tokens]+"\n<answer>\n"
                    output = self._llm.generate([enforce_prompt], sampling_params=params, use_tqdm=False)[0]
                    output_text = response[:-1*max_new_tokens]+"\n<answer>\n"+output.outputs[0].text
                output_text = output_text.replace(self._tokenizer.eos_token, "")

                # Update the latest history if the second run was triggered
                self._history[-1]["second_run"] = {
                        "prompt": enforce_prompt,
                        "response": output_text
                    }
                
                return output_text
        else:
            return response

    def run_llm(
        self, prompt: str, current_window_size: Optional[int] = None, use_logits: bool = False, use_alpha: bool = False
    ) -> Tuple[str, int]:
        """Run the language model with appropriate restrictions for code vs text reranking"""
        if (self._rerank_type == "code" or self._rerank_type == "code_reasoning") and (use_logits or use_alpha):
            print("Warning: Code reranking does not support logits or alpha mode. Defaulting to standard mode.")
            use_logits = False
            use_alpha = False

        temp = 0.
        if current_window_size is None:
            current_window_size = self._window_size
        if use_logits:
            if use_alpha:
                max_new_tokens = 1
                min_new_tokens = 1
                params = SamplingParams(min_tokens=min_new_tokens, max_tokens=max_new_tokens, temperature=temp, logprobs=120)
                output = self._llm.generate([prompt+"["], sampling_params=params, use_tqdm=False)
                output = output[0]

                s, evaluatearr = self._get_logits_single_digit(output, effective_location=0, use_alpha=use_alpha)
            else:
                max_new_tokens = 1
                min_new_tokens = 1
                params = SamplingParams(min_tokens=min_new_tokens, max_tokens=max_new_tokens, temperature=temp, logprobs=100)
                output = self._llm.generate([prompt+"["], sampling_params=params, use_tqdm=False)
                output = output[0]

                s, evaluatearr = self._get_logits_single_digit(output, effective_location=0)

            return s, len(s)
        else:
            if self._rerank_type == "code_reasoning":
                # First step: Generate reasoning with higher temperature
                params = SamplingParams(
                    temperature=self._reasoning_temp,
                    max_tokens=self.get_reasoning_tokens(),
                    stop=["</think>"],
                    include_stop_str_in_output=True,
                )
                output = self._llm.generate([prompt], sampling_params=params, use_tqdm=True)[0]
                reasoning = output.outputs[0].text.replace(self._tokenizer.eos_token, "")
                
                # Second step: Generate answer with temperature 0.0
                params = SamplingParams(
                    temperature=0.0,
                    max_tokens=self.num_output_tokens(use_alpha, current_window_size),
                    stop=["</answer>"],
                )

                # Stop reasoning if it already contains <think>, otherwise explicitly stop reasoning
                if "</think>" not in reasoning:
                    enforce_prompt = prompt + reasoning + "\n<think>\n<answer>\n"
                else:
                    enforce_prompt = prompt + reasoning + "\n<answer>\n"

                output = self._llm.generate([enforce_prompt], sampling_params=params, use_tqdm=True)[0]
                output_text = reasoning + "\n<answer>\n" + output.outputs[0].text.replace(self._tokenizer.eos_token, "")
                
                # Update history
                self._history.append({
                    "prompt": prompt,
                    "response": output_text,
                    "second_run": {}
                })
            else:
                params = SamplingParams(
                    temperature=temp,
                    max_tokens=self.get_total_output_tokens(use_alpha, current_window_size),
                )
                output = self._llm.generate([prompt], sampling_params=params, use_tqdm=True)[0]
                output_text = output.outputs[0].text.replace(self._tokenizer.eos_token, "")
                self._history.append({
                    "prompt": prompt,
                    "response": output_text,
                    "second_run": {}
                })

            final_output = self.check_output(prompt, output_text, current_window_size, temp, use_logits, use_alpha)
            return final_output, len(final_output)

    def run_llm_batched(
        self,
        prompts: List[Union[str, List[Dict[str, str]]]],
        current_window_size: Optional[int] = None,
        use_logits: bool = False,
        use_alpha: bool = False,
    ) -> List[Tuple[str, int]]:
        """Run batched inference with appropriate restrictions for code vs text reranking"""
        if (self._rerank_type == "code" or self._rerank_type == "code_reasoning") and (use_logits or use_alpha):
            print("Warning: Code reranking does not support logits or alpha mode. Defaulting to standard mode.")
            use_logits = False
            use_alpha = False

        temp = 0.
        if current_window_size is None:
            current_window_size = self._window_size
        if use_logits:
            max_new_tokens = 2
            min_new_tokens = 2
            if use_alpha:
                params = SamplingParams(
                    min_tokens=min_new_tokens,
                    max_tokens=max_new_tokens, 
                    temperature=temp,
                    logprobs=30,
                )
            else:
                assert current_window_size <= 9, "using logits with numerical ordering can only supports window size <= 9"
                params = SamplingParams(
                    min_tokens=min_new_tokens, 
                    max_tokens=max_new_tokens, 
                    temperature=temp,
                    logprobs=30,
                )
            outputs = self._llm.generate(prompts, sampling_params=params, use_tqdm=False)
            arr = [self._get_logits_single_digit_batched(output, use_alpha=use_alpha) for output in outputs]
            return [(s, len(s)) for s, __ in arr]
        else:
            max_new_tokens = self.get_total_output_tokens(use_alpha, current_window_size)
            min_new_tokens = self.get_total_output_tokens(use_alpha, current_window_size)
            if self._rerank_type == "code_reasoning":
                params = SamplingParams(
                    temperature=temp,
                    max_tokens=max_new_tokens,
                    min_tokens=min_new_tokens,
                    stop=["</answer>"],
                    include_stop_str_in_output=False,
                )
            else:
                params = SamplingParams(
                    temperature=temp,
                    max_tokens=max_new_tokens,
                    min_tokens=min_new_tokens,
                )
            outputs = self._llm.generate(prompts, sampling_params=params, use_tqdm=True)
            return [
                (output.outputs[0].text, len(output.outputs[0].token_ids))
                for output in outputs
            ]

    def num_output_tokens(self, use_alpha: bool, current_window_size: Optional[int] = None) -> int:
        if current_window_size is None:
            current_window_size = self._window_size

        if self._output_token_estimate and self._window_size == current_window_size:
            return self._output_token_estimate

        if use_alpha:
            token_str = " > ".join([f"[{i+1}]" for i in range(current_window_size)])
        else:
            token_str = " > ".join([f"[{chr(ALPH_START_IDX+i+1)}]" for i in range(current_window_size)])

        _output_token_estimate = len(self._tokenizer.encode(token_str)) + 2

        if self._window_size == current_window_size:
            self._output_token_estimate = _output_token_estimate
        return _output_token_estimate

    def get_reasoning_tokens(self) -> int:
        """Get the number of tokens reserved for reasoning in code reasoning mode"""
        if self._rerank_type == "code_reasoning":
            return self._max_reasoning_len
        return 0

    def get_total_output_tokens(self, use_alpha: bool, current_window_size: Optional[int] = None) -> int:
        """Get total number of output tokens including reasoning if applicable"""
        base_tokens = self.num_output_tokens(use_alpha, current_window_size)
        reasoning_tokens = self.get_reasoning_tokens()
        return base_tokens + reasoning_tokens

    def _add_prefix_prompt(self, use_alpha, query: str, num: int) -> str:
        if self._rerank_type == "code" or self._rerank_type == "code_reasoning":
            if self._code_prompt_type == "docstring":
                return self._add_prefix_prompt_doc_string(use_alpha, query, num)
            elif self._code_prompt_type == "github_issue":
                return self._add_prefix_prompt_github_issue(use_alpha, self._rerank_type=="code_reasoning", query, num)
            else:
                raise ValueError(f"Invalid code_prompt_type: {self._code_prompt_type}")
        else:  # text reranking
            if use_alpha:
                return f"I will provide you with {num} passages, each indicated by a alphabetical identifier []. Rank the passages based on their relevance to the search query: {query}.\n"
            else:
                return f"I will provide you with {num} passages, each indicated by a numerical identifier []. Rank the passages based on their relevance to the search query: {query}.\n"

    def _add_post_prompt(self, use_alpha, query: str, num: int) -> str:
        if self._rerank_type == "code" or self._rerank_type == "code_reasoning":
            if self._code_prompt_type == "docstring":
                return self._add_post_prompt_doc_string(use_alpha, query, num)
            elif self._code_prompt_type == "github_issue":
                return self._add_post_prompt_github_issue(use_alpha, self._rerank_type=="code_reasoning", query, num)
            else:
                raise ValueError(f"Invalid code_prompt_type: {self._code_prompt_type}")
        else:  # text reranking
            if use_alpha:
                example_ordering = "[B] > [A]" if self._variable_passages else "[D] > [B]"
            else:
                example_ordering = "[2] > [1]" if self._variable_passages else "[4] > [2]"
            return f"Search Query: {query}.\nRank the {num} passages above based on their relevance to the search query. All the passages should be included and listed using identifiers, in descending order of relevance. The output format should be [] > [], e.g., {example_ordering}, Only respond with the ranking results, do not say any word or explain."

    def _add_prefix_prompt_doc_string(self, use_alpha, query: str, num: int) -> str:
        if use_alpha:
            return f"I will provide you with {num} code functions, each indicated by an alphabetical identifier []. Rank the code functions based on their relevance to the functionality described by the following doc string: {query}.\n"
        else:
            return f"I will provide you with {num} code snippets, each indicated by a numerical identifier []. Rank the code snippets based on their relevance to the functionality described by the following doc string: {query}.\n"

    def _add_post_prompt_doc_string(self, use_alpha, query: str, num: int) -> str:
        if use_alpha:
            example_ordering = "[B] > [A]" if self._variable_passages else "[D] > [B]"
        else:
            example_ordering = "[2] > [1]" if self._variable_passages else "[4] > [2]"
        return f"Doc String: {query}.\nRank the {num} code snippets above based on their relevance to the functionality described by the doc string. All the code snippets should be included and listed using identifiers, in descending order of relevance. The output format should be [] > [], e.g., {example_ordering}. Only respond with the ranking results, do not say any word or explain."

    def _add_prefix_prompt_github_issue(self, use_alpha, ask_reasoning: bool, query: str, num: int) -> str:
        if use_alpha:
            prefix_prompt = f"I will provide you with {num} code functions, each indicated by a alphabetical identifier []."
            if ask_reasoning:
                prefix_prompt += f" You first think about the reasoning process as an internal monologue and then rank the code functions based on their relevance to contain the faults causing the GitHub issue: {query}.\n"
            else:
                prefix_prompt += f" Rank the code functions based on their relevance to contain the faults causing the GitHub issue: {query}.\n"
        else:
            prefix_prompt = f"I will provide you with {num} code functions, each indicated by a numerical identifier []."
            if ask_reasoning:
                prefix_prompt += f" You first think about the reasoning process as an internal monologue and then rank the code functions based on their relevance to contain the faults causing the GitHub issue: {query}.\n"
            else:
                prefix_prompt += f" Rank the code functions based on their relevance to contain the faults causing the GitHub issue: {query}.\n"
        return prefix_prompt

    def _add_post_prompt_github_issue(self, use_alpha, ask_reasoning: bool, query: str, num: int) -> str:
        if use_alpha:
            example_ordering = "[B] > [A]" if self._variable_passages else "[D] > [B]"
        else:
            example_ordering = "[2] > [1]" if self._variable_passages else "[4] > [2]"
        if ask_reasoning:
            return f"GitHub Issue: {query}. \nFirst think about the reasoning process as an internal monologue and rank the {num} code functions above based on their relevance to contain the faults causing the GitHub issue. All the code functions should be included and listed using identifiers, in descending order of relevance. Respond in the following format: <think>\n...\n</think>\n<answer>\n[] > [] ...\n</answer>. The output format of the answer format should be [] > [], e.g., {example_ordering}."
        else:
            return f"GitHub Issue: {query}. \nRank the {num} code functions above based on their relevance to contain the faults causing the GitHub issue. All the code functions should be included and listed using identifiers, in descending order of relevance. The output format should be [] > [], e.g., {example_ordering}. Only respond with the ranking results, do not say any word or explain."

    def _add_few_shot_examples(self, conv):
        for _ in range(self._num_few_shot_examples):
            ex = random.choice(self._examples)
            obj = json.loads(ex)
            prompt = obj["conversations"][0]["value"]
            response = obj["conversations"][1]["value"]
            conv.append_message(conv.roles[0], prompt)
            conv.append_message(conv.roles[1], response)
        return conv
    
    def _add_few_shot_examples_messages(self, messages):
        for _ in range(self._num_few_shot_examples):
            ex = random.choice(self._examples)
            obj = json.loads(ex)
            prompt = obj["conversations"][0]["value"]
            response = obj["conversations"][1]["value"]
            messages.append({"role": "user", "content": prompt})
            messages.append({"role": "assistant", "content": response})
        return messages

    def create_prompt(self, result: Result, use_alpha: bool, rank_start: int, rank_end: int) -> Tuple[str, int]:
        query = result.query
        max_query_len = self.get_num_tokens(query)
        num = len(result.hits[rank_start:rank_end])
        max_doc_length = 1024 if (self._rerank_type == "code" or self._rerank_type == "code_reasoning") else 300
        min_doc_length = 300
        while True:
            messages = list()
            if self._system_message and self.system_message_supported:
                messages.append({"role": "system", "content": self._system_message})
            messages = self._add_few_shot_examples_messages(messages)
            query_tokens = self._tokenizer.tokenize(query)[:int(max_query_len)]
            truncated_query = self._tokenizer.convert_tokens_to_string(query_tokens)
            prefix = self._add_prefix_prompt(use_alpha, truncated_query, num)
            rank = 0
            input_context = f"{prefix}\n"
            for hit in result.hits[rank_start:rank_end]:
                rank += 1
                if self._rerank_type == "code" or self._rerank_type == "code_reasoning":
                    content = hit["content"]
                    content = content.replace("Title: Content: ", "")
                    tokenized_content = self._tokenizer.tokenize(content)
                    content_tokens = tokenized_content[:int(max_doc_length)]
                    truncated_content = self._tokenizer.convert_tokens_to_string(content_tokens)
                    identifier = chr(ALPH_START_IDX + rank) if use_alpha else str(rank)
                    input_context += f"[{identifier}] {self._replace_number(truncated_content, use_alpha)}\n"
                else:
                    content = hit["content"].replace("Title: Content: ", "").strip()
                    content = " ".join(content.split()[:max_doc_length])
                    identifier = chr(ALPH_START_IDX + rank) if use_alpha else str(rank)
                    input_context += f"[{identifier}] {self._replace_number(content, use_alpha)}\n"
            input_context += self._add_post_prompt(use_alpha, truncated_query, num)
            messages.append({"role": "user", "content": input_context})
            if self._system_message and not self.system_message_supported:
                messages[0]["content"] = self._system_message + "\n " + messages[0]["content"]
            prompt = self._tokenizer.apply_chat_template(messages, tokenize=False, add_generation_prompt=True)
            prompt = fix_text(prompt)
            num_tokens = self.get_num_tokens(prompt)
            if num_tokens <= self.max_tokens() - self.get_total_output_tokens(use_alpha, rank_end - rank_start):
                break
            else:
                prefix_len = len(self._tokenizer.encode(prefix))
                if (len(query_tokens) + prefix_len) > (self.max_tokens() - min_doc_length *(rank_end - rank_start) - self.get_total_output_tokens(use_alpha, rank_end - rank_start)):
                    # Query truncation to ensure min doc length for each candidate document/code
                    offset = num_tokens - (self.max_tokens() - self.get_total_output_tokens(use_alpha, rank_end - rank_start))
                    max_query_len -= (offset//2 + 1)
                else:
                    # Document truncation
                    max_doc_length -= max(
                        1,
                        (
                            num_tokens - self.max_tokens() + self.get_total_output_tokens(use_alpha, rank_end - rank_start)
                        ) // ((rank_end - rank_start) * 4),
                    )            
        return prompt, num_tokens

    def create_prompt_batched(
        self,
        results: List[Result],
        use_alpha: bool,
        rank_start: int,
        rank_end: int,
        batch_size: int = 32,
    ) -> List[Tuple[str, int]]:
        def chunks(lst, n):
            """Yield successive n-sized chunks from lst."""
            for i in range(0, len(lst), n):
                yield lst[i : i + n]

        all_completed_prompts = []

        with ThreadPoolExecutor() as executor:
            for batch in chunks(results, batch_size):
                completed_prompts = list(
                    executor.map(
                        lambda result: self.create_prompt(result, use_alpha, rank_start, rank_end),
                        batch,
                    )
                )
                all_completed_prompts.extend(completed_prompts)
        return all_completed_prompts

    def get_num_tokens(self, prompt: str) -> int:
        return len(self._tokenizer.encode(prompt))

    def cost_per_1k_token(self, input_token: bool) -> float:
        return 0
